const url = require('url');
const _ = require('lodash');
const dayjs = require('dayjs');
const when = require('when');
const PLog = require('plog');
const moment = require('moment');
const lookup = require('../lib/getLookup.js');
const config = require('../configs/current');
const rumCounterSetup = require('./common/rumCounterSetup');
const urlFormat = require('./common/urlFormat.js').urlFormat;
const apiSetup = require('./common/apiSetup');
const checkAuth = require('./common/checkAuth');
const yaphoneLite = require('./common/yaphoneLite');
const showPlus = require('./common/plusGuard').showPlus;
const isUnsupportedBro = require('./common/isUnsupportedBro');
const checkUidFromQuery = require('./avatars').checkUidFromQuery;
const removePhonishes = require('./profile.social').removePhonishes;
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const {getSocialProviders} = require('./common/socialSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const getBnplStat = require('./common/getBnplStat');
const processSocialProfiles = require('./profile.social').processProfiles;
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const geoCountries = require('../lib/geo/countries.json');
const getDefaultUser = require('./common/checkAuth/getDefaultUser');
const generateProcessUUID = require('./authv2/generateProcessUUID');
const writeStatbox = require('./common/writeStatbox');
const doCheckAuth = require('./common/checkAuth/doCheck');
const getMetrics = require('./common/getMetrics');
const getPassportLinks = require('./common/getPassportLinks');
const isFamilyEnable = require('./common/isFamilyEnable');
const getMethodUrls = require('./login-method').getMethodUrls;
const validateAMParams = require('./authv2/validateAMParams');
const createAMState = require('./authv2/createAMState');
const setupUserTicket = require('./common/userTicketsSetup.js');
const validateRetpath = require('./common/validateRetpath');
const {addFamilyOpenGraph, payBindMiddleware, putFamilyInStore} = require('./family/middlewares');

const getCurrentPage = (req, mergeData) =>
    Object.assign(
        {},
        {
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: req.path,
            query: req.query
        },
        mergeData
    );

const redirectMap = {
    passport: '/profile',
    changereg: '/profile/personal-info',
    changehint: '/profile/change-hint',
    changedisplayname: '/profile/display-name',
    changeemails: '/profile/emails/list',
    changeavatar: '/profile/avatars'
};

const FAMILY_PUSH_ENTRIES_MAP = {
    limits: 'react.family-lazy',
    limitsMember: 'react.family-lazy',
    main: 'react.push-family'
};
const ORIGINS_WITHOUT_PAY_EXP = ['taxi_familycard'];

function priceDelimeter(_price) {
    const parts = [];

    if (!_price) {
        return '';
    }

    let price = _price.toFixed(0);

    if (price === '0') {
        return '';
    }

    while (price.length) {
        parts.push(price.slice(-3));
        price = price.slice(0, -3);
    }

    return parts.reverse().join(' ');
}

const prepareFactory = ({req, lang, publicId}) => {
    const afishaData = ({orders: {items}}) => {
        const data = items.map((order) => prepareFactory({lang}).afishaDataInfo({order}));
        const sortedData = data.sort((a, b) => a.dateStamp - b.dateStamp);

        return {
            actual: sortedData.filter((item) => !item.sessionPassed),
            expired: sortedData.filter((item) => item.sessionPassed)
        };
    };

    const afishaDataInfo = ({order}) => {
        const {event, place, ticketsCount, ticketsPdf, id, name, passed = false} = order;

        const date =
            order.date &&
            moment(order.date)
                .locale(lang)
                .format('dddd D MMMM YYYY, HH:mm');

        const linksData = {};
        const eventData = {};
        const hallData = {};
        const orderData = {
            id,
            venueName: name,
            sessionPassed: passed
        };

        // eslint-disable-next-line no-extra-boolean-cast
        if (Boolean(date)) {
            orderData.date = date;
            orderData.dateStamp = order.date ? Date.parse(order.date) : null;
        }

        // eslint-disable-next-line no-extra-boolean-cast
        if (Boolean(ticketsCount)) {
            orderData.total_ticket_count = ticketsCount;
        }

        // eslint-disable-next-line no-extra-boolean-cast
        if (Boolean(event)) {
            eventData.eventId = event.id;
            const eventImageUrl = event.image && event.image.image && event.image.image.url;
            const eventName = event.title;
            const eventUrl = event.url;

            // eslint-disable-next-line no-extra-boolean-cast
            if (Boolean(eventImageUrl)) {
                eventData.image = eventImageUrl;
            }
            // eslint-disable-next-line no-extra-boolean-cast
            if (Boolean(eventName)) {
                eventData.eventName = eventName;
            }
            // eslint-disable-next-line no-extra-boolean-cast
            if (Boolean(eventUrl)) {
                linksData.eventUrl = eventUrl;
            }
        }

        // eslint-disable-next-line no-extra-boolean-cast
        if (Boolean(place)) {
            const placeTitle = place.title;
            const placeUrl = place.url;

            // eslint-disable-next-line no-extra-boolean-cast
            if (Boolean(placeTitle)) {
                hallData.hallName = placeTitle;
            }
            // eslint-disable-next-line no-extra-boolean-cast
            if (Boolean(placeUrl)) {
                linksData.venue = placeUrl;
            }
        }

        const ticketsUrl = ticketsPdf && ticketsPdf.url;
        // eslint-disable-next-line no-extra-boolean-cast

        if (ticketsUrl) {
            linksData.tickets = ticketsUrl;
        }

        return Object.assign({}, orderData, {links: linksData}, eventData, hallData);
    };

    const favAfishaData = (data) => {
        data = data || {};
        const favoriteEvents = data.favoriteEvents || {};
        const items = favoriteEvents.items || [];

        return {
            items: items.map(({event, scheduleInfo}) => {
                const date = scheduleInfo.preview.text;
                const onlyPlace = scheduleInfo.onlyPlace;

                return {
                    image: event.image && event.image.image && event.image.image.url,
                    links: {
                        event: event.url,
                        venue: onlyPlace ? onlyPlace.url : null
                    },
                    eventName: event.title,
                    date: date && date.charAt(0).toUpperCase() + date.slice(1),
                    venueName: scheduleInfo.placePreview,
                    endDate: null,
                    typeName: event.type.name,
                    places: scheduleInfo.placesTotal
                };
            })
        };
    };

    return {
        diskData: ({used_space: usedSpace, total_space: totalSpace}) => ({
            usedSpace: Math.min(usedSpace, totalSpace),
            totalSpace
        }),
        afishaData,
        afishaDataInfo,
        favAfishaData,
        marketData: ({orders}) => ({
            items: orders.map((order) => {
                const history = order.history[0] || {};

                return {
                    shopName: order.shopName,
                    orderNo: order.shopOrderId,
                    price: priceDelimeter(order.total),
                    deliveryPrice: order.total - order.subtotal,
                    paymentMethod: order.paymentMethod,
                    date:
                        history.time &&
                        moment(history.time)
                            .locale(lang)
                            .format('D MMMM YYYY, HH:mm'),
                    status: history.status
                };
            })
        }),
        musicData: ({subscription = {}}) => {
            const newSub = {};
            const oneDayMs = 1000 * 60 * 60 * 24;
            const now = Date.now();

            if (subscription.hasOwnProperty('nonAutoRenewable')) {
                const {start, end} = subscription.nonAutoRenewable;
                const endTime = new Date(end).getTime();
                const startTime = new Date(start).getTime();
                const duration = endTime - startTime;

                newSub.types = {
                    promo: true
                };

                newSub.vendor = 'Yandex';
                newSub.estimated = Math.max(1, Math.ceil(((now - startTime) / duration) * 100));
                newSub.dateParts = {
                    expireDate: moment(endTime).format('MM.DD.YYYY'),
                    expires: moment(endTime)
                        .locale(lang)
                        .format('D MMMM YYYY'),
                    daysLeft: Math.round((endTime - now) / oneDayMs)
                };
            } else if (
                subscription.hasOwnProperty('autoRenewable') ||
                subscription.hasOwnProperty('familyAutoRenewable')
            ) {
                const promoDays = (subscription.nonAutoRenewableRemainder || {}).days || 0;
                const autoRenewable = subscription.autoRenewable || subscription.familyAutoRenewable;

                for (const data of autoRenewable) {
                    const expires = new Date(data.expires).getTime();
                    const {product = {}} = data;
                    const duration = Number(product.duration);
                    const trialDuration = Number(product.trialDuration);
                    const oneDuration = trialDuration || duration;
                    const daysLeft = Math.floor((expires - now) / oneDayMs) - 1;

                    if (expires < now) {
                        continue;
                    }

                    newSub.finished = data.finished;
                    newSub.estimated = Math.max(1, Math.ceil(((oneDuration - daysLeft) / oneDuration) * 100));
                    if (!data.finished) {
                        const expiresMoment = moment(expires);

                        newSub.dateParts = {
                            daysLeft,
                            renewDate: expiresMoment.locale(lang).format('D MMMM YYYY'),
                            renewDateDN: expiresMoment.format('DD.MM.YYYY')
                        };
                    } else {
                        newSub.dateParts = {daysLeft};
                    }
                    newSub.types = {
                        monthly: duration < 40,
                        annual: duration > 40,
                        promo: false
                    };
                    newSub.vendor = data.vendor;
                    newSub.dateParts.expireDate = moment(expires).format('DD.MM.YYYY');
                    newSub.dateParts.expireDateLoc = moment(expires)
                        .locale(lang)
                        .format('D MMMM YYYY');
                    newSub.dateParts.expiresTS = expires + promoDays * oneDayMs;
                    newSub.dateParts.expires = moment(newSub.dateParts.expiresTS)
                        .locale(lang)
                        .format('D MMMM YYYY');

                    if (promoDays) {
                        newSub.dateParts.promoDays = promoDays;
                    }

                    break;
                }
            } else {
                newSub.canStartTrial = true;
            }

            return newSub;
        },
        videoData: ({videos}) => ({
            items: videos.map((video) =>
                Object.assign({}, video, {
                    formatted:
                        video.duration &&
                        moment(video.duration * 1000)
                            .locale(lang)
                            .format(video.duration >= 3600 ? 'k:mm:ss' : 'mm:ss'),
                    thumbnail: video.thumbnail
                        ? (video.thumbnail.indexOf('?') !== -1 && `${video.thumbnail}&h=150&n=1040`) ||
                          `${video.thumbnail}/244x150`
                        : '',
                    yurl: config.paths.video && config.paths.video.replace('%id%', video.id)
                })
            )
        }),
        favMarketData: ({items}) => ({
            items: items.map(({model}) => {
                const data = {
                    link: model.link,
                    name: model.name,
                    kind: model.kind,
                    image: model.photo && model.photo.url
                };

                if (model.hasOwnProperty('price')) {
                    data.priceMax = model.price.max;
                    data.priceMin = model.price.min;
                    data.priceAvg = model.price.avg;
                }

                return data;
            })
        }),
        collectionsData: ({collections}) => ({
            items: collections.map((item) => {
                const newItem = {
                    title: item.title,
                    slug: item.slug
                };
                const {preview} = item;

                if (preview) {
                    newItem.image = `${config.paths.mds}/get-pdb/${preview.mds_group_id}/${preview.mds_key}/preview`;
                }

                if (publicId && item.slug) {
                    newItem.link = `${config.paths.collectionsUser}/${publicId}/${item.slug}`;
                }

                return newItem;
            })
        }),
        mapsBookmarksData: ({bookmarks}) => {
            const geoBookmarks = bookmarks.filter((item) => Boolean(item.uri));

            return when
                .settle(geoBookmarks.map((item) => req.api.externalDataMapBookmarkInfo(item.uri, lang)))
                .then((descriptors) => ({
                    items: prepareFactory({lang}).mapsBookmarksDataInfo(descriptors, geoBookmarks)
                }));
        },
        mapsBookmarksDataInfo: (descriptors, geoBookmarks) =>
            descriptors
                .map((descriptor, bookmarkIndex) => {
                    if (descriptor.value && descriptor.value.body) {
                        descriptor.value.body.bookmark = geoBookmarks[bookmarkIndex];
                    }

                    return descriptor;
                })
                .filter((descriptor) => descriptor.state !== 'rejected')
                .map((descriptor) => descriptor.value.body)
                .filter(({bookmark_info: {photo_url_template: photoUrlTemplate}}) => Boolean(photoUrlTemplate))
                .map(
                    ({
                        bookmark,
                        bookmark_info: {
                            ratings,
                            photo_url_template: photoUrlTemplate,
                            description,
                            score,
                            address,
                            type,
                            categories,
                            name,
                            coordinates
                        }
                    }) => ({
                        ratings: ratings || 0,
                        score: score || 0,
                        address: address || '',
                        type,
                        categories,
                        photoUrl: (photoUrlTemplate || '').replace('%s', 'L').replace(/^http:\/\//, 'https://'),
                        description: bookmark.description || description,
                        coordinates,
                        id: bookmark.id,
                        uri: bookmark.uri,
                        name: bookmark.title || name
                    })
                )
    };
};

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then((lang) => {
            res.locals.language = lang;
        })
        .catch((err) => {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport.v2')
                .write(err);
        })
        .finally(() => {
            next();
        });
}

function renderPage(req, res) {
    const lang = res.locals.language;
    const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
    const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

    req.api.statboxLogger({
        track_id: res.locals.track_id || null,
        action: 'opened',
        mode: 'profile.passport.v2',
        host: req.hostname,
        pathname: req.path,
        referer: req.headers.referer || null,
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: req.cookies.yandexuid,
        uid: (res.locals.store && res.locals.store.person && res.locals.store.person.uid) || null,
        origin: (req.query && req.query.origin) || null,
        experiment_flags: experimentFlags,
        experiment_boxes: experimentBoxes
    });

    res.render(`profile.passportv2.${lang}.jsx`);
}

function renderFamilyPushPage(req, res) {
    const lang = res.locals.language;
    const entry = req.entry;

    res.render(`${FAMILY_PUSH_ENTRIES_MAP[entry]}.${lang}.jsx`);
}

const FAMILY_WEBVIEW_PAGES = {
    'invite.confirm': 'family-lazy',
    invite: 'family-lazy',
    payCards: 'family-lazy',
    limits: 'family-lazy',
    deleteMember: 'family-lazy',
    leaveMember: 'family-lazy',
    inviteSlot: 'family-lazy',
    main: 'family-lazy',
    default: 'family-webview'
};

const renderFamilyWebviewPage = (entry) => (req, res) =>
    res.render(`react.${FAMILY_WEBVIEW_PAGES[entry] || FAMILY_WEBVIEW_PAGES.default}.${res.locals.language}.jsx`);

const createStore = [
    getSocialProviders,
    (req, res, next) => {
        // eslint-disable-line
        const locals = res.locals;
        const langSwitcher = new LangSwitcherView(req._controller, locals.track_id);
        const account = locals.account;
        const lastUpdate = account && account.password_info && account.password_info.last_update;
        const tld = req._controller.getTld();
        const yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
        const emptyService = {
            isLoading: true,
            initialLoad: true
        };
        const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
        const links = config.links[tld] || config.links.ru || {};
        const defaultAccount = (locals.accounts && locals.accounts.defaultAccount) || {};
        const plusInfo = defaultAccount.plus || {};
        const retpath = (req.query && (req.query.url || req.query.retpath)) || null;
        const backpath = (req.query && req.query.backpath) || null;
        const embeddedauth = config.paths.embeddedauth && config.paths.embeddedauth.replace('%tld%', tld);
        const experiments = Object.assign(
            {},
            {
                flags: [],
                flagsString: '',
                boxes: '',
                encodedBoxes: ''
            },
            res.locals.experiments
        );
        const {chatGuid} = config.subscriptions;

        [retpath, backpath].forEach((el) => {
            el ? handlers.push(req.api.validateRetpath({retpath: el})) : null;
        });

        locals.reactPage = 'profile.passportv2';

        if (plusInfo.subscriptionExpireTime) {
            const expDate = moment(new Date(plusInfo.subscriptionExpireTime).getTime());

            plusInfo.expireDate = expDate.locale(locals.language).format('D MMMM YYYY');
            plusInfo.expireDateDN = expDate.format('DD.MM.YYYY');
            plusInfo.subscriptionExpireDays = Math.max(
                0,
                Math.round((plusInfo.subscriptionExpireTime - Number(new Date())) / 24 / 60 / 60 / 1000)
            );
        }

        let personCountry = '';

        if (account.person.country) {
            const locsCountries = geoCountries[locals.language];

            personCountry = ((locsCountries || []).filter((country) => country.iso === account.person.country)[0] || {})
                .name;
        }

        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}} = locals;
        const {metricsUserType, isPortal} = userType;

        const {default_uid: defaultUid, users, connection_id: ci} = req.blackbox;
        const defaultUser = getDefaultUser(users, defaultUid);
        const {attributes = {}} = defaultUser;
        const appPlatform = req.query.app_platform || '';

        locals.store = {
            subscriptions: Object.assign(
                {
                    chatGuid
                },
                res.locals.subscriptionsState
            ),
            dashboard: {
                plus: Object.assign(
                    {},
                    {
                        country: locals.country,
                        allowed: locals.showPlus,
                        enabled: false,
                        enabledTime: 0,
                        trialUsedTime: 0,
                        subscriptionStoppedTime: 0,
                        subscriptionExpireTime: 0,
                        nextChargeTime: 0,
                        expireDate: ''
                    },
                    plusInfo
                ),
                items: {
                    getplusLink: url.format({
                        protocol: req.headers['x-real-scheme'],
                        hostname: config.paths.plus && config.paths.plus.replace('%tld%', tld),
                        pathname: 'getplus',
                        query: {
                            resume: 1,
                            retpath: url.format(
                                getCurrentPage(req, {
                                    pathname: '/profile/services'
                                })
                            )
                        }
                    })
                },
                errors: {},
                messages: {},
                isLoading: {},
                experiments: {},
                diskData: locals.diskData || emptyService,
                afishaData: locals.afishaData || emptyService,
                marketData: locals.marketData || emptyService,
                musicData: locals.musicData || emptyService,
                videoData: locals.videoData || emptyService,
                favMarketData: locals.favMarketData || emptyService,
                collectionsData: locals.collectionsData || emptyService,
                favAfishaData: locals.favAfishaData || emptyService,
                mapsBookmarksData: locals.mapsBookmarksData || emptyService
            },
            settings: {
                isNewLayout: !locals.isLite,
                location: req.path,
                plusHost: config.paths && config.paths.plus && `//${config.paths.plus.replace('%tld%', tld)}`,
                host: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname
                }),
                avatar: config.paths.avatar || {},
                embeddedauth,
                logoutURL: `${embeddedauth}&yu=${yandexuid}&action=logout&uid=${account.uid}&retpath=${links.yandex}`,
                links,
                passportLinks: getPassportLinks(req),
                help: config.paths.help || {},
                tld,
                language: locals.language,
                ua: {},
                isLite: locals.isLite,
                isPhone: false,
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                billingUpdateStatus: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: config.paths.billingUpdateStatusPath
                }),
                yamoneyCards: config.paths.yamoneyCards,
                tunePlaces: config.paths.tunePlaces,
                marketAddresses: config.paths.marketAddresses,
                staticMaps: config.paths.staticMaps,
                staticPath,
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType,
                yaPayLanding: config.paths.yaPayLanding
            },
            common: {
                v2: !locals.isLite, // PASSP-32231
                origin: (req.query || {}).origin,
                process_uuid: res.locals.process_uuid,
                canChangePassword: account.can_change_password,
                edit: locals.edit || '',
                showRegPopup: false,
                hasComponentsForNav: true,
                actionForRepeat: null,
                uid: account.uid,
                yandexuid,
                // при загрузке срабатывает POP event и уменьшает историю на 1, а так тут должен быть 0
                historyOnPassport: 1,
                track_id: locals.track_id,
                currentPage: url.format(getCurrentPage(req)),
                retpath: null,
                backpath: null,
                defaultPage: url.format({
                    pathname: '/profile',
                    query: req.query
                }),
                isPDD: account.domain !== undefined,
                isWSUser: locals.isWSUser,
                dev: config.dev,
                experiments,
                isYandexoid: locals.isYandexoid,
                isYandex: locals.isYandex,
                locationCountryId: locals.countryId,
                isFamilyEnable: isFamilyEnable(locals.countryId),
                isSupportCenterEnable:
                    ((experiments.flags && experiments.flags.includes('support-window-enable')) ||
                        locals.isYandex ||
                        locals.isYandexoid) &&
                    locals.ua &&
                    locals.ua.BrowserName !== 'MSIE'
            },
            captcha: {
                loading: false,
                playing: false,
                type: 'text',
                key: null,
                imageUrl: null,
                introSound: null,
                captchaSound: null
            },
            person: Object.assign({}, account.person || {}, {
                birthday: account.person.birthday || '',
                city: account.person.city || '',
                countryName: personCountry,
                isSocialchik: defaultAccount.isSocial,
                uid: account.uid,
                login: account.login,
                displayLogin: account.display_login,
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayNames: account.display_names || {},
                displayName: defaultAccount.displayName || '',
                loginOptions: defaultAccount.loginOptions,
                hasPhone: defaultAccount.hasPhone,
                hasPublicProfile: defaultAccount.hasPublicProfile,
                hasThirdPartyAccess: defaultAccount.hasThirdPartyAccess,
                havePassword: defaultAccount.havePassword,
                escapedDisplayName: (defaultAccount.displayName && _.escape(defaultAccount.displayName)) || '',
                isDisplayNameEmpty: defaultAccount.isDisplayNameEmpty,
                publicDisplayName: defaultAccount.publicDisplayName || '',
                escapedPublicDisplayName: _.escape(defaultAccount.publicDisplayName || ''),
                isLiteUser: defaultAccount.is_liteUser,
                isNeoPhonish: defaultAccount.isNeoPhonish,
                isFederal: defaultAccount.isFederal,
                isVerifiedAccount: defaultAccount.isVerified,
                errors: {}
            }),
            appPasswords: {
                createAppPassword: {
                    configClients: config.appPasswordsClientIdMapping,
                    clients: [],
                    clientId: '',
                    deviceName: '',
                    clientName: '',
                    passwordCreated: ''
                },
                tokens: {
                    appPasswordsCount: 0,
                    list: [],
                    showDisableScreen: false
                }
            },
            security: {
                securityLevel:
                    experiments.flags.includes('passport-security-v2-exp') && account.security_level === 16
                        ? config.securityLevels[4]
                        : config.securityLevels[account.security_level] || 'undefined',
                lastAuth: account.lastauth
                    ? Object.assign({}, account.lastauth, {
                          date: account.lastauth.timestamp
                              ? moment(new Date(account.lastauth.timestamp * 1000).getTime())
                                    .locale(locals.language)
                                    .format('D MMMM, HH:mm')
                              : '',
                          region: (account.lastauth.ip || {}).geoid
                              ? lookup.getRegionById(account.lastauth.ip.geoid)
                              : {}
                      })
                    : {},
                controlQuestion: {
                    errors: {},
                    available: [],
                    visible: false,
                    loading: false,
                    current: account.question,
                    selected: '0'
                },
                isClosedPhoneAlert: Boolean(req.cookies['security_alert_phone']),
                isClosed2faAlert: Boolean(req.cookies['security_alert_2fa'])
            },
            social: {
                providers: res.locals.socialProviders.providers.filter((i) => {
                    const {data: {code} = {}, enabled} = i;

                    if (!req._controller.hasExp('social-auth-esia-exp') && code === 'esia') {
                        return false;
                    }
                    return enabled;
                }),
                profiles:
                    (account.profiles && removePhonishes(processSocialProfiles(account.profiles, locals.language))) ||
                    [],
                brokerPath: config.paths.broker,
                showAllSettings: false,
                allowAuthMethodError: {
                    profileId: 0,
                    error: ''
                },
                phonishes: [],
                kpshniki: res.locals.kpshniki
            },
            publicId: {
                isEmbedded: false,
                showLoginInSuggest: isPortal,
                hasPublicIdSet: defaultAccount.hasPublicIdSet,
                id: account.public_id,
                publicIdSuggest: [],
                hasUpdated: false,
                updatesNumber: 0
            },
            emails: {
                loading: null,
                error: null,
                states: [],
                addedEmail: null,
                openAliasesList: false,
                emails: locals.emails
            },
            domik: {
                errors: {},
                requestPassword: false,
                requestCaptcha: false,
                passwordNotMatched: false,
                captchaNotMatched: false,
                loading: false
            },
            phones: {
                restore: [],
                other: [],
                completeSms2fa: req.query.completeSms2fa && req.query.completeSms2fa === '1'
            },
            billing: {},
            access: {
                is2faEnabled: account.is_2fa_enabled,
                isSms2faEnabled: attributes['200'] === '1',
                isAppPasswordsEnabled: account.app_passwords_enabled,
                isSocialchik: defaultAccount.isSocial,
                passwordInfo: account.password_info
                    ? Object.assign({}, account.password_info, {
                          timeFromUpdate: lastUpdate
                              ? moment()
                                    .locale(locals.language)
                                    .subtract(new Date().getTime() - lastUpdate * 1000, 'ms')
                                    .fromNow()
                              : ''
                      })
                    : {},
                featureHint: ''
            },
            form: {},
            header: {
                defaultAccount,
                accounts: (locals.accounts && locals.accounts.accounts) || [],
                canAddMore: locals.accounts && locals.accounts['can-add-more'],
                ci
            },
            footer: {
                langlist: []
            },
            metrics: {
                header: 'Новый профиль',
                experiments: experiments.encodedBoxes || ''
            },
            monitoring: {
                page: 'passport.v2'
            },
            pageMessage: (locals.store || {}).pageMessage || null,
            changeAvatar: {
                track_id: '',
                url: null,
                defaultUrl: config.paths.avatar && config.paths.avatar.default_300,
                defaultSize: 'islands-300',
                defaultId: '0-0',
                backupAvatar: {
                    id: 0,
                    url: ''
                },
                hasChanged: false,
                queryUid: res.locals.queryUid || '',
                id: 0,
                isByUrl: false,
                status: 'normal',
                error: '',
                clear: false,
                isCropNotSupported: false
            },
            subs: {
                list: [],
                actualList: [],
                error: null,
                hasChanged: false,
                hasAllChecked: true,
                hasAllActualChecked: true,
                hasAllUnchecked: false,
                hasAllActualUnchecked: false,
                listType: 'actual',
                wasListLoad: false
            }
        };

        if (locals.bnpl) {
            const today = dayjs(new Date())
                .set('h', 0)
                .set('m', 0)
                .set('s', 0);

            const paymentDate = dayjs(locals.bnpl.nextPaymentDate)
                .set('h', 0)
                .set('m', 0)
                .set('s', 1);

            const differenceInDays = paymentDate.diff(today, 'days');

            let bnplTitlekey = differenceInDays;

            if (differenceInDays >= 4) {
                bnplTitlekey = 'date';
            } else if (differenceInDays === 0) {
                bnplTitlekey = 'today';
            } else if (differenceInDays === 1) {
                bnplTitlekey = 'tomorrow';
            } else {
                bnplTitlekey = paymentDate.get('d');
            }

            locals.store.bnpl = {
                titleKey: bnplTitlekey,
                hasNextPayment: Boolean(locals.bnpl.nextPaymentDate),
                formatedDate: paymentDate.locale(locals.language).format('D MMMM'),
                nextAmount: priceDelimeter(locals.bnpl.nextAmount),
                amountLeft: priceDelimeter(locals.bnpl.amountLeft),
                url: locals.bnpl.url,
                showWhenOpen: Boolean(req.query.showSplit)
            };
        }

        if (['ios', 'android'].includes(appPlatform)) {
            const {
                am_version_name: amVersionName,
                app_id: appId,
                app_platform: platform,
                app_version_name: appVersionName,
                device_id: deviceId,
                lang,
                locale,
                theme,
                uid,
                track_id: trackId
            } = req.query;

            locals.store.am = {
                isAm: true,
                amVersionName,
                appId,
                platform,
                appVersionName,
                deviceId,
                lang,
                locale,
                theme: theme || req._controller.getCookie('theme'),
                uid,
                trackId,
                closeUrl: '/closewebview'
            };
        }

        if (account.phones) {
            Object.keys(account.phones).forEach(function(code) {
                const phone = account.phones[code];

                if (phone.bound) {
                    locals.store.phones[phone.secured ? 'restore' : 'other'].push({
                        id: phone.id,
                        number: phone.number.masked_international,
                        isDefault: phone.is_default,
                        isAlias: phone.is_alias
                    });
                }
            });
        }

        if (locals.store.security.lastAuth) {
            const geoId = (locals.store.security.lastAuth.ip && locals.store.security.lastAuth.ip.geoid) || '';

            if (geoId) {
                const regionLocales = geoId ? lookup.getLinguistics(geoId, locals.language) : '';

                if (regionLocales) {
                    locals.store.security.lastAuth.region = regionLocales.nominative || '';
                } else {
                    const region = lookup.getRegionById(geoId);

                    locals.store.security.lastAuth.region = region.name || '';
                }
            }
        }

        if (req._controller.hasExp('auth_params_exp')) {
            locals.store.loginMethod = {
                openLoginMethodModal: req.query.openLoginMethodModal && req.query.openLoginMethodModal === '1',
                methodLinks: getMethodUrls(req)
            };
        }

        when.all(handlers)
            .then((response) => {
                const [uatraits = {}, {langlist = {}} = {}, validatedRetpath = {}, validatedBackpath = {}] = response;

                locals.store.settings.ua = uatraits;
                locals.store.footer.langlist = langlist;
                locals.store.settings.isPhone = (uatraits.isMobile || uatraits.isTouch) && !uatraits.isTablet;
                locals.store.settings.isTouch = uatraits.isMobile || uatraits.isTouch;

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                    locals.store.header.retpath = validatedRetpath.body.retpath;
                }

                if (validatedBackpath.body && validatedBackpath.body.retpath) {
                    locals.store.common.backpath = validatedBackpath.body.retpath;
                    locals.store.header.backpath = validatedBackpath.body.retpath;
                }

                locals.store.changeAvatar.isCropNotSupported = isUnsupportedBro(uatraits);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.v2')
                    .write(err);
            })
            .finally(() => {
                delete locals.account;
                delete locals.accounts;

                return next();
            });
    },
    getMetrics({header: 'Новый профиль'})
];

function getState(req, res, next) {
    const getStateParams = {
        need_family_info: false,
        need_family_members: false,
        need_family_kids: false,
        need_family_invites: false
    };

    Promise.all([req.api.profileGetState(getStateParams), req.api.getEmails()])
        .then((response) => {
            const profile = response[0].body;
            const emails = response[1].body;
            const account = profile.account || {};

            res.locals.account = account;
            res.locals.emails = req._controller.decodePunycodeEmails(emails.emails || {});

            return next();
        })
        .catch((errors) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.passport.v2')
                .write(errors);

            if (Array.isArray(errors)) {
                const retpath = url.format(req._controller.getUrl());

                if (errors.indexOf('sessionid.invalid') !== -1 || errors.indexOf('account.disabled') !== -1) {
                    return res.redirect(
                        url.format(
                            Object.assign({}, req._controller.getAuthUrl(), {
                                query: {retpath}
                            })
                        )
                    );
                }

                if (errors[0].code && errors[0].code === 'unknowntrack') {
                    const query = Object.assign({}, req.query);

                    delete query.track_id;

                    return res.redirect(
                        url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: req.path,
                            query
                        })
                    );
                }
            }

            return next(errors);
        });
}

function getTrack(req, res, next) {
    return req.api
        .initTrack({
            type: 'authorize'
        })
        .then(function(response) {
            res.locals.track_id = (response.body && response.body.track_id) || '';
            return next();
        })
        .catch(function(errors) {
            PLog.warn()
                .logId(req.logID)
                .type('profile.passport.v2')
                .write(errors);

            return next();
        });
}

function getPhonishes(req, res, next) {
    res.locals.phonishes = [];
    res.locals.kpshniki = [];

    if (!res.locals.track_id) {
        return next();
    }

    return req.api
        .getPhonishProfiles(res.locals.track_id)
        .then((data) => {
            if (data.hasOwnProperty('body') && data.body.hasOwnProperty('profiles')) {
                for (let i = 0; i < data.body.profiles.length; i++) {
                    const profile = data.body.profiles[i];

                    if (profile.provider_code === 'ya' && profile.hasOwnProperty('phonish')) {
                        res.locals.phonishes.push(
                            Object.assign({}, profile, {
                                provider: {
                                    code: profile.provider_code
                                },
                                profileId: profile.profile_id,
                                registrationDate: profile.phonish.registration_timestamp
                                    ? moment(profile.phonish.registration_timestamp * 1000)
                                          .locale(res.locals.language)
                                          .format('D MMMM YYYY')
                                    : ''
                            })
                        );
                    } else if (profile.provider_code === 'kp') {
                        profile.profileId = profile.profile_id;
                        res.locals.kpshniki.push(profile);
                    }
                }
            }

            return next();
        })
        .catch((err) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);

            return next();
        });
}

function checkSocialtschick(req, res, next) {
    const account = res.locals.account;
    const currentUrl = req._controller.getUrl();

    if (account.password_info.strength === -1 && !account.is_2fa_enabled) {
        return res.redirect(
            url.format(
                Object.assign({}, currentUrl, {
                    pathname: '/profile/upgrade',
                    query: {
                        retpath: url.format(currentUrl)
                    }
                })
            )
        );
    }

    return next();
}

function checkStrongPolicy(req, res, next) {
    const account = res.locals.account;

    if (account.password_info.strong_policy_on || (account.is_2fa_enabled && !account.question)) {
        return res.redirect(
            url.format({
                pathname: '/profile',
                query: req.query
            })
        );
    }

    return next();
}

const getCountryId = (req, res, next) => {
    res.locals.plusAvailableTld = config.plusAvailableTld && config.plusAvailableTld.slice();
    res.locals.plusAvailableCountries = Object.assign({}, config.plusAvailableCountries);

    if (res.locals.countryId || res.locals.country) {
        return next();
    }

    const regionId = res.locals.regionId;
    const tld = req._controller.getTld();
    const countries = res.locals.plusAvailableCountries;

    try {
        res.locals.countryId = lookup.getCountryId(regionId, tld) || null;
        res.locals.country = res.locals.countryId && countries[res.locals.countryId];

        if (!res.locals.country) {
            throw new Error();
        }

        return next();
    } catch (e) {
        for (const key in countries) {
            if (countries.hasOwnProperty(key) && countries[key] === tld) {
                res.locals.country = tld;
                break;
            }
        }

        return next();
    }
};

const setupBase = [getCountryId, getYaExperimentsFlags, getLanguage, apiSetup, getTrack, rumCounterSetup];

const setup = [
    getCountryId,
    showPlus,
    function(req, res, next) {
        if (res.locals.showPlus) {
            res.locals.blackboxPlus = true;
        }

        return next();
    },
    multiAuthAccountsSetup,
    ...setupBase,
    yaphoneLite,
    getBnplStat,
    getPhonishes
];

const checkFamilyAuth = (origin) => (req, res, next) =>
    doCheckAuth(req, res)
        .then(() => next())
        .catch(() => req._controller.redirectToAuthWithOrigin(origin));

const checkSupportExp = (_, res, next) => {
    const locals = res.locals || {};
    const store = locals.store || {};
    const common = store.common || {};

    if (!common.isSupportCenterEnable) {
        next('route');

        return;
    }

    next();
};

const kvkoEnter = [checkAuth, setup, getState, checkSocialtschick, checkStrongPolicy, createStore, renderPage];

const emailsEnter = [checkAuth, setup, getState, createStore, renderPage];

const supportEnter = [checkAuth, setup, getState, createStore, checkSupportExp, renderPage];

const enter = [checkAuth, setup, generateProcessUUID, getState, createStore, renderPage];
const enterFamily = (origin, entry) => [
    checkFamilyAuth(origin),
    setup,
    addFamilyOpenGraph(entry),
    generateProcessUUID,
    setupUserTicket,
    getState,
    createStore,
    putFamilyInStore(entry),
    renderPage
];

const enterFamilyWebview = (origin, entry) => [
    checkFamilyAuth(origin),
    setupBase,
    (req, res, next) => {
        const origin = req.query && req.query.origin;

        if (ORIGINS_WITHOUT_PAY_EXP.includes(origin)) {
            return next();
        }

        return next('route');
    },
    generateProcessUUID,
    setupUserTicket,
    putFamilyInStore(entry, {withAccount: true, withBBUsers: entry !== 'invite.confirm'}),
    getMetrics({header: 'Webview семьи'}),
    renderFamilyWebviewPage(entry)
];

const enterFamilyAfter3ds = [
    checkFamilyAuth('family_after3ds'),
    setup,
    validateRetpath,
    (req, res, next) => {
        const origin = req.query && req.query.origin;

        const hasPayExp = req._controller.hasExp('profile-family-pay-exp');

        return hasPayExp || ORIGINS_WITHOUT_PAY_EXP.includes(origin) ? next() : next('route');
    },
    generateProcessUUID,
    setupUserTicket,
    payBindMiddleware,
    getState,
    createStore,
    putFamilyInStore('after3ds'),
    renderPage
];

const enterFamilyPush = (origin, entry) => [
    checkFamilyAuth(origin),
    setup,
    generateProcessUUID,
    setupUserTicket,
    putFamilyInStore(entry, {withAccount: true, isAm: true}),
    (req, res, next) => {
        req.entry = entry;

        return next();
    },
    validateAMParams,
    createAMState,
    getMetrics({header: 'Пуш в семье'}),
    renderFamilyPushPage
];

const avatarEnter = [
    checkAuth,
    setup,
    generateProcessUUID,
    writeStatbox({
        action: 'opened',
        mode: 'profile_avatar'
    }),
    getState,
    checkUidFromQuery,
    createStore,
    renderPage
];

const getBundle = [
    setup,
    getState,
    createStore,
    function(req, res) {
        return res.json(res.locals.store);
    }
];

const kubrFilter = (req, res, next) => {
    if (!['kz', 'ua', 'by', 'ru'].includes(req._controller.getTld())) {
        return req._controller.redirectToFrontpage();
    }

    return next();
};

const servicesMiddleware = [kubrFilter];

const isYandexoidOrIsYandex = (req, res, next) => {
    if (!res.locals || (!res.locals.isYandex && !res.locals.isYandexoid)) {
        return next('route');
    }

    return next();
};

const checkAccess = (req, res, next) => {
    if (req.blackbox) {
        const {default_uid: defaultUid, users} = req.blackbox;
        const defaultUser = getDefaultUser(users, defaultUid);
        const {attributes = {}, have_password} = defaultUser;
        const {1003: is2faEnabled} = attributes;

        if (!have_password || Boolean(is2faEnabled)) {
            return req._controller.redirectToFrontpage();
        }
    }

    return next();
};

const checkFamilyAdminAccess = (req, res, next) => {
    if (req.blackbox) {
        const {default_uid: defaultUid, users} = req.blackbox;
        const defaultUser = getDefaultUser(users, defaultUid);

        const {family_info: {admin_uid: adminUid} = {}} = defaultUser;

        if (adminUid !== defaultUid) {
            return req._controller.redirectToLocalUrl({
                pathname: '/profile/family',
                query: req.query
            });
        }
    }

    return next();
};

const checkFamilyNotExistOrAdminAccess = (req, res, next) => {
    if (req.blackbox) {
        const {default_uid: defaultUid, users} = req.blackbox;
        const defaultUser = getDefaultUser(users, defaultUid);

        const {family_info: {admin_uid: adminUid} = {}} = defaultUser;

        if (Boolean(adminUid) && adminUid !== defaultUid) {
            return req._controller.redirectToLocalUrl({
                pathname: '/profile/family',
                query: req.query
            });
        }
    }

    return next();
};

module.exports = {
    route: (app) => {
        // REDIRECT FROM OLD PATHS
        app.get('/passport', (req, res, next) => {
            const mode = req.body.mode || req.query.mode;
            const query = Object.assign({}, req.query);

            delete query.mode;

            if ((mode && mode in redirectMap) || !mode) {
                PLog.info()
                    .logId(req.logID)
                    .type('profile.passport')
                    .write(
                        'Redirect from',
                        mode ? `/passport?mode=${mode}` : '/passport',
                        'to',
                        redirectMap[mode ? mode : 'passport']
                    );

                return res.redirect(
                    301,
                    url.format({
                        protocol: req.headers['x-real-scheme'],
                        hostname: req.hostname,
                        pathname: redirectMap[mode ? mode : 'passport'],
                        query
                    })
                );
            }

            return next('route');
        });

        app.get('/profile', enter)
            .get('/profile/personal-info', enter)
            .get('/profile/display-name', enter)
            .get('/profile/change-hint', kvkoEnter)
            .get('/profile/emails', emailsEnter)
            .get('/profile/emails/*', emailsEnter)
            .get('/profile/social', enter)
            .get('/profile/social/:profileId', enter)
            .get('/profile/avatars', avatarEnter)
            .get('/profile/cards', enter)
            .get('/profile/generate-apppassword', emailsEnter)
            .get('/profile/apppasswords-list', emailsEnter)
            .get('/profile/address', [
                function(req, res, next) {
                    const tld = req._controller.getTld();

                    if (tld === 'com') {
                        return next('route');
                    }

                    return next();
                },
                enter
            ])
            .get('/profile/devices', emailsEnter)
            .get('/profile/password', enter)
            .get('/profile/services', servicesMiddleware, enter)
            .get('/support', supportEnter)
            .get('/support/:guid/:timestamp?', supportEnter)
            .get('/profile/subscriptions/:orderId', enter)
            .get('/profile/alice-settings', enter)
            .get('/profile/auth-settings', isYandexoidOrIsYandex, checkAccess, enter)
            .get('/profile/family/invite/:inviteId', enterFamily('family_invite', 'invite.confirm'))
            .get('/profile/family/invite', checkFamilyNotExistOrAdminAccess, enterFamily('family_main', 'invite'))
            .get('/profile/family/limits', checkFamilyAdminAccess, enterFamily('family_limits', 'limits'))
            .get(
                '/profile/family/pay-cards',
                checkFamilyNotExistOrAdminAccess,
                enterFamily('family_pay_cards', 'payCards')
            )
            .get('/profile/family/after3ds', enterFamilyAfter3ds)
            .get('/profile/family/kids', enterFamily('family_kiddish', 'kiddish'))
            .get('/profile/family-webview/invite/:inviteId', enterFamilyWebview('family_invite', 'invite.confirm'))
            .get(
                '/profile/family-webview/invite',
                checkFamilyNotExistOrAdminAccess,
                enterFamilyWebview('family_main', 'invite')
            )
            .get('/profile/family/welcome', enterFamily('family_main', 'inviteWelcome'))
            .get(
                '/profile/family-webview/pay-cards',
                checkFamilyNotExistOrAdminAccess,
                enterFamilyWebview('family_pay_cards', 'payCards')
            )
            .get(
                '/profile/family-webview/limits',
                checkFamilyAdminAccess,
                enterFamilyWebview('family_limits', 'limits')
            )
            .get(
                '/profile/family-webview/delete-member',
                checkFamilyAdminAccess,
                enterFamilyWebview('family_member', 'deleteMember')
            )
            .get('/profile/family-webview/leave-member', enterFamilyWebview('family_member', 'leaveMember'))
            .get(
                '/profile/family-webview/invite-slot',
                checkFamilyAdminAccess,
                enterFamilyWebview('family_invite_slot', 'inviteSlot')
            )
            .get('/profile/family-webview', enterFamilyWebview('family_main', 'main'))
            .get('/profile/family', enterFamily('family_main', 'main'))
            .get('/am/push/family-limits-member', enterFamilyPush('family_limits_member_am', 'limitsMember'))
            .get('/am/push/family-limits', checkFamilyAdminAccess, enterFamilyPush('family_limits_am', 'limits'))
            .get('/am/push/family', enterFamilyPush('family_main_am', 'main'))
            .get('/profile/subscriptions', function(req, res) {
                return res.redirect(
                    urlFormat({
                        protocol: req.headers['x-real-scheme'],
                        hostname: req.hostname,
                        pathname: '/profile',
                        query: req.query
                    })
                );
            })
            .get('/profile/bonus', enter)
            .get('/profile/passwords', enter)
            .get('/profile/documents', enter)
            // Отдельный путь к док-ам для ПП для открытия по шорткату.
            // [checkAuth, setupBase, generateProcessUUID, getState, createStore, renderPage]
            .get('/profile/documents/pp', [
                checkAuth,
                setupBase,
                generateProcessUUID,
                function(req, res, next) {
                    const locals = res.locals;
                    const langSwitcher = new LangSwitcherView(req._controller, locals.track_id);
                    const tld = req._controller.getTld();
                    const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
                    const defaultAccount = (locals.accounts && locals.accounts.defaultAccount) || {};
                    const retpath = (req.query && (req.query.url || req.query.retpath)) || null;
                    const backpath = (req.query && req.query.backpath) || null;
                    const experiments = Object.assign(
                        {},
                        {
                            flags: [],
                            flagsString: '',
                            boxes: '',
                            encodedBoxes: ''
                        },
                        res.locals.experiments
                    );

                    [retpath, backpath].forEach((el) => {
                        el ? handlers.push(req.api.validateRetpath({retpath: el})) : null;
                    });

                    locals.reactPage = 'profile.passportv2';

                    const {default_uid: defaultUid, users} = req.blackbox;
                    const defaultUser = getDefaultUser(users, defaultUid);

                    const {paths = {}, version} = config;
                    const {static: staticPath} = paths;

                    const {userType = {}} = locals;
                    const {metricsUserType} = userType;

                    locals.store = {
                        common: {
                            experiments
                        },
                        settings: {
                            isNewLayout: !locals.isLite,
                            location: req.path,
                            tld,
                            language: locals.language,
                            ua: {},
                            staticPath,
                            env: {
                                type: process.env.NODE_ENV,
                                name: process.env.INTRANET
                            },
                            version,
                            metricsUserType
                        },
                        person: Object.assign(
                            {},
                            {
                                isSocialchik: defaultAccount.isSocial,
                                uid: defaultUser.uid.value,
                                login: defaultUser.login,
                                havePassword: defaultUser.have_password,
                                language: res.locals.language,
                                errors: {}
                            }
                        ),
                        social: {
                            brokerPath: config.paths.broker
                        },
                        metrics: {
                            header: 'Новый профиль',
                            experiments: experiments.encodedBoxes || ''
                        }
                    };

                    if (req._controller.hasExp('auth_params_exp')) {
                        locals.store.loginMethod = {
                            openLoginMethodModal:
                                req.query.openLoginMethodModal && req.query.openLoginMethodModal === '1',
                            methodLinks: getMethodUrls(req)
                        };
                    }

                    when.all(handlers)
                        .then((response) => {
                            const [
                                uatraits = {},
                                {langlist = {}} = {},
                                validatedRetpath = {},
                                validatedBackpath = {}
                            ] = response;

                            locals.store.settings.ua = uatraits;
                            locals.store.footer.langlist = langlist;
                            locals.store.settings.isPhone =
                                (uatraits.isMobile || uatraits.isTouch) && !uatraits.isTablet;
                            locals.store.settings.isTouch = uatraits.isMobile || uatraits.isTouch;

                            if (validatedRetpath.body && validatedRetpath.body.retpath) {
                                locals.store.common.retpath = validatedRetpath.body.retpath;
                                locals.store.header.retpath = validatedRetpath.body.retpath;
                            }

                            if (validatedBackpath.body && validatedBackpath.body.retpath) {
                                locals.store.common.backpath = validatedBackpath.body.retpath;
                                locals.store.header.backpath = validatedBackpath.body.retpath;
                            }

                            locals.store.changeAvatar.isCropNotSupported = isUnsupportedBro(uatraits);
                        })
                        .catch((err) => {
                            PLog.warn()
                                .logId(req.logID)
                                .type('profile.passport.v2')
                                .write(err);
                        })
                        .finally(() => {
                            delete locals.account;
                            delete locals.accounts;

                            return next();
                        });
                },
                renderPage
            ]);
    },
    enter,
    getBundle,
    prepareFactory
};
