var isIntranet = process.env.INTRANET === 'intranet';
var apiSetup = require('./common/apiSetup');
var createCSRFValidator = require('./common/createCSRFValidator');
var PForm = require('pform');
var config = require('../configs/current');
var url = require('url');
var locs = require('../loc/auth.json');
var when = require('when');
var util = require('util');
var _ = require('lodash');
var Api = require('../lib/api/passport');
var OauthApi = require('../lib/api/oauth');
var tokensModel = require('papi/OAuth/models/Token');
var dheaderSetup = require('./common/dheaderSetup');
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

exports.pass = {};

var setup = [
    function(req, res, next) {
        res.locals.isPdd = Boolean(req.params.pdd_domain);

        if (!req.cookies['Session_id']) {
            next('sessionid.invalid');
        } else {
            next();
        }
    },
    apiSetup,
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('password')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    function(req, res, next) {
        _.assign(res.locals, {
            formType: 'base',
            loc: locs[res.locals.language]['Passwd'],
            isIntranet
        });

        res.page = 'password';
        next();
    }
];

var getForm = {
    base() {
        return new PForm(
            new (require('../blocks/control/password-current/password-current.field'))()
                .setOption('showEntry', true)
                .setRequired(),
            new (require('../blocks/control/password/password.field'))()
                .setLabel('%field_password_new')
                .setOption('showEntry', true)
                .setRequired(),
            new (require('../blocks/control/password-confirm/password-confirm.field'))()
                .setOption('showEntry', true)
                .setRequired(),
            new (require('../blocks/control/captcha/captcha.field'))()
                .setLabel('%field_answer_without_colon')
                .setOption('asyncCheck', true),
            new (require('../blocks/control/login/login.field'))()
        );
    },
    captcha_and_phone() {
        return new PForm(
            new (require('../blocks/control/password-current/password-current.field'))()
                .setOption('showEntry', true)
                .setRequired(),
            new (require('../blocks/control/password/password.field'))()
                .setOption('showEntry', true)
                .setLabel('%field_password_new')
                .setRequired(),
            new (require('../blocks/control/password-confirm/password-confirm.field'))()
                .setOption('showEntry', true)
                .setRequired(),
            new (require('../blocks/control/captcha/captcha.field'))()
                .setLabel('%field_answer_without_colon')
                .setOption('asyncCheck', true),
            new (require('../blocks/control/phone-confirm/phone-confirm.field'))().setRequired().setOptions({
                hasConfirmedPhone: true,
                prevCheckCaptcha: true,
                mode: 'tracked'
            }),
            new (require('../blocks/control/login/login.field'))()
        );
    }
};

var processError = function(err, req, res, next) {
    if (!err) {
        return next();
    }

    var exitErrors = ['account.disabled', 'account.disabled_on_deletion', 'sessionid.invalid', 'sslsession.required'];

    var needExit;
    var origUrl;
    var reUrl;
    var reUrlQuery;
    var reUrlPathname;

    if (util.isArray(err)) {
        needExit = err.some(function(code) {
            return exitErrors.indexOf(code) > -1;
        });
    } else {
        needExit = exitErrors.indexOf(err) > -1;
    }

    if (needExit) {
        origUrl = url.parse(req.originalUrl);
        reUrlQuery = {
            mode: 'add-user',
            retpath: url.format(
                _.assign(origUrl, {
                    protocol: req.headers['x-real-scheme'],
                    host: req.hostname
                })
            )
        };

        reUrlPathname = '/auth/';

        if (util.isArray(err) && err.indexOf('sslsession.required') !== -1) {
            reUrlPathname = '/auth/secure/';
            delete reUrlQuery.mode;
        }

        if (res.locals.isPdd) {
            reUrlQuery['pdd_domain'] = req.params.pdd_domain;
        }

        reUrl = url.format({
            query: reUrlQuery,
            pathname: reUrlPathname
        });

        require('plog')
            .info()
            .logId(req.logID)
            .type('passport', 'passwordRoute', 'processError')
            .write('Account error, needs redirect. Redirecting to %s', reUrl);

        res.redirect(reUrl);
        return;
    }

    if (util.isArray(err) ? err.indexOf('account.2fa_enabled') > -1 : err === 'account.2fa_enabled') {
        var controller = req._controller;
        var redirectUrl = url.format(
            _.assign(controller.getUrl(), {
                pathname: '/profile/access',
                search: null,
                query: {
                    changepass: Date.now()
                }
            })
        );

        require('plog')
            .info()
            .logId(req.logID)
            .type('passport', 'passwordRoute', 'processError')
            .write('2fa_enabled for this user. Redirecting to profile/access on %s', redirectUrl);

        return controller.redirect(redirectUrl);
    }

    var domain = (res.locals.user && res.locals.user.domain && res.locals.user.domain.unicode) || '';
    var locErr = locs[res.locals.language]['Errors'];
    var locMend = locs[res.locals.language]['Mend'];
    var errors = [];

    var aliases = {
        password: {
            name: 'password',
            prohibitedsymbols: 'invalid',
            tooshort: 'short',
            toolong: 'long',
            missingvalue: 'empty',
            likeoldpassword: 'equals_previous',
            foundinhistory: 'found_in_history'
        },
        current_password: {
            name: 'password',
            invalid: 'not_matched',
            not_matched: 'blah-blah' // пересекаются коды ошибок
        },
        captcha: {
            name: 'captcha'
        },
        answer: {
            name: 'captcha',
            missingvalue: 'required'
        },
        phone_number: {
            name: 'phone',
            needsconfirmation: 'required'
        }
    };

    var errIds = {
        'account.disabled': 'ErrorsTexts.badlog_blocked',
        'account.not_found': 'ErrorsTexts.deleted',
        'account.disabled_on_deletion': 'ErrorsTexts.deleted',
        'account.disabled_with_money': 'ErrorsTexts.deleted',
        internal: 'ErrorsTexts.internal',
        'backend.blackbox_failed': 'ErrorsTexts.internal',
        'backend.database_failed': 'ErrorsTexts.internal',
        'backend.redis_failed': 'ErrorsTexts.internal',
        'backend.yasms_failed': 'ErrorsTexts.internal',
        'sessionid.invalid': 'ErrorsTexts.auth_try_again',
        'account.compromised': 'ErrorsTexts.account_compromised',
        password_change_forbidden: 'password.pdd.forbidden',
        complete_social: 'account_without_password',
        'account.without_password': 'account_without_password'
    };

    var mendIds = {
        internal: 'crap',
        'account.disabled': 'disabled',
        'account.not_found': 'deleted',
        'account.disabled_on_deletion': 'deleted',
        'account.disabled_with_money': 'deleted'
    };

    var skip = [
        'password.long',
        'password.short',
        'password.weak',
        'password.prohibitedsymbols',
        'password.likelogin',
        'password.equals_previous',
        'password.found_in_history',
        'password.not_matched',
        'password.likephonenumber',
        'password.empty',
        'current_password.empty',
        'captcha.required',
        'captcha.incorrect',
        'captcha.captchalocate',
        'phone.required',
        // 'phone.compromised',
        'phone_secure.bound_and_confirmed'
    ];

    var skipFormErrors = ['complete_social', 'password_change_forbidden', 'account.without_password'];

    var internal = [
        'ip.empty',
        'host.empty',
        'cookie.empty',
        'exception.unhandled',
        'track.invalid_state',
        'consumer.empty',
        'consumer.invalid'
    ];

    function alternativeErrorMsg(code) {
        var f = res.locals.form && res.locals.form.control;

        if (!f) {
            return false;
        }

        f.forEach(function(item) {
            if (util.isArray(item.error)) {
                item.error.forEach(function(err) {
                    if (code.match('phone_secure')) {
                        code = code.replace('phone_secure', 'phone');
                    }

                    var controlName = (aliases[item.name] && aliases[item.name]['name']) || item.name;
                    var errorCode = (aliases[item.name] && aliases[item.name][err.code]) || err.code;

                    if (code === [controlName, errorCode].join('.')) {
                        err.active = true;
                    }
                });
            }
        });
    }

    function getErrorMessage(code) {
        var errId = code;

        if (skip.indexOf(code) !== -1) {
            alternativeErrorMsg(code);
            return;
        }

        if (internal.indexOf(code) !== -1) {
            errId = 'internal';
        }

        if (!errIds[errId]) {
            errId = 'internal';
        }

        if (skipFormErrors.indexOf(errId) !== -1) {
            res.locals.message = locErr[errIds[errId]];
            res.locals.skipForm = true;

            if (errId === 'password_change_forbidden') {
                res.locals.message = res.locals.message.replace(/%1/g, domain);
            }

            return;
        }

        var error = {
            code,
            msg: locErr[errIds[errId]]
        };

        if (mendIds[errId]) {
            var login = (res.locals.user && res.locals.user.login) || req.body.login || '';

            if (util.isArray(mendIds[errId])) {
                var result = [];

                mendIds[errId].forEach(function(id) {
                    result.push(locMend[id].replace('%1', login));
                });
                error.mend = result.join(' ');
            } else {
                error.mend = locMend[mendIds[errId]].replace('%1', login);
            }
        }
        errors.push(error);
    }

    if (util.isArray(err)) {
        err.forEach(function(code) {
            getErrorMessage(code);
        });
    } else {
        getErrorMessage(err);
    }

    res.locals.errors = errors;
    require('plog')
        .info()
        .logId(req.logID)
        .type('passport', 'passwordRoute', 'processError')
        .write('Changepass input error %s', err);

    next();
};

var parseInfo = function(req, res, data) {
    var user = {};

    if (data) {
        if (data.account) {
            user = _.clone(data.account);
            user.phone = data.number || null;
            user.yu = req.cookies['yandexuid'] || null;

            if (user.uid) {
                var avatar = _.clone(config.paths.avatar);

                avatar.protocol = req.headers['x-real-scheme'];
                avatar.pathname = avatar.pathname.replace('%uid%', user.uid).replace('%size%', 'middle');
                avatar.query = {
                    rnd: String(Math.round(Math.random() * 1000000))
                };
                user.avatar = url.format(avatar);
            }

            if (user.domain && user.domain.unicode && user.login) {
                var dn = [user.login, '@', user.domain.unicode].join('');

                if (user.display_name) {
                    user.display_name.name = dn;
                } else {
                    user.display_name = {
                        name: dn
                    };
                }
            }
        }

        if (data.revokers) {
            if (data.revokers.allow_select) {
                res.locals.select_revokers = true;
            }
        }
    }

    res.locals.user = user;

    if (data.validation_method && data.validation_method === 'captcha_and_phone') {
        res.locals.formType = 'captcha_and_phone';
    }

    res.locals.track_id = req.body.track_id || data.track_id || null;
};

var getHistoryTokens = function(req, res, next) {
    if (res.locals.select_revokers) {
        var controller = req._controller;
        var api = new Api(
            req.logID,
            controller.getHeaders({withPassportServiceTicket: true}),
            res.locals.language,
            controller.getAuth().getUid()
        );
        var dateToChunks = controller.dateToChunks;

        api.getLastAuth()
            .then(function(result) {
                res.locals.historyTokens = {
                    tokens: result.tokens.concat(result.appPass).map(function(entry) {
                        return {
                            id: entry.getTokenId(),
                            lastUsage: dateToChunks(entry.getTimestamp())
                        };
                    })
                };
                next();
            })
            .catch(function(error) {
                processError(error, req, res, next);
            });
    } else {
        next();
    }
};

var processTokensList = function(allTokens, controller) {
    var IRRELEVANT_ISSUE_TIME_CUTOFF = 15552000000; // 6 months in ms
    var isInstanceOfTokenModel = allTokens.every(function(token) {
        return token instanceof tokensModel;
    });

    var dateToChunks = controller.dateToChunks;

    if (Array.isArray(allTokens) && isInstanceOfTokenModel) {
        return allTokens.reverse().map(function(token) {
            var isAppPassword = token.isAppPassword();
            var hasDeviceId = Boolean(token.getDeviceId());
            var issueTime = token.getIssueTime();
            var extendedScopes;

            var base = {
                isAppPassword,
                hasDeviceId,
                id: token.getId(),
                client: isAppPassword ? token.getClientSlug() : token.getClient().getTitle(),
                isYandex: token.getClient().isYandex(),
                clientIcon: token.getClient().getIcon(),
                clientId: token.getClient().getId(),
                clientDescription: token.getClient().getDescription(),
                scopes: token.getScopes().map(function(scope) {
                    return scope.getTitle();
                }),
                extendedScopes: [],
                issueTime: Date.now() - IRRELEVANT_ISSUE_TIME_CUTOFF < issueTime ? dateToChunks(issueTime) : null
            };

            extendedScopes = token.getScopes().groupBy(function(scope) {
                return scope.getSectionTitle();
            });

            _.forEach(extendedScopes, function(scopes, section) {
                var items = _.map(scopes, function(scope) {
                    return scope.getTitle();
                });

                base.extendedScopes.push({
                    section,
                    items
                });
            });

            if (hasDeviceId) {
                _.assign(base, {
                    deviceName: token.getDeviceName(),
                    deviceId: token.getDeviceId()
                });
            }
            return base;
        });
    }
};

var getTokensList = function(req, res, next) {
    if (res.locals.select_revokers) {
        var controller = req._controller;

        return controller
            .getAuth()
            .sessionID({
                // glogouttime, revoker.tokens, revoker.app_passwords and revoker.web_sessions,
                // see https://beta.wiki.yandex-team.ru/passport/dbmoving
                attributes: '4,134,135,136'
            })
            .then(function() {
                var oauthApi = new OauthApi(
                    controller.getLogId(),
                    controller.getHeaders(),
                    res.locals.language,
                    controller.getAuth().getUid()
                );

                return oauthApi
                    .listTokensVer3()
                    .then(function(response) {
                        var slugs = _.map(config.appPasswordsClientIdMapping, function(name, clientId) {
                            return {
                                name,
                                client_id: clientId
                            };
                        });

                        res.locals.slugs = {clientSlugs: slugs.length ? slugs : null};
                        res.locals.tokens = processTokensList(response, controller);
                        res.locals.oauthUrl = config.paths.oauth.replace('%tld%', controller.getTld());
                    })
                    .then(function() {
                        next();
                    });
            })
            .catch(function(error) {
                processError(error, req, res, next);
            });
    } else {
        next();
    }
};

var renderPage = [
    getTokensList,
    getHistoryTokens,
    function(req, res, next) {
        var form = getForm[res.locals.formType || 'base']().setApi(req.api);

        var preFiled = {
            phone_number:
                (res.locals.user && res.locals.user.phone && res.locals.user.phone.masked_international) || null,
            login: (res.locals.user && res.locals.user.login) || null
        };

        var phone = form.getField('phone_number');

        if (phone && preFiled['phone_number']) {
            phone.setValue(preFiled['phone_number']);

            var curNumber = req.body && req.body.hidden_phone;
            var userNumber = preFiled.phone_number;

            if (curNumber && userNumber && userNumber === curNumber) {
                phone.setOption('state', 'confirmed');
            }
        }

        if (preFiled['login']) {
            form.getField('login').setValue(preFiled['login']);
        }

        if (res.locals.user && res.locals.user.person && res.locals.user.person.country) {
            form.getField('answer').setOption('country', res.locals.user.person.country);
        }

        when.resolve()
            .then(function() {
                return form.validate(req.body);
            })
            .then(function() {
                return form.compile(res.locals.language);
            })
            .then(function(data) {
                res.locals.form = {
                    control: data
                };

                if (res.formErrors || res.captchaError) {
                    if (res.captchaError) {
                        res.formErrors = [res.captchaError];
                    }

                    next(res.formErrors);
                } else {
                    next();
                }
            });
    },
    processError,
    function(req, res) {
        res.render(`profile.${res.page}.${res.locals.language}.js`);
    }
];

var checkCaptcha = function(req, res, next) {
    if (!req.body.key) {
        return next();
    }

    req.api.captchaCheckStatus(req.body).then(
        function(result) {
            var body = (result && result.body) || null;

            if (body && body.is_recognized && body.is_checked) {
                return next();
            }

            req.api.captchaCheck(req.body).then(
                function(result) {
                    delete req.body.answer;
                    if (result.body.correct) {
                        next();
                    } else {
                        res.captchaError = 'captcha.incorrect';
                        next();
                    }
                },
                function() {
                    res.captchaError = 'captcha.captchalocate';
                    next();
                }
            );
        },
        function() {
            next();
        }
    );
};

exports.pass.enter = [
    function(req, res, next) {
        if (parseInt(req.query.new, 10) === 1) {
            next('route');
        } else {
            next();
        }
    },
    setup,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function passwordSubmit(req, res, next) {
        when(
            req.api.changepassSubmit({is_pdd: res.locals.isPdd, retpath: req.body['retpath'] || req.query['retpath']})
        ).then(
            function(r) {
                parseInfo(req, res, r.body);

                if (r.body.state) {
                    res.formErrors = r.body.state;
                }

                next();
            },
            function(error) {
                res.formErrors = error;
                next();
            }
        );
    },
    function(req, res, next) {
        const retpath = req.body.retpath || req.query.retpath;

        res.locals.passportHost = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname
        });

        if (retpath) {
            return req.api
                .validateRetpath({retpath})
                .then(({body = {}}) => {
                    if (body.retpath) {
                        res.locals.retpath = body.retpath;
                    }
                })
                .catch((err) => {
                    require('plog')
                        .warn()
                        .logId(req.logID)
                        .type('password')
                        .write(err);
                })
                .done(next);
        }

        return next();
    },
    function(req, res, next) {
        var controller = req._controller;

        controller.prepareStatData({
            mode: 'change_password_voluntarily',
            track_id: res.locals.track_id || null,
            uid: res.locals.user && res.locals.user.uid,
            origin: (req.nquery && req.nquery.origin) || null
        });

        return next();
    },
    renderPage
];

exports.pass.submit = [
    createCSRFValidator(
        function(req, res, next) {
            return next();
        },
        function(req, res) {
            return res.redirect('/profile/password');
        }
    ),
    setup,
    checkCaptcha,
    function passwordCommit(req, res, next) {
        var controller = req._controller;

        if (req.body.allow_select_revokers) {
            res.locals.select_revokers = true;
        }
        req.api
            .changepassCommit(_.merge({is_pdd: res.locals.isPdd}, req.body))
            .then(function(r) {
                var result = r.body;

                parseInfo(req, res, result);

                if (result.errors) {
                    res.formErrors = result.errors;
                }

                controller.augmentResponse(result);

                next();
            })
            .catch(function(errors) {
                res.locals.track_id = req.body && req.body.track_id;

                if (Array.isArray(errors)) {
                    res.formErrors = errors;
                }
                return next();
            });
    },
    function redirectToFinish(req, res, next) {
        if (res.formErrors) {
            return next();
        }

        var pathname = '/auth/finish/';

        req.locals = res.locals;

        res.redirect(
            url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname,
                query: {
                    track_id: req.body['track_id'] || res.locals.track_id
                }
            })
        );
    },
    renderPage
];
