const url = require('url');
const express = require('express');
const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const validateCSRF = require('./common/validateCSRF.js');
const OauthApiLib = require('../lib/api/oauth');
const moment = require('moment');

const commonRouter = express.Router();
const pddRouter = express.Router();

function createOAuthApi(req, res, next) {
    req.oauthApi =
        req.oauthApi ||
        new OauthApiLib(
            req._controller.getLogId(),
            req._controller.getHeaders(),
            req.body.lang,
            req._controller.getAuth().getUid()
        );
    next();
}

function newFilter(req, res, next) {
    if (parseInt(req.query.new, 10) === 1) {
        return next();
    }

    return next('route');
}

function redirect(req, res) {
    const query = Object.assign({}, req.query);

    delete query.mode;
    res.redirect(
        url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/profile/password',
            query
        })
    );
}

pddRouter.get('/for/:pdd_domain/profile/password', newFilter, redirect);
pddRouter.get('/passport/?|/for/:pdd_domain*', newFilter, (req, res, next) => {
    const mode = req.body.mode || (req.nquery && req.nquery.mode);

    if (mode === 'changepass') {
        return redirect(req, res);
    }

    return next('route');
});

function revokeTokens(req, res) {
    const devices = req.body.tokens ? req.body.tokens.split(',') : [];
    const tokens = req.body.passTokens ? req.body.passTokens.split(',') : [];
    const other = req.body.otherTokens ? req.body.otherTokens.split(',') : [];
    const otherYandex = req.body.otherYandexTokens ? req.body.otherYandexTokens.split(',') : [];
    const count = devices.length + tokens.length + other.length + otherYandex.length;
    const failed = {
        passwords: 0,
        devices: 0
    };

    let finished = 0;
    const done = (status, key) => {
        if (status === 'failed') {
            ++failed[key];
        }

        if (++finished >= count) {
            res.json({
                status: 'ok',
                failed
            });
        }
    };

    if (!count) {
        res.json({
            status: 'ok',
            failed
        });
        return;
    }

    devices.forEach((token) => {
        req.oauthApi
            .revokeDeviceTokens(token)
            .then(done)
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type(`profile.password.new-revoke_device_tokens: ${token}`)
                    .write(err);
                done('failed', 'devices');
            });
    });
    tokens
        .concat(other)
        .concat(otherYandex)
        .forEach((token) => {
            req.oauthApi
                .revokeSingleTokenVer3(Number(token))
                .then(done)
                .catch((err) => {
                    PLog.warn()
                        .logId(req.logID)
                        .type(`profile.password.new-revoke_single_token: ${token}`)
                        .write(err);
                    done('failed', 'passwords');
                });
        });
}

function checkCaptcha(req, res, next) {
    req.api
        .captchaCheck({
            track_id: res.locals.password_track,
            key: req.body.captcha_key,
            answer: req.body.captcha
        })
        .then((response) => {
            const body = response.body;

            if (Array.isArray(body)) {
                for (let i = 0; i < body.length; i++) {
                    if ((body[i] || {}).code === 'captchalocate') {
                        return res.json({
                            status: 'error',
                            errors: ['captcha.cannot_locate']
                        });
                    }
                }

                return res.json({
                    status: 'error',
                    errors: ['captcha.not_matched']
                });
            } else if (body.correct === false) {
                return res.json({
                    status: 'error',
                    errors: ['captcha.not_matched']
                });
            }

            return next();
        })
        .catch(() =>
            res.json({
                status: 'error',
                errors: ['captcha.cannot_locate']
            })
        );
}

commonRouter.post('/submit', [
    apiSetup,
    validateCSRF,
    checkCaptcha,
    function(req, res) {
        const {
            current_password,
            password,
            is_pdd,
            revoke_web_sessions = false, // eslint-disable-line camelcase
            revoke_tokens = false, // eslint-disable-line camelcase
            revoke_app_passwords = false // eslint-disable-line camelcase
        } = req.body;

        const controller = req._controller;

        req.api
            .changepassCommit({
                current_password,
                password,
                track_id: res.locals.password_track,
                is_pdd,
                revoke_web_sessions,
                revoke_tokens,
                revoke_app_passwords
            })
            .then((result) => {
                controller.augmentResponse(result.body);

                res.json({
                    track_id: res.locals.password_track,
                    status: 'ok'
                });
            })
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.password.new-commit')
                    .write(errors);
                res.json({
                    status: 'error',
                    errors
                });
            });
    }
]);

commonRouter.post('/revoke', [
    apiSetup,
    function(req, res, next) {
        const auth = req._controller.getAuth();
        const authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        authRequest
            .then(() => next())
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.password.new-auth_request')
                    .write(err);
                res.json({status: 'error'});
            });
    },
    createOAuthApi,
    revokeTokens
]);

commonRouter.post('/revokers', [
    function(req, res, next) {
        if (req.body && req.body.track_id && req.headers['x-requested-with'] === 'XMLHttpRequest') {
            return next();
        }

        return res.sendStatus(403);
    },
    apiSetup,
    validateCSRF,
    function(req, res, next) {
        req.api.changepassSubmit(req.body).then(
            (result = {}) => {
                const body = result.body || result;

                if (!body.revokers || !body.revokers.allow_select) {
                    return res.json(body);
                }

                if (body.revokers.default) {
                    const def = body.revokers.default;
                    const keys = ['tokens', 'web_sessions', 'app_passwords'];

                    keys.forEach((key) => {
                        if (def.hasOwnProperty(key)) {
                            def[`revoke_${key}`] = def[key];
                            delete def[key];
                        }
                    });
                } else {
                    body.revokers.allow_select = false;
                    return res.json(body);
                }

                body.revokers.passTokens = [];
                body.revokers.tokens = [];
                body.revokers.otherTokens = [];
                res.locals.passwordStatus = body;
                return next();
            },
            (errors) =>
                res.status(200).json({
                    status: 'error',
                    errors: errors || []
                })
        );
    },
    function(req, res, next) {
        const auth = req._controller.getAuth();
        const authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        authRequest.then(() => next()).catch(() => res.json(res.locals.passwordStatus));
    },
    createOAuthApi,
    function(req, res) {
        const passwordStatus = res.locals.passwordStatus;

        return req.oauthApi
            .listTokenGroups()
            .then((response) => {
                if (response.status === 'ok') {
                    passwordStatus.revokers.otherTokens = (response.other_tokens || [])
                        .filter(({client}) => !client.is_yandex)
                        .map(({id, client}) => ({
                            token: id,
                            title: client.title
                        }));
                    passwordStatus.revokers.otherYandexTokens = (response.other_tokens || [])
                        .filter(({client}) => client.is_yandex)
                        .map(({id, client}) => ({
                            token: id,
                            title: client.title
                        }));
                    passwordStatus.revokers.tokens = (response.device_tokens || []).map((device) => ({
                        title: device.device_name,
                        token: device.device_id,
                        count: (device.tokens || []).length,
                        created:
                            (device.tokens || []).length > 0 &&
                            moment(device.tokens[0].create_time * 1000)
                                .locale(req.body.lang)
                                .format('D MMMM'),
                        last:
                            (device.tokens || []).length > 0 &&
                            moment(device.tokens[0].issue_time * 1000)
                                .locale(req.body.lang)
                                .fromNow()
                    }));
                    passwordStatus.revokers.passTokens = (response.app_passwords || []).map((token) => ({
                        token: token.id,
                        created:
                            token.create_time &&
                            moment(token.create_time * 1000)
                                .locale(req.body.lang)
                                .format('D MMMM'),
                        title: token.device_name || token.client.title
                    }));
                    res.json(passwordStatus);
                }
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.password.new-get_grouped_tokens')
                    .write(err);
                res.json(passwordStatus);
            });
    }
]);

exports.router = {
    common: commonRouter,
    pdd: pddRouter
};
