'use strict';

var url = require('url');
var _ = require('lodash');
var PForm = require('pform');
var PLog = require('plog');
var PUtils = require('putils');
var apiSetup = require('./common/apiSetup');
var CaptchaField = require('../blocks/control/captcha/captcha.field');
var yasmsStateParser = require('./common/yasmsStateParser');
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
var validateCSRF = require('./common/validateCSRF.js');
var dheaderSetup = require('./common/dheaderSetup');
const checkAuth = require('./common/checkAuth');
const yaphoneLite = require('./common/yaphoneLite');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

function getForm() {
    return new PForm(new CaptchaField().setOptions({asyncCheck: true, popup: true}).setRequired());
}

function writeUserPhone(req, res, next) {
    var data = {
        namespace: req.body.ysaNamespace,
        esecId: req.body.ysaEsecId,
        events: [
            {
                name: 'provide-phone-number',
                ts: Math.floor(Date.now() / 1000),
                payload: {
                    phone_number: req.body.number
                }
            }
        ]
    };

    if (!data.esecId) {
        return next();
    }

    return req.api
        .ysaWriteUserEvent(data)
        .then(function() {
            next();
        })
        .catch(function() {
            next();
        });
}

var manageSms2fa = [
    apiSetup,
    validateCSRF,
    function checkCode(req, res, next) {
        req.api
            .confirmPhoneCommit(req.body.code)
            .then(function(response) {
                if (response.status === 'ok') {
                    return next();
                }

                return res.json({
                    status: 'error',
                    errors: response.errors || []
                });
            })
            .catch(function(error) {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    },
    function toggleSms2fa(req, res) {
        return req.api
            .toggleSms2fa({isEnabled: req.body.isEnabled})
            .then(function(response) {
                const body = response.body;

                if (body && body.status === 'ok') {
                    return res.json({status: 'ok'});
                }

                return res.json({status: 'error'});
            })
            .catch(function(error) {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    }
];

var secureBindCommit = [
    apiSetup,
    validateCSRF,
    function checkCode(req, res, next) {
        req.api
            .yasmsSecurePhoneCommit(req.body)
            .then(function(response) {
                if (response.status === 'ok') {
                    if (req.body.isSms2fa) {
                        return next();
                    }

                    return res.json({status: 'ok'});
                }

                return res.json({
                    status: 'error',
                    errors: response.errors || []
                });
            })
            .catch(function(error) {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    },
    function toggleSms2fa(req, res) {
        return req.api
            .toggleSms2fa({isEnabled: true})
            .then(function(response) {
                const body = response.body;

                if (body && body.status === 'ok') {
                    return res.json({status: 'ok'});
                }

                return res.json({
                    status: 'error'
                });
            })
            .catch(function(error) {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    }
];

exports.route = function(app) {
    app.get('/profile/phones', this.enter);
};

exports.enter = [
    checkAuth,
    multiAuthAccountsSetup,
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getAuth()
            .sessionID({
                attributes: '200,1003', // isSms2FA or is2FA?
                userinfo: true
            })
            .then(function(sessioninfo) {
                if (sessioninfo) {
                    const {attributes} = sessioninfo || {};

                    _.assign(res.locals, {
                        aliasAvailable: !sessioninfo.uid.hosted, // it isn't PDD account
                        isSms2FAOn: attributes['200'] === '1',
                        is2FAOn: attributes['1003'] === '1',
                        canSwitchInputType: true
                    });
                }

                next();
            })
            .catch(function(error) {
                if (error && error.code === 'need_resign') {
                    return controller.getAuth().resign();
                }

                if (error) {
                    return next(error);
                }

                return null;
            });
    },
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('profile.phones')
                    .write(err);
            })
            .done(function() {
                next();
            });
    },
    apiSetup,
    function(req, res, next) {
        req.api.initTrack().then(function(data) {
            var trackId = (data && data.body && data.body.track_id) || undefined;

            if (trackId) {
                res.locals.track = trackId;
                req.api.track(trackId);
            }

            next();
        });
    },
    function(req, res, next) {
        var controller = req._controller;

        req.api
            .yasmsGetState()
            .then(function(data) {
                if (data.status === 'error') {
                    if (data.errors && data.errors.length) {
                        if (data.errors.indexOf('request.credentials_all_missing') !== -1) {
                            return controller.getAuth().authorize();
                        }

                        if (data.errors.indexOf('sessionid.invalid') !== -1) {
                            return controller.getAuth().authorize();
                        }

                        if (data.errors.indexOf('sslsession.required') !== -1) {
                            return controller.getAuth().obtainSecureCookie();
                        }
                    }

                    PLog.warn()
                        .logId(req.logID)
                        .type('profile.phones')
                        .write(data.errors);
                }

                if (data.status === 'ok') {
                    if (data && data.account) {
                        _.assign(res.locals, yasmsStateParser.parse(data.account));
                    }
                }

                return null;
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.phones')
                    .write(err);
            })
            .done(function() {
                next();
            });
    },
    function getState(req, res, next) {
        const controller = req._controller;
        const {accounts = {}} = res.locals;
        const {defaultAccount = {}} = accounts;
        const {isNeoPhonish} = defaultAccount;

        req.api
            .profileGetState()
            .then(function(response) {
                const profile = response.body;
                const account = profile.account || {};

                res.locals.account = account;
                res.locals.isSocialchik = (account.password_info || {}).strength === -1 && !account.is_2fa_enabled;
                res.locals.isNeoPhonish = isNeoPhonish;

                return next();
            })
            .catch(function(errors) {
                let query;

                PLog.warn()
                    .logId(req.logID)
                    .type('profile.phones')
                    .write(errors);

                if (Array.isArray(errors)) {
                    if (errors.includes('sessionid.invalid') || errors.includes('account.disabled')) {
                        return controller.redirectToAuth();
                    }

                    if (errors[0].code && errors[0].code === 'unknowntrack') {
                        query = Object.assign({}, req.query);

                        delete query.track_id;

                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: req.path,
                                query
                            })
                        );
                    }
                }

                return next(errors);
            });
    },
    function getCkey(req, res, next) {
        res.locals._sk = PUtils.csrf(null, req.cookies.yandexuid);
        next();
    },
    function(req, res, next) {
        var form = getForm().setApi(req.api);

        form.compile(res.locals.language).then(
            function(data) {
                res.locals.form = {
                    control: data
                };

                next();
            },
            function(err) {
                next(err);
            }
        );
    },
    function(req, res, next) {
        var retpath = req.query.retpath;

        if (!retpath) {
            return next();
        }

        req.api
            .validateRetpath({retpath})
            .then(function(result) {
                if (result.body && result.body.retpath) {
                    res.locals.retpath = result.body.retpath;
                }
                next();
            })
            .catch(function() {
                next();
            });
    },

    function(req, res, next) {
        res.locals.passportHost = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname
        });

        return next();
    },
    getYaExperimentsFlags,
    dheaderSetup,
    yaphoneLite,
    function(req, res) {
        const {experiments = {}} = res.locals;
        const {flagsString, boxes} = experiments;

        req.api.statboxLogger({
            ignoreMissedTrack: true,
            track_id: res.locals.track_id || null,
            action: 'opened',
            mode: 'yasms',
            host: req.hostname,
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            yandexuid: req.cookies.yandexuid,
            origin: (req.nquery && req.nquery.origin) || null,
            experiment_flags: flagsString,
            experiment_boxes: boxes
        });

        res.locals.helpLinkType = 'phones';
        res.locals.isShowPhonesForNonPasswordExp = true;

        res.render('profile.phones.ru.js');
    }
];

exports.writePhoneToYsa = writeUserPhone;
exports.manageSms2fa = manageSms2fa;
exports.secureBindCommit = secureBindCommit;
