const url = require('url');
const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const langSetup = require('./common/langSetup');
const createState = require('./common/createState');
const createFormState = require('./common/createFormState');
const getMetrics = require('./common/getMetrics');
const getUatraitsData = require('./common/getUatraitsData');
const rumCounterSetup = require('./common/rumCounterSetup');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const validateCSRF = require('./common/validateCSRF.js');
const checkAuth = require('./common/checkAuth');
const yaphoneLite = require('./common/yaphoneLite');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const getCaptcha = require('./common/getCaptcha');
const setTheme = require('./authv2/setTheme');
const {urlFormat} = require('./common/urlFormat');

const ENTRY_PAGES_MAP = {
    main: 'main',
    addPhone: 'addPhone'
};

const MODAL_PAGES_MAP = {
    addPhone: 'addPhone'
};

const getPage = (entry) => ENTRY_PAGES_MAP[entry] || ENTRY_PAGES_MAP.main;
const getModal = (modal) => MODAL_PAGES_MAP[modal] || null;

const manageSms2fa = [
    apiSetup,
    validateCSRF,
    function checkCode(req, res, next) {
        req.api
            .confirmPhoneCommit(req.body.code)
            .then((response) => {
                if (response.status === 'ok') {
                    return next();
                }

                return res.json({
                    status: 'error',
                    errors: response.errors || []
                });
            })
            .catch((error) => {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    },
    function toggleSms2fa(req, res) {
        return req.api
            .toggleSms2fa({isEnabled: req.body.isEnabled})
            .then((response) => {
                const body = response.body;

                if (body && body.status === 'ok') {
                    return res.json({status: 'ok'});
                }

                return res.json({status: 'error'});
            })
            .catch((error) => {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    }
];

const setup = [
    apiSetup,
    checkAuth,
    getUatraitsData,
    langSetup,
    getYaExperimentsFlags,
    function checkExp(req, res, next) {
        if (req._controller.hasExp('profile-phones-v2-exp')) {
            return next();
        }

        return next('route');
    },
    function getTrack(req, res, next) {
        return req.api
            .initTrack({
                type: 'authorize'
            })
            .then(function(response) {
                res.locals.track_id = (response.body && response.body.track_id) || '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('phones')
                    .write(errors);

                return next();
            });
    },
    multiAuthAccountsSetup,
    function getState(req, res, next) {
        req.api
            .profileGetState()
            .then((response) => {
                const profile = response.body;

                res.locals.account = profile.account || {};

                return next();
            })
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.phones')
                    .write(errors);

                if (Array.isArray(errors)) {
                    const retpath = url.format(req._controller.getUrl());

                    if (errors.indexOf('sessionid.invalid') !== -1 || errors.indexOf('account.disabled') !== -1) {
                        return res.redirect(
                            url.format(
                                Object.assign({}, req._controller.getAuthUrl(), {
                                    query: {retpath}
                                })
                            )
                        );
                    }

                    if (errors[0].code && errors[0].code === 'unknowntrack') {
                        const query = Object.assign({}, req.query);

                        delete query.track_id;

                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: req.path,
                                query
                            })
                        );
                    }
                }

                return next(errors);
            });
    },
    function putPhonesInStore(req, res, next) {
        const {locals = {}} = res;
        const {account = {}} = locals;

        locals.store = {
            ...locals.store,
            phones: {
                pageInfo: {},
                error: null,
                isLoading: false,
                restore: [],
                other: [],
                otherInReplaceQuarantine: [],
                completeSms2fa: false,
                hasPhoneAccess: true,
                isCodeConfirmed: false,
                isPasswordConfirmed: false,
                denyResendUntil: '',
                isPasswordRequired: false,
                isCodeRequired: false,
                isCaptchaRequired: false,
                currentSecurePhoneId: '',
                currentSimplePhoneId: '',
                isConfirmationCodeSent: false,
                processedNumber: '',
                isSecondReplaceStep: false,
                settingsPageOptions: {},
                isReplaceAddPhoneStep: false,
                notifications: {
                    isVisible: false,
                    notificationsList: []
                }
            }
        };

        if (account.phones) {
            Object.keys(account.phones).forEach((code) => {
                const phone = account.phones[code];
                const operation = phone.operation || {};
                const alias = phone.alias || {};
                const {in_quarantine: inQuarantine, id, type, started, finished} = operation;
                const {login_enabled: isLoginAlias, email_enabled: isEmailAlias} = alias;
                const operatonObj = {
                    inQuarantine,
                    id,
                    type,
                    started,
                    finished
                };
                const phoneObj = {
                    id: phone.id,
                    isSecure: Boolean(phone.secured),
                    number: phone.number.masked_international,
                    numberMaskedOriginal: phone.number.masked_original,
                    isDefault: phone.is_default,
                    isAlias: phone.is_alias,
                    needProlong: phone.need_admission,
                    operation: phone.operation ? operatonObj : {},
                    isLoginAlias,
                    isEmailAlias
                };

                if (phone.bound) {
                    if (phone.secured) {
                        return res.locals.store.phones['restore'].push(phoneObj);
                    }

                    if (!phone.secured && phone.operation && phone.operation.type === 'mark') {
                        return res.locals.store.phones['otherInReplaceQuarantine'].push(phoneObj);
                    }

                    res.locals.store.phones['other'].push(phoneObj);
                }
            });
        }

        return next();
    },
    createState,
    createFormState,
    getCaptcha(),
    function cancelNotQuarantineOperations(req, res, next) {
        const {store: {phones: {restore = [], other = []} = {}} = {}} = res.locals;

        const allNotQuarantineOperationsId = [...restore, ...other].reduce((acc, phone) => {
            const {operation: {id, inQuarantine} = {}} = phone;

            if (id && !inQuarantine) {
                acc.push(id);
            }

            return acc;
        }, []);

        if (allNotQuarantineOperationsId.length) {
            const cancelPromisesArr = allNotQuarantineOperationsId.map((id) => {
                return new Promise((resolve) => {
                    req.api
                        .yasmsCancelOperation({id})
                        .then(resolve())
                        .catch((error) => {
                            PLog.warn()
                                .logId(req.logID)
                                .type('phones.cancelNotQuarantineOperation, langSwitcher')
                                .write(error);
                        });
                });
            });

            Promise.allSettled([cancelPromisesArr]).then(() => next());
        } else {
            return next();
        }
    },
    getMetrics({
        header: 'Телефоны'
    }),
    rumCounterSetup,
    setTheme,
    yaphoneLite
];

const enterPhonesPage = (entry, isWebview = false) => [
    setup,
    function setPage(req, res, next) {
        const {locals: {store: {phones = {}, settings = {}, common = {}, form = {}} = {}} = {}} = res;
        const origin = req.query && req.query.origin;
        const contentOnly = req.query && req.query.contentOnly === '1';

        settings.origin = origin;
        settings.isWebview = isWebview;
        phones.contentOnly = contentOnly;
        if (isWebview) {
            phones.pageInfo = {page: getPage(entry), modal: null, isModalOpened: false};
        } else {
            phones.pageInfo = settings.isTouch
                ? {page: getPage(entry), modal: null, isModalOpened: false}
                : {page: 'main', modal: getModal(entry), isModalOpened: Boolean(getModal(entry))};
        }

        if (entry === 'addPhone') {
            const {method, body: {phone} = {}} = req;
            const {restore} = phones;
            const {retpath} = common;
            const defaultRetpath = urlFormat({
                hostname: req.hostname,
                pathname: '/profile/phones'
            });
            const hasRestore = Boolean(restore.length);

            if (method === 'POST' && phone) {
                form.values.phone = phone.replace(/[^+-/(/)\d]/g, '');
            }

            if (hasRestore) {
                const finalPath = retpath || defaultRetpath;

                req._controller.redirect(finalPath);
            }
        }

        next();
    },
    function render(req, res) {
        const {experiments = {}, language} = res.locals;
        const {flagsString, boxes} = experiments;

        req.api.statboxLogger({
            ignoreMissedTrack: true,
            track_id: res.locals.track_id || null,
            action: 'opened',
            mode: 'yasms',
            host: req.hostname,
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            yandexuid: req.cookies.yandexuid,
            origin: (req.nquery && req.nquery.origin) || null,
            experiment_flags: flagsString,
            experiment_boxes: boxes
        });

        res.locals.helpLinkType = 'phones';
        res.locals.isShowPhonesForNonPasswordExp = true;

        isWebview ? res.render(`react.phones-webview.${language}.jsx`) : res.render(`react.phones.${language}.jsx`);
    }
];

exports.route = (app) => {
    app.get('/profile/phones', enterPhonesPage('main'));
    app.get('/profile/phones/add', enterPhonesPage('addPhone'));
    app.post('/profile/phones/add', enterPhonesPage('addPhone'));
    app.get('/profile/phones-webview/add', enterPhonesPage('addPhone', true));
    app.post('/profile/phones-webview/add', enterPhonesPage('addPhone', true));
};

exports.manageSms2fa = manageSms2fa;
