const apiSetup = require('./common/apiSetup');
const url = require('url');
const _ = require('lodash');
const PLog = require('plog');
const journal = require('./profile.journal');
const locsSecurity = require('../loc/profilesecurity.json');
const PForm = require('pform');
const PPhoneConfirm = require('../blocks/control/phone-confirm/phone-confirm.field');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const dheaderSetup = require('./common/dheaderSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const securityPage = require('./profile.security.v2');

const setup = [
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('security')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    apiSetup,
    function(req, res, next) {
        req.api.getTrack().then(function(result) {
            res.locals.track_id = result.body.id;
            return next();
        });
    }
];

exports.routes = {};
exports.route = function(app) {
    const routes = this.routes;

    app.get('/profile/security', securityPage);
    app.get('/profile/security', routes.main);
};

exports.routes.main = [
    setup,
    getAccountInfo,
    renderPhoneConfirm,
    journal.getAllScopes,
    journal.preloadAuthEvents,
    journal.getAuthLog,
    journal.getTokensInfo,
    function(req, res, next) {
        const allData = res.locals.listData.itemsToRender;

        res.locals.listData = {
            itemsToRender: journal.groupByDates(allData)
        };

        return next();
    },
    function(req, res, next) {
        const queryRetpath = req.nquery && req.nquery.retpath;

        res.locals.passportHost = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname
        });

        if (queryRetpath) {
            req._controller.validateRetpath(queryRetpath).then(function(response) {
                if (response) {
                    res.locals.serviceRetpath = queryRetpath;
                    res.locals.retpath_orig = queryRetpath;
                }
                return next();
            });
        } else {
            return next();
        }
    },
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function(req, res) {
        const serviceRetpath = res.locals.serviceRetpath;
        const urlObj = {
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/profile/security'
        };

        if (serviceRetpath) {
            urlObj.query = {retpath: serviceRetpath};
        }
        res.locals.retpath = url.format(urlObj);
        res.render(`profile.security.${res.locals.language}.js`);
    }
];

exports.updateData = [setup, getAccountInfo];

function getAccountInfo(req, res, next) {
    req.api
        .profileGetState()
        .then(function(result) {
            const tld = req._controller.getTld();
            const locTexts = locsSecurity[res.locals.language].ProfileSecurity;
            const body = result.body.account;
            const hasSecuredPhone = () => {
                let isNumberBound = false;
                const phonesData = body.phones && body.phones[Object.keys(body.phones)[0]];

                if (phonesData) {
                    const phonesDataKeys = Object.keys(phonesData);

                    isNumberBound = phonesDataKeys.includes('secured') && phonesDataKeys.includes('bound');
                }
                return isNumberBound;
            };
            const securityValues = {
                '-1': 'unknown',
                4: 'low',
                16: 'middle',
                32: 'high'
            };
            const helpLinksMainUrl = `https://yandex.${tld}/support/passport/`;
            const helpLinks = {
                email: `${helpLinksMainUrl}authorization/email.xml`,
                phone: `${helpLinksMainUrl}authorization/phone.xml`,
                passwordAbout: `${helpLinksMainUrl}security.xml#password-done-right`,
                password: `${helpLinksMainUrl}support/passport/security.xml#hacked-how`,
                passwordReason: `${helpLinksMainUrl}security.html`
            };
            const account = {
                securityLevel: securityValues[body.security_level] || 'unknown',
                is2faEnabled: body.is_2fa_enabled,
                isAppPasswordsEnabled: body.app_passwords_enabled,
                phone: createPhoneInfo(body.phones, locTexts, helpLinks),
                email: createEmailInfo(body.emails, locTexts, helpLinks)
            };

            if (!account.is2faEnabled) {
                const pwdData = body && body.password_info;

                account.passwordInfo = createPasswordInfo(pwdData, locTexts, helpLinks);
            }

            if (!hasSecuredPhone()) {
                res.locals.noPhoneNumber = true;
            }

            if (body.question && body.question.text) {
                account.question = body.question.text;
            }

            account.securityMessage = setSecurityMessage(account, res.locals.language);

            if (req.body.status) {
                return res.json(account);
            }

            res.locals.account = account;
            res.locals.yu = req.cookies.yandexuid;
            next();
        })
        .catch(function(error) {
            const errorCode = error[0];
            const retpath = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: '/profile/security/'
            });

            PLog.warn()
                .logId(req.logID)
                .type('getAccountInfo')
                .write(error);

            if (errorCode === 'sslsession.required') {
                return res.redirect(
                    url.format({
                        protocol: req.headers['x-real-scheme'],
                        hostname: req.hostname,
                        pathname: '/auth/secure/',
                        query: {
                            retpath
                        }
                    })
                );
            }

            return res.redirect(
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: '/auth/',
                    query: {
                        retpath
                    }
                })
            );
        });
}

function setSecurityMessage(account, lang) {
    const hasRestorationMethods = (account.phone && account.phone.number) || account.email.addresses;
    const locPath = locsSecurity[lang].ProfileSecurity;
    const pwdInfo = account.passwordInfo || {};
    const isWarningPasswordStatus = pwdInfo.status === 'warning';
    const isNormalPasswordStatus = pwdInfo.status === 'normal';

    if (account.is2faEnabled) {
        return locPath['profile.security.status-good-2fa'];
    }

    if (isWarningPasswordStatus && !hasRestorationMethods) {
        return locPath['profile.security.status-bad']; // все плохо
    }

    if (isWarningPasswordStatus && hasRestorationMethods) {
        return locPath['profile.security.status-bad_pwd']; // плохой пароль
    }

    if (isNormalPasswordStatus && !hasRestorationMethods) {
        return locPath['profile.security.status-bad_restore']; // нет восстановления
    }

    // isNormalPasswordStatus && hasRestorationMethods - ok
    return locPath['profile.security.status-good'];
}

function renderPhoneConfirm(req, res, next) {
    if (res.locals.noPhoneNumber) {
        const form = new PForm(
            new PPhoneConfirm().setRequired().setOptions({
                mode: 'confirmAndBindSecure',
                resendTimer: true
            })
        ).setApi(req.api);

        form.compile(res.locals.language)
            .then(function(data) {
                res.locals.form = {control: data};
            })
            .catch(function() {
                next();
            });
        next();
    } else {
        return next();
    }
}

function checkPasswordStatus(timestamp) {
    const PASSWORD_CREATION_DATE_IN_MS = timestamp * 1000; // Sec to ms
    const YEAR_IN_MS = 60 * 60 * 24 * 365 * 1000;
    const TIME_NOW = Date.now();

    const passwordCreatedDate = new Date(PASSWORD_CREATION_DATE_IN_MS);
    const yearAgo = TIME_NOW - YEAR_IN_MS;
    const isPasswordOutdated = yearAgo - PASSWORD_CREATION_DATE_IN_MS > 0;
    const result = {
        age: 'normal'
    };

    if (isPasswordOutdated) {
        const FIVE_YEAR_IN_MS = YEAR_IN_MS * 5;
        const isPasswordOlderThanFive = FIVE_YEAR_IN_MS - PASSWORD_CREATION_DATE_IN_MS > 0;

        result.age = isPasswordOlderThanFive ? 'too_old' : 'old';
    } else {
        const minutes = passwordCreatedDate.getMinutes();

        result.date = {
            year: passwordCreatedDate.getFullYear(),
            month: passwordCreatedDate.getMonth() + 1,
            day: passwordCreatedDate.getDate(),
            time: `${passwordCreatedDate.getHours()}:${minutes >= 10 ? minutes : `0${minutes}`}`
        };
    }
    return result;
}

function createPasswordInfo(passwordData, locTexts, helpLinks) {
    const passwordInfo = {
        strength: passwordData.strength,
        status: 'warning',
        updated: checkPasswordStatus(passwordData.last_update),
        buttonTxt: locTexts['profile.security.change-pwd'],
        alternateButtonTxt: locTexts['profile.security.change-pass']
    };

    if (passwordInfo.strength === 0) {
        passwordInfo.title = locTexts['profile.security.title_pwd-simple'];
        passwordInfo.text = locTexts['profile.security.field-pwd_simple'].replace('%s', helpLinks.passwordAbout);
    }

    if (passwordInfo.updated.age === 'old') {
        passwordInfo.title = locTexts['profile.security.title_pwd-old'];
        passwordInfo.text = locTexts['profile.security.field-pwd_old'].replace('%s', helpLinks.passwordReason);
    }

    if (passwordInfo.updated.age === 'too_old') {
        passwordInfo.title = locTexts['profile.security.title_pwd-toold'];
        passwordInfo.text = locTexts['profile.security.field-pwd_toold'].replace('%s', helpLinks.passwordReason);
    }

    if (passwordInfo.updated.age === 'normal' && passwordInfo.strength) {
        passwordInfo.status = 'normal';
        passwordInfo.title = locTexts['profile.security.title_pwd-date'];
        passwordInfo.text = locTexts['profile.security.field-pwd_ok'];
        passwordInfo.alternateButtonTxt = locTexts['profile.security.next-btn'];
    }

    return passwordInfo;
}

function createPhoneInfo(phoneData, locTexts, helpLinks) {
    const accountPhone = {
        title: locTexts['profile.securiy.field-no-phone'], // К аккаунту не привязан номер телефона
        // С телефоном ваш аккаунт будет лучше защищен.
        text: locTexts['profile.securiy.field-phone_hint'].replace('%s', helpLinks.phone),
        buttonTxt: locTexts['profile.security.bind-phone_now'], // Привязать номер
        alternateButtonTxt: locTexts['profile.security.bind-phone'], // Я привяжу телефон позже
        blockIcon: 'warning'
    };

    if (!_.isEmpty(phoneData)) {
        _.forIn(phoneData, function(value) {
            if (value.secured && value.bound) {
                accountPhone.number = value.number.masked_international;
                accountPhone.title = locTexts['profile.securiy.field-phone_clarify'].replace('%s', accountPhone.number);
                accountPhone.text = locTexts['profile.securiy.field-phone_bind'].replace('%s', helpLinks.phone);
                accountPhone.buttonTxt = locTexts['profile.security.replace-phone_btn'];
                accountPhone.alternateButtonTxt = locTexts['profile.security.confirm-phone'];
                accountPhone.blockIcon = 'normal';
                accountPhone.id = value.id;
                accountPhone.updatedText = {
                    title: `${locTexts['profile.securiy.field-phone_binded']} ${accountPhone.number}`,
                    buttonTxt: locTexts['profile.security.edit-phone'],
                    alternateButtonTxt: locTexts['profile.security.next-btn']
                };
            }
        });
    }

    return accountPhone;
}

function createEmailInfo(emailsList, locTexts, helpLinks) {
    const NO_RESTORE_EMAILS = 0;
    const emailInfo = {
        title: locTexts['profile.security.field-email_bind'], // К аккаунту не привязан запасной адрес электронной почты
        // С запасным адресом ваш аккаунт будет лучше защищён.
        text: locTexts['profile.security.bind-email_hint'].replace('%s', helpLinks.email),
        buttonTxt: locTexts['profile.security.bind-address_now'], // Привязать адрес
        alternateButtonTxt: locTexts['profile.security.bind-address'], // Я привяжу адрес позже
        blockIcon: 'warning'
    };

    if (emailsList && emailsList.suitable_for_restore && emailsList.suitable_for_restore.length > NO_RESTORE_EMAILS) {
        emailInfo.addresses = emailsList.suitable_for_restore;
        emailInfo.updatedText = {};

        if (emailInfo.addresses.length > 1) {
            emailInfo.title = locTexts['profile.security.field-binded-emails'];
            emailInfo.text = locTexts['profile.security.emails-ok'];
            emailInfo.buttonTxt = locTexts['profile.security.edit-plural-btn'];
            emailInfo.alternateButtonTxt = locTexts['profile.security.ok-btn'];
            emailInfo.updatedText.text = ' ';
        } else {
            emailInfo.title = locTexts['profile.security.field-email-clarify'].replace('%s', emailInfo.addresses[0]);
            emailInfo.text = locTexts['profile.security.field-email_hint'].replace('%s', helpLinks.email);
            emailInfo.buttonTxt = locTexts['profile.security.replace-phone_btn'];
            emailInfo.alternateButtonTxt = locTexts['profile.security.confirm-address'];
            // eslint-disable-next-line max-len
            emailInfo.updatedText.title = `${locTexts['profile.security.field-binded-email']} ${emailInfo.addresses[0]}`;
            emailInfo.updatedText.buttonTxt = locTexts['profile.security.edit-btn'];
        }

        emailInfo.updatedText.alternateButtonTxt = locTexts['profile.security.next-btn'];
        emailInfo.blockIcon = 'normal';
    }

    return emailInfo;
}
