var url = require('url');
var router = require('express').Router();
var _ = require('lodash');
var brokerPath = require('../configs/current').paths.broker;
var apiSetup = require('./common/apiSetup');
var locs = require('../loc/profilesocial.json');
var locsSocial = require('../loc/social');
var PLog = require('plog');
var accountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
var {getSocialProviders} = require('./common/socialSetup');

var routes = {};

// Old page
routes.main = [apiSetup, accountsSetup, getLanguage, getSocialProviders, getUserData, errorHandling, renderPage];

routes.socialProfiles = [
    apiSetup,
    accountsSetup,
    getLanguage,
    getSocialProviders,
    getUserData,
    function(req, res) {
        res.json(res.locals.user_social);
    }
];

router.get('/\\d*', routes.main);

function getLanguage(req, res, next) {
    var controller = req._controller;

    controller
        .getLanguage()
        .then(function(lang) {
            res.locals.language = lang;
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('social profiles')
                .write(err);
        })
        .done(function() {
            return next();
        });
}

function getUserData(req, res, next) {
    var lang = res.locals.language;

    req.api
        .socialProfileInit('/1/change_social/init/')
        .then(function(results) {
            var body = results.body;

            res.locals.track_id = res.locals.social_track_id = body.track_id;
            res.locals.providersList = res.locals.socialProviders.providers.slice().filter((i) => i.enabled);

            if (body.profiles && body.profiles.length) {
                res.locals.user_social = processProfiles(body.profiles, lang);
                res.locals.retpath = `${req.headers['x-real-scheme']}://${req.hostname}/profile/social/`;
            }
            return next();
        })
        .catch(function(error) {
            new PLog(req.logID, 'passport', 'social profiles', 'initSocial').warn(`Error:${error}`);
            next(error);
        });
}

function processProfiles(profiles, lang) {
    var socialProfiles = [];
    var profileSocialText = locs[lang].ProfileSocial;

    profiles.forEach(function(item) {
        var personInfo = item.person;
        var authInfo = item.allow_auth;
        var fullname;

        if (personInfo && personInfo.firstname && personInfo.lastname) {
            fullname = [personInfo.firstname, personInfo.lastname].join(' ').trim();
        }

        var profile = {
            username: '',
            profileLink: item.addresses[0] || '',
            allowAuth: '',
            profileId: item.profile_id,
            provider: item.provider,
            subscriptions: []
        };

        if (fullname) {
            profile.username = fullname;
        } else if (item.username) {
            profile.username = item.username;
        } else {
            profile.username = item.userid;
        }

        if (typeof authInfo !== 'undefined') {
            profile.allowAuth = authInfo;
        } else {
            profile.allowAuth = 'no_auth';
        }

        if (item.subscriptions && item.subscriptions.length) {
            item.subscriptions.forEach(function(subs) {
                var sidNum = subs.sid;
                var serviceList = {
                    25: {
                        service: 'market',
                        serviceName: profileSocialText['profile.social.service-market']
                    },

                    74: {
                        service: 'search',
                        serviceName: profileSocialText['profile.social.service-search']
                    },

                    2: {
                        service: 'mail',
                        serviceName: profileSocialText['profile.social.service-mail']
                    },

                    83: {
                        service: 'news',
                        serviceName: profileSocialText['profile.social.service-news']
                    }
                };
                var s = {
                    checked: subs.checked,
                    sid: sidNum,
                    service: '',
                    serviceName: '',
                    profileId: item.profile_id
                };

                if (serviceList[sidNum]) {
                    s.service = serviceList[sidNum].service;
                    s.serviceName = serviceList[sidNum].serviceName;
                    profile.subscriptions.push(s);
                }
            });
        }

        socialProfiles.push(profile);
    });

    return socialProfiles;
}

function errorHandling(error, req, res, next) {
    var lang = res.locals.language;
    var loc = locs[lang];
    var socialLoc = locsSocial[lang];
    var errorCode = error[0];
    var message;
    var basicUrlObj = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: '/profile/social/'
    };

    var retpath = url.format(basicUrlObj);

    var errorsWithMessage = {
        'sessionid.overflow': socialLoc['error']['sessionid-overflow'],
        'account.disabled': socialLoc['error']['account-disabled'],
        'social_profiles.invalid': loc['ProfileSocial']['profile.social.profile-invalid'],
        'exception.unhandled': socialLoc['error']['internal-exception']
    };

    if (!Array.isArray(error)) {
        return next();
    }

    if (errorCode === 'session.invalid' || errorCode === 'sessionid.invalid') {
        return res.redirect(
            url.format(
                _.extend(basicUrlObj, {
                    pathname: '/auth/',
                    query: {
                        retpath: retpath
                    }
                })
            )
        );
    }

    if (errorCode === 'track_id.empty' || errorCode === 'track.invalid_state') {
        return res.redirect(retpath);
    }

    if (errorCode === 'sslsession.required') {
        return res.redirect(
            url.format(
                _.extend(basicUrlObj, {
                    pathname: '/auth/secure/',
                    query: {
                        retpath: retpath
                    }
                })
            )
        );
    }

    message = errorsWithMessage[errorCode];
    if (message) {
        res.locals.errors = message;
    } else {
        res.locals.errors = errorsWithMessage['exception.unhandled'];
    }

    return next();
}

function renderPage(req, res) {
    if (req.cookies['profile_social.showSettings']) {
        res.locals.showSettings = req.cookies['profile_social.showSettings'];
    }
    res.locals.brokerPath = brokerPath;
    res.render(`profile.social.${res.locals.language}.js`);
}

function removePhonishes(profiles) {
    const newProfiles = [];

    for (let i = 0; i < profiles.length; i++) {
        if (
            profiles[i].provider.code !== 'kp' &&
            profiles[i].provider.code !== 'ya' &&
            !profiles[i].hasOwnProperty('phonish')
        ) {
            newProfiles.push(profiles[i]);
        }
    }

    return newProfiles;
}

exports.processProfiles = processProfiles;
exports.removePhonishes = removePhonishes;
exports.router = router;
exports.routes = routes;
