var PLog = require('plog');
const CloudApi = require('../lib/api/cloud');
const PraktikumApi = require('../lib/api/praktikum');
const tvm = require('../lib/tvm');
const BillingInternalApi = require('../lib/api/billingInternal');
const BillingApi = require('../lib/api/billing');

const CHECK_PLUS_OFFERS_TARGET = 'passport';
const CARD_TYPES = {plus: 'plus', music: 'music', mail: 'mail', disk: 'disk', practicum: 'practicum'};
const CARD_STATUS = {loading: 'loading', active: 'active', offer: 'offer', error: 'error'};

// const getFakePromise = (req) => {
//     let result = [];

//     if (req.cookies.fake_subs) {
//         result = req.cookies.fake_subs
//             .split(',')
//             .map((pair) => pair.split('|'))
//             .map(([type, status = 'active']) => ({type, status}));
//     }

//     return result;
// };

const getPassportInfoPromise = (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const tld = controller.getTld();
    const {language} = res.locals;
    const serviceTicket = req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.MEDIABILLING];

    return new Promise((resolve, reject) => {
        if (!uid) {
            reject({
                status: CARD_STATUS.error,
                errors: 'no_auth'
            });
        }

        new BillingInternalApi(req.logID, serviceTicket)
            .getPassportInfo(uid, tld, language)
            .then((resp) => resolve(resp.result))
            .catch(() => resolve({status: CARD_STATUS.error}));
    });
};

const hasPlusOffersPromise = (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const {language} = res.locals;

    return new BillingApi(req.logID)
        .getNativeProducts({target: CHECK_PLUS_OFFERS_TARGET, uid, ip: req.ip, language})
        .then(
            ({result: {nativeProducts = []} = {}} = {}) => nativeProducts.length > 0,
            () => false
        );
};

const isPlus = (subscription) => subscription.features.includes('basic-plus');

const preparePassportInfoCard = (subscriptions = [], hasPlusOffers) => {
    const hasError = subscriptions.some((item) => item.status === 'error');
    const hasMusicSub = subscriptions.some((sub) => sub.bundle === 'basic-music');
    const plusSubs = subscriptions.filter(isPlus);

    const result = [];

    if (plusSubs.length === 0 && hasError) {
        result.push({
            type: CARD_TYPES.plus,
            status: CARD_STATUS.error
        });
    }

    if (plusSubs.length === 0 && hasPlusOffers) {
        result.push({
            type: CARD_TYPES.plus,
            status: CARD_STATUS.offer
        });
    }

    if (plusSubs.length === 1) {
        result.push({
            type: CARD_TYPES.plus,
            status: CARD_STATUS.active,
            title: plusSubs[0].title,
            statusGender: ['station-lease-plus-kinopoisk', 'station-lease-plus'].includes(plusSubs[0].bundle)
                ? 'f'
                : 'm'
        });
    }

    if (plusSubs.length > 1) {
        result.push({
            type: CARD_TYPES.plus,
            status: CARD_STATUS.active
        });
    }

    if (hasMusicSub) {
        result.push({
            type: CARD_TYPES.music,
            status: CARD_STATUS.active
        });
    }

    return result;
};

const yandex360Promise = (req) => {
    return req._controller
        .getAuth()
        .sessionID()
        .then((session) => {
            const uid = session && session.uid && session.uid.value;
            const serviceTicket = req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.CLOUD];

            return new CloudApi(req.logID, uid, serviceTicket)
                .getUsersPassportSubscriptionsStatuses()
                .then((response) => [
                    {
                        type: CARD_TYPES.mail,
                        status: response[360] ? CARD_STATUS.active : CARD_STATUS.offer
                    }
                ])
                .catch(() => []);
        });
};

const practicumPromise = (req, res) => {
    if (![225, 149, 159].includes(res.locals.countryId)) {
        return [];
    }

    return req._controller
        .getAuth()
        .sessionID()
        .then((session) => {
            const uid = session && session.uid && session.uid.value;
            const serviceTicket = req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.PRAKTIKUM];
            const {language} = res.locals;

            return new PraktikumApi(req.logID, serviceTicket)
                .getSubscriptions({uid, language})
                .then(({intervals = []} = {}) => [
                    {
                        type: CARD_TYPES.practicum,
                        status: intervals.filter(({status}) => status === 'active').length
                            ? CARD_STATUS.active
                            : CARD_STATUS.offer
                    }
                ])
                .catch(() => []);
        });
};

exports.getSubscriptionCards = (req, res) => {
    Promise.all([
        getPassportInfoPromise(req, res),
        hasPlusOffersPromise(req, res), // start in parallel to reduce total wait time
        yandex360Promise(req),
        practicumPromise(req, res)
        // getFakePromise(req, res)
    ])
        .then(([plusResp, hasPlusOffers, disk, fakes]) =>
            res.json([...preparePassportInfoCard(plusResp.subscriptions, hasPlusOffers), ...disk, ...fakes])
        )
        .catch((err) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.subscriptions.getSubscriptionCards')
                .write(err);
            res.json({status: 'error'});
        });
};
