const PLog = require('plog');
const {getBGNumber} = require('./utils');
const langSetup = require('./common/langSetup');
const apiSetup = require('./common/apiSetup');
const getMetrics = require('./common/getMetrics');
const createState = require('./common/createState');
const createAMState = require('./authv2/createAMState');
const createAuthState = require('./authv2/createAuthState');
const createCommonState = require('./authv2/createCommonState');
const getUatraitsData = require('./common/getUatraitsData');
const userIsValid = require('./common/checkAuth/userIsValid');
const rumCounterSetup = require('./common/rumCounterSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const setTheme = require('./authv2/setTheme');
const express = require('express');
const router = express.Router();
const lookup = require('../lib/getLookup.js');
const config = require('../configs/current');
const getCustomConfig = require('./common/getCustomConfig');
const authCustomsVersion = getCustomConfig('customs.version.json').version;
const authCustomsStatic = config.paths.authCustomsStatic;

const enter = [
    apiSetup,
    getYaExperimentsFlags,
    getUatraitsData,
    langSetup,
    createState,
    createAMState,
    createAuthState(),
    userIsValid,
    function(req, res, next) {
        const {store = {}} = res.locals;
        const {
            am: {isAm, finishErrorUrl}
        } = store;

        if (isAm && !res.locals.userIsValid) {
            return req._controller.redirect(finishErrorUrl);
        }

        if (!isAm && !res.locals.userIsValid) {
            return req._controller.redirectToAuth();
        }

        return next();
    },
    createCommonState(),
    getMetrics({
        header: 'Попап с подтверждением QR входа'
    }),
    rumCounterSetup,
    function getAccount(req, res, next) {
        const params = {
            need_display_name_variants: false,
            need_phones: false,
            need_emails: false,
            need_social_profiles: false,
            need_question: false,
            need_additional_account_data: false,
            need_family_info: false,
            need_family_members: false,
            need_family_kids: false,
            need_family_invites: false
        };

        return req.api
            .profileGetState(params)
            .then(function(response) {
                const profile = response.body;

                res.locals.account = profile.account || {};
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('service_qr_secure_auth, getAccount')
                    .write(errors);

                return next(errors);
            });
    },
    function(req, res, next) {
        const {store = {}, account = {}} = res.locals;

        store.access = {
            is2faEnabled: account.is_2fa_enabled
        };

        return next();
    },
    function(req, res, next) {
        const {store = {}} = res.locals;

        if (!store.am.isAm) {
            if (!store.customs) {
                store.customs = {};
            }
            store.customs = {
                bgNumber: getBGNumber(18)
            };

            store.settings.authCustomsVersion = authCustomsVersion;
            store.settings.authCustomsStatic = authCustomsStatic;
        }

        return next();
    },
    function getDeviceInfoQrAuth(req, res, next) {
        const {store = {}} = res.locals;
        const {
            am: {trackId},
            settings: {language}
        } = store;

        req.api
            .getDeviceInfoQrAuth(trackId)
            .then((result = {}) => {
                const {body = {}} = result;

                if (body.status === 'ok') {
                    const regionId = body.region_id;
                    const region = typeof regionId === 'number' ? lookup.getRegionById(regionId) : null;
                    const isRu = ['ru', 'uk'].includes(language);
                    const regionName = region ? (isRu ? region.name : region.en_name) : '';
                    const osFamily = body.os_family && Array.isArray(body.os_family) ? body.os_family.join(' ') : null;

                    store.qrSecureAuth = {
                        browser: body.browser,
                        osFamily,
                        regionName,
                        deviceName: body.device_name,
                        isAuthPrepareWithCredError: false,
                        isLoading: false,
                        error: null,
                        isShowSuccessScreen: false,
                        isTrackError: false
                    };
                }

                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('service_qr_secure_auth, getDeviceInfoQrAuth')
                    .write(error);
                if (Array.isArray(error)) {
                    const errorCode = error[0];

                    if (['track_id.empty', 'track.not_found', 'track_id.invalid'].includes(errorCode)) {
                        store.qrSecureAuth = {
                            isAuthPrepareWithCredError: false,
                            error: errorCode,
                            isShowSuccessScreen: false,
                            isTrackError: true
                        };
                    }
                }
                return next();
            });
    },
    setTheme,
    function renderPage(req, res) {
        const lang = res.locals.language;

        res.render(`react.push-qr-secure-auth.${lang}.jsx`);
    }
];

router.get('/', enter);
exports.router = router;
