const url = require('url');
const PLog = require('plog');
const config = require('../configs/current');
const apiSetup = require('./common/apiSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const RegController = require('../lib/controller/RegistrationController');
const validateCSRF = require('./common/validateCSRF.js');
const getGenderSuggest = require('./common/getGenderSuggest');
const registerNewAccount = require('./common/registerAccount');
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');

const REGISTRATION_PDD_MOBILE_GOAL_PREFIX = 'registration_pdd_mobile';
const REGISTRATION_PDD_GOAL_PREFIX = 'registration_pdd';

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then((lang) => {
            res.locals.language = lang;
        })
        .catch((err) => {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);
        })
        .done(() => {
            next();
        });
}

function renderPage(req, res) {
    const lang = res.locals.language;

    req.api.statboxLogger({
        track_id: res.locals.track_id || null,
        action: 'opened',
        mode: 'registration.connect',
        host: req.hostname,
        pathname: req.path,
        referer: req.headers.referer || null,
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: res._yandexuid && res._yandexuid.replace(/[^0-9]/g, ''),
        origin: (req.query && req.query.origin) || null,
        from: (req.query && req.query.from) || null,
        experimentsFlags: res.locals.experiments && res.locals.experiments.flagsString,
        experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
    });

    res.render(`react.connect.${lang}.jsx`);
}

const createStore = [
    (req, res, next) => {
        const langSwitcher = new LangSwitcherView(req._controller);
        const locals = res.locals;
        const tld = req._controller.getTld();
        const yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
        const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
        const retpath = (req.query && (req.query.url || req.query.retpath)) || null;
        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}} = locals;
        const {metricsUserType} = userType;
        const authUrl = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/auth',
            query: req.query
        });

        if (retpath) {
            handlers.push(req.api.validateRetpath({retpath}));
        }

        locals.store = {
            fetchingStatus: {
                isFetching: false
            },
            settings: {
                location: req.path,
                host: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname
                }),
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.com || {},
                help: config.paths.help || {},
                tld,
                language: locals.language,
                uatraits: {},
                isMobile: false,
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                authUrl,
                origin: req.query.origin,
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType,
                staticPath
            },
            common: {
                actionForRepeat: null,
                yandexuid,
                track_id: locals.track_id,
                retpath: null,
                currentPage: req._controller.getUrl().href,
                dev: config.dev,
                defaultPage: url.format({
                    pathname: '/registration/connect',
                    query: req.query
                }),
                experiments: res.locals.experiments || {
                    flags: [],
                    flagsString: '',
                    boxes: '',
                    encodedBoxes: ''
                }
            },
            captcha: {
                loading: false,
                loadingAudio: false,
                playing: false,
                type: 'text',
                key: null,
                imageUrl: null,
                introSound: null,
                captchaSound: null
            },
            logins: {
                loginsList: [],
                showAll: false,
                isFetching: false
            },
            form: {
                activeField: '',
                validation: {
                    method: 'phone',
                    humanConfirmationDone: false
                },
                isEulaShowedInPopup: true,
                isEulaCheckedByDefault: false,
                canSwitchConfirmationMethod: false,
                states: {
                    firstname: '',
                    domain: '',
                    lastname: '',
                    login: '',
                    password: '',
                    password_confirm: '',
                    hint_question_id: '',
                    hint_question: '',
                    hint_question_custom: 'valid',
                    hint_answer: '',
                    captcha: '',
                    phone: '',
                    phoneCode: '',
                    phoneCodeStatus: ''
                },
                values: {
                    firstname: '',
                    domain: '',
                    lastname: '',
                    login: '',
                    password: '',
                    password_confirm: '',
                    hint_question_id: '',
                    hint_question: '',
                    hint_question_custom: '',
                    hint_answer: '',
                    captcha: '',
                    phone: '',
                    phoneCode: ''
                },
                errors: {
                    active: '',
                    domain: {
                        code: '',
                        text: ''
                    },
                    firstname: {
                        code: '',
                        text: ''
                    },
                    lastname: {
                        code: '',
                        text: ''
                    },
                    login: {
                        code: '',
                        text: ''
                    },
                    password: {
                        code: '',
                        text: ''
                    },
                    password_confirm: {
                        code: '',
                        text: ''
                    },
                    hint_question_id: {
                        code: '',
                        text: ''
                    },
                    hint_question: {
                        code: '',
                        text: ''
                    },
                    hint_question_custom: {
                        code: '',
                        text: ''
                    },
                    hint_answer: {
                        code: '',
                        text: ''
                    },
                    captcha: {
                        code: '',
                        text: ''
                    },
                    phone: {
                        code: '',
                        text: ''
                    },
                    phoneCode: {
                        code: '',
                        text: ''
                    },
                    phoneCodeStatus: ''
                },
                humanConfirmation: {
                    questionList: [],
                    isFetching: false
                }
            },
            registrationErrors: {
                status: 'ok',
                code: '',
                text: ''
            },
            footer: {
                langlist: []
            },
            metrics: {
                experiments: res.locals.experiments ? res.locals.experiments.encodedBoxes : ''
            },
            monitoring: {
                page: 'registration.connect'
            },
            pageMessage: null
        };

        Promise.all(handlers)
            .then((response) => {
                const defaultRetpath = url.format(
                    Object.assign({}, req._controller.getUrl(), {
                        pathname: 'profile',
                        search: null,
                        query: null
                    })
                );
                const uatraits = response[0] || {};
                const langlist = (response[1] && response[1].langlist) || {};
                const validatedRetpath = response[2] || defaultRetpath;

                locals.store.settings.uatraits = uatraits;
                locals.store.footer.langlist = langlist;
                locals.store.settings.isMobile = (uatraits.isMobile || uatraits.isTouch) && !uatraits.isTablet;
                locals.store.form.prefix = locals.store.settings.isMobile
                    ? REGISTRATION_PDD_MOBILE_GOAL_PREFIX
                    : REGISTRATION_PDD_GOAL_PREFIX;

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                }

                if (req._controller.getRequestParam('from')) {
                    locals.store.common.from = req._controller.getRequestParam('from');
                }

                return next();
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.connect')
                    .write(err);

                return next();
            });
    },
    getMetrics({header: 'Регистрация для коннекта'}),
    rumCounterSetup
];

const setup = [
    getLanguage,
    apiSetup,
    (req, res, next) =>
        req.api
            .getTrack({type: 'register'})
            .then((response) => {
                res.locals.track_id = response.body !== undefined ? response.body.id : '';
                return next();
            })
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.connect')
                    .write(errors);

                return next();
            }),
    writeTrack,
    getYaExperimentsFlags
];

exports.register = [
    validateCSRF,
    getLanguage,
    apiSetup,
    function(req, res, next) {
        res.locals.regType = 'connect';
        return next();
    },
    getGenderSuggest,
    registerNewAccount,
    createSession
];

const enter = [setup, createStore, renderPage];

exports.route = function(app) {
    const oldFilter = (req, res, next) => {
        if (req.query && req.query.old === '1') {
            return next('route');
        }

        return next();
    };

    app.get('/registration/connect', oldFilter, enter);
    app.post('/registration/connect', oldFilter, enter);
};

function createSession(req, res) {
    const reqistratonController = new RegController(req, res, req.logID);

    return reqistratonController
        .createSession()
        .then(() => {
            res.json({status: 'ok'});
        })
        .catch((error) => {
            res.json({status: 'error', error: [{field: 'global', code: 'sessionidoverflow'}]});
            PLog.warn()
                .logId(req.logID)
                .type('registration.connect')
                .write('createSession ', error);
        });
}

function writeTrack(req, res, next) {
    const data = {
        track_id: res.locals.track_id,
        retpath: req._controller.getRequestParam('retpath'),
        origin: req._controller.getRequestParam('origin')
    };

    if (data.retpath || data.origin) {
        req._controller.writeTrack(data);
    }

    return next();
}
