const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const getCountry = require('./common/getCountry');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const generateProcessUUID = require('./authv2/generateProcessUUID');
const rumCounterSetup = require('./common/rumCounterSetup');
const writeStatbox = require('./registration/writeStatbox');
const finish = require('./auth.js').routes.finish;
const redirectToLite = require('./registration/redirectToLite');
const setRegistrationPhoneOnly = require('./registration/setRegistrationPhoneOnly');
const redirectToRegistrationWithoutModes = require('./registration/redirectToRegistrationWithoutModes');
const redirectToRegistrationInDaHouse = require('./registration/redirectToRegistrationInDaHouse');
const getSocial = require('./auth/common/getStore/getSocial');
const getGenderSuggest = require('./common/getGenderSuggest');
const registerNewAccount = require('./common/registerAccount');
const createSession = require('./common/createSession');
const createCustomsState = require('./common/createCustomsState');
const registerNewAccountWithoutAJAX = require('./common/registerAccountWithoutAJAX');
const createStore = require('./registration/createRegistrationState');
const setRegistrationMode = require('./registration/setRegistrationMode');
const onAccountCreated = require('./registration/onAccountCreated');
const writeTrack = require('./common/writeTrack');
const checkMobileExperiments = require('./registration/checkMobileExperiments');
const setRegExperiments = require('./registration/setRegExperiments');
const setConfirmationRetryCountdown = require('./registration/setConfirmationRetryCountdown');
const regAvatar = require('./avatarRegistration');
const regQr = require('./qrRegistration');
const regBrowser = require('./registration.browser');
const createSignUpState = require('./common/createSignUpState');
const createFormState = require('./common/createFormState');
const getUatraitsData = require('./common/getUatraitsData');
const getMetrics = require('./common/getMetrics');
const config = require('../configs/current');
const getPlusPromocode = require('./registration/getPlusPromocode');
const checkAuth = require('./common/checkAuth');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const {urlFormat} = require('./common/urlFormat.js');

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then(function(lang) {
            res.locals.language = lang;
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);
        })
        .done(function() {
            next();
        });
}

function renderPage(req, res) {
    const lang = res.locals.language;

    res.locals.ysaId = config.ysaIdMap.registration;

    writeStatbox(req, res);
    res.render(`react.registration.${lang}.jsx`);
}

const setup = [
    getLanguage,
    apiSetup,
    getCountry,
    getYaExperimentsFlags,
    getUatraitsData,
    redirectToRegistrationInDaHouse,
    redirectToRegistrationWithoutModes,
    redirectToLite,
    function(req, res, next) {
        if (req.body.track_id) {
            res.locals.track_id = req.body.track_id;
            res.locals.registrationName = 'default';
            return next();
        }

        return req.api
            .getTrack({type: 'register', process: 'web_registration', scenario: 'register'})
            .then(function(response) {
                res.locals.track_id = response.body !== undefined ? response.body.id : '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.v2')
                    .write(errors);

                return next();
            });
    },
    writeTrack,
    (req, res, next) => {
        const isAjax = req.headers['x-requested-with'] === 'XMLHttpRequest';

        if (isAjax || req.method !== 'POST' || req.body['reload-captcha']) {
            return next();
        }

        const captchaData = {
            track_id: res.locals.track_id,
            key: req.body.captcha_key,
            answer: req.body.captcha
        };

        return req.api
            .captchaCheck(captchaData)
            .then((response) => {
                if (response.body.status === 'ok' && response.body.correct) {
                    return registerNewAccountWithoutAJAX(req, res, next);
                }

                res.locals.captchaErrors = [{field: 'captcha', code: 'captcha.not_matched'}];
                return next();
            })
            .catch((errors) => {
                res.locals.captchaErrors = errors.map((error) => {
                    return {field: 'captcha', code: `${error.field}.${error.code}`};
                });

                return next();
            });
    },
    (req, res, next) => {
        return req.api
            .getQuestions()
            .then((response) => {
                res.locals.questions = response.body.questions;

                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.v2')
                    .write(errors);

                return next();
            });
    },
    (req, res, next) => {
        return req.api
            .captchaGenerate({
                wave: true,
                track_id: res.locals.track_id
            })
            .then((response) => {
                res.locals.captcha = response.body;

                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.v2')
                    .write(errors);

                return next();
            });
    },
    generateProcessUUID
];

exports.route = function(app) {
    app.get('/registration', this.enter);
    // app.post('/registration', this.enter);

    app.get('/registration/child', this.childishEnter);
    // app.post('/registration/child', this.childishEnter);

    app.get('/registration/mail', this.enter);
    // app.post('/registration/mail', this.enter);

    app.get('/registration/unified(/:experiment)?', this.enter);
    // app.post('/registration/unified(/:experiment)?', this.enter);

    app.get('/registration/market/partner', this.enter);
    // app.post('/registration/market/partner', this.enter);

    app.get('/registration/music', this.enter);
    // app.post('/registration/music', this.enter);

    app.get('/registration/phone', this.enter);
    // app.post('/registration/phone', this.enter);

    app.get('/registration/toloka', this.enter);
    // app.post('/registration/toloka', this.enter);

    app.get('/registration-new', this.redirectFromNew);
    // app.post('/registration-new', this.enter);

    app.get('/registration/finish', finish);
    app.post('/registration/finish', finish);

    app.get('/registration/avatar', regAvatar);
    app.get('/registration/qr', regQr);
    app.get('/registration/browser', regBrowser);
};

exports.register = [
    getLanguage,
    apiSetup,
    function(req, res, next) {
        res.locals.regType = 'default';
        return next();
    },
    getGenderSuggest,
    registerNewAccount,
    onAccountCreated,
    createSession
];

exports.registerNeoPhonish = [
    getLanguage,
    apiSetup,
    function(req, res, next) {
        res.locals.regType = 'neophonish';
        return next();
    },
    getGenderSuggest,
    registerNewAccount,
    onAccountCreated,
    createSession
];

exports.enter = [
    setup,
    handleRegistrationErrors,
    createStore,
    createFormState,
    createSignUpState,
    getSocial(),
    rumCounterSetup,
    setRegistrationPhoneOnly,
    setRegistrationMode,
    setConfirmationRetryCountdown,
    checkMobileExperiments,
    setRegExperiments,
    getPlusPromocode,
    createCustomsState,
    getMetrics({
        header: 'Регистрация v2'
    }),
    renderPage
];

exports.childishEnter = [
    setup,
    checkAuth,
    function checkExp(req, res, next) {
        if (req._controller.hasExp('profile-family-childish-exp')) {
            return next();
        }

        return next('route');
    },
    multiAuthAccountsSetup,
    function redirectToUpgrade(req, res, next) {
        const {locals: {userType: {isPortal} = {}, accounts: {defaultAccount: {hasPhone} = {}} = {}} = {}} = res;

        const urlQuery = {
            retpath: urlFormat(req._controller.getUrl()),
            origin: 'need_access_to_childish_registration'
        };

        if (!isPortal) {
            return req._controller.redirectToLocalUrl({
                pathname: 'auth/complete',
                query: urlQuery
            });
        }

        if (!hasPhone) {
            return req._controller.redirectToLocalUrl({
                pathname: 'phoneconfirm',
                query: urlQuery
            });
        }

        return next();
    },
    handleRegistrationErrors,
    createStore,
    createFormState,
    getSocial(),
    rumCounterSetup,
    setRegistrationMode,
    setConfirmationRetryCountdown,
    checkMobileExperiments,
    setRegExperiments,
    getPlusPromocode,
    createCustomsState,
    getMetrics({
        header: 'Регистрация чайлдиша'
    }),
    renderPage
];

function handleRegistrationErrors(req, res, next) {
    if (!res.locals.formErrors || !Array.isArray(res.locals.formErrors) || !res.locals.formErrors.length) {
        return next();
    }

    const needReloadPage = res.locals.formErrors.filter((error) => error.code === 'registrationalreadycompletederror')
        .length;

    if (needReloadPage) {
        return req._controller.redirectToLocalUrl({
            pathname: '/registration',
            query: req.query
        });
    }

    return next();
}

exports.redirectFromNew = [
    function(req) {
        const urlObj = req._controller.getUrl();

        return req._controller.redirectToLocalUrl(
            Object.assign({}, urlObj, {
                pathname: 'registration'
            })
        );
    }
];
