var apiSetup = require('./common/apiSetup');
var createCSRFValidator = require('./common/createCSRFValidator');
var config = require('../configs/current');
var locs = require('../loc/auth.json');

var PForm = require('pform');
var putil = require('putils');
var url = require('url');
var when = require('when');
var _ = require('lodash');

var helper = {
    errorIds: {
        'account.disabled': 'ErrorsTexts.badlog_blocked',
        'account.not_found': 'ErrorsTexts.deleted',
        'account.disabled_on_deletion': 'ErrorsTexts.deleted',
        'compare.not_matched': 'import.account_not_found',
        'backend.historydb_api_failed': 'AutoLogin.dberror',
        'backend.oauth_failed': 'AutoLogin.dberror',
        password_change_forbidden: 'pdd_password_change_forbidden',
        'account.without_password': 'without_password',
        complete_social: 'social_without_password',
        complete_pdd: 'request_pdd_admin',
        internal: 'ErrorsTexts.internal'
    },

    mendIds: {
        'account.disabled': 'disabled',
        'account.not_found': 'deleted',
        'account.disabled_on_deletion': 'deleted',
        internal: 'crap',
        'track.not_found': 'crap',
        'compare.not_matched': 'account_not_found'
    },

    internalErrIds: ['exception.unhandled', 'track_id.invalid_state', 'track_id.invalid', 'track_id.not_found'],

    stateMessages: {
        pin_checks_left: 'state_pin_checks_left'
    },

    skipErrIds: [
        'password.empty',
        'password.likelogin',
        'password.tooshort',
        'password.toolong',
        'password.weak',
        'password.prohibitedsymbols',
        'password.likeoldpassword',
        'password.foundinhistory',
        'password.found_in_history',
        'password.likephonenumber',
        'password_auth_date.empty'
    ]
};

function getForm() {
    return new PForm(
        new (require('../blocks/control/password/password.field'))()
            .setRequired()
            .setLabel('%field_password_createnew'),

        new (require('../blocks/control/password-confirm/password-confirm.field'))()
            .setRequired()
            .setLabel('%field_password_confrimnew'),

        new (require('../blocks/control/login/login.field'))(),
        new (require('../blocks/control/pin/pin.field'))()
            .setRequired()
            .setOption('maxlength', 16)
            .setOption('minlength', 4),

        new (require('../blocks/control/phone-confirm/phone-confirm.field'))()
            .setRequired()
            .setLabel('%field_phone_enter_number')
            .setHint('%phone-confirm_code_label')
            .setOptions({
                codeLabel: '%phone-confirm_code_title',
                mode: 'confirmForRestore',
                link2restore: '/restoration/2fa/restore'
            })
    );
}

function getStep(data) {
    if (!data) {
        return false;
    }

    if (data.restore_finished) {
        return 'finish';
    }

    if (data.pin_checked) {
        return 'password';
    }

    if (data.phone_confirmed) {
        return 'pin';
    }

    return 'phone';
}

function getTitle(locals) {
    var title;

    if (!locals.login || locals.step === 'finish') {
        return false;
    }

    if (locals.step === 'phone' || locals.step === 'pin') {
        title = locals.loc['Frontend']['title.restorefor'] || '';
    }

    if (locals.step === 'password') {
        title = locals.loc['Frontend']['title.changepasswordfor'] || '';
    }

    if (title) {
        title = title.replace(
            '%1',
            `<span class="head-2fa-username head-2fa-username__firstletter">${locals.login.charAt(
                0
            )}</span>${locals.login.substr(1)}`
        );
    }

    return title;
}

function errorProcess(errors, req, res, next) {
    var locErr = locs[res.locals.language]['Errors'];
    var locMend = locs[res.locals.language]['Mend'];

    res.locals.errors = [];
    res.locals.fieldsErrors = [];

    function getErrorMessage(code) {
        var errId = code;

        if (helper.stateMessages[code]) {
            res.locals.state = (locErr[helper.stateMessages[code]] || '').replace('%1', res.locals.track_id || '');

            return;
        }

        if (helper.skipErrIds.indexOf(code) >= 0) {
            res.locals.fieldsErrors.push(code);
            return;
        }

        if (helper.internalErrIds.indexOf(code) >= 0) {
            errId = 'internal';
        }

        if (!helper.errorIds[errId]) {
            errId = 'internal';
        }

        var error = {
            code
        };

        var errorResult = [];

        [].concat(helper.errorIds[errId]).forEach(function(id) {
            errorResult.push(locErr[id]);
        });

        error.msg = errorResult.join(' ');

        if (helper.mendIds[errId]) {
            var mendResult = [];

            [].concat(helper.mendIds[errId]).forEach(function(id) {
                mendResult.push(locMend[id]);
            });

            error.mend = mendResult.join(' ');
        }

        res.locals.errors.push(error);
    }

    [].concat(errors).forEach(function(code) {
        getErrorMessage(code);
    });

    next();
}

var setup = [
    function(req, res, next) {
        if (!req.body['track_id'] && !req.query['track_id']) {
            // без трека отправляем на рестор
            return res.redirect('/passport?mode=restore');
        }

        next();
    },
    function(req, res, next) {
        new (require('../lib/controller'))(req, res, req.logID)
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('restore 2fa')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    apiSetup,
    function(req, res, next) {
        _.assign(res.locals, {
            loc: locs[res.locals.language],
            track_id: (req.nquery && req.nquery['track_id']) || req.body['track_id'],
            _: putil.csrf(null, req.cookies.yandexuid)
        });

        next();
    }
];

exports.route = function(app) {
    app.post('/restoration/2fa', this.restore2fa.submit)
        .all('/restoration/2fa', this.restore2fa.enter)
        .get('/restoration/2fa/check', this.restore2fa.checkSession)
        .get('/restoration/2fa/restore', this.restore2fa.link2restore);
};

exports.restore2fa = {};
exports.restore2fa.enter = [
    setup,
    function getState(req, res, next) {
        req.api.restore2faGetState().then(
            function(data) {
                _.assign(res.locals, {
                    login: data.body.account && data.body.account.login,
                    step: getStep(data.body.account),
                    exitLink: (req.nquery && req.nquery['retpath']) || req.query['retpath']
                });

                res.locals.title = getTitle(res.locals);

                if (data.body.account && data.body.account.pin_checks_left === 0) {
                    return next('pin_checks_left');
                }

                next();
            },
            function(errors) {
                var exitErrors = ['track_id.invalid', 'track.not_found'];

                errors.forEach(function(error) {
                    if (exitErrors.indexOf(error) >= 0) {
                        return res.redirect('/passport?mode=restore');
                    }
                });

                next(errors);
            }
        );
    },
    function(req, res, next) {
        req.api.statboxLogger({
            ignoreMissedTrack: true,
            host: req.hostname,
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            yandexuid: req.cookies.yandexuid,
            track_id: res.locals.track_id,
            process_uuid: req.query.process_uuid || null,
            action: 'opened',
            mode: `restore_2fa_${res.locals.step}`,
            experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
            experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
        });

        next();
    },
    function(req, res, next) {
        var form = getForm().setApi(req.api);

        if (res.locals.fieldsErrors && res.locals.fieldsErrors.length) {
            res.locals.fieldsErrors.forEach(function(code) {
                var c = code.split('.');
                var field = form.getField(c[0]);

                if (field) {
                    var err = field.getErrorByCode(c[1]);

                    if (!err) {
                        err = field.getErrorByCode(c[1].replace(/_/g, ''));
                    }

                    if (err) {
                        err.setActive();
                    }
                }
            });
        }

        if (res.locals.step === 'finish') {
            res.locals.auth_track_id = req.nquery && req.nquery['auth_track_id'];
            req.api.delTrack();
        }

        form.compile(res.locals.language).then(function(data) {
            res.locals.form = {
                control: data
            };

            next();
        });
    },
    errorProcess,
    function(req, res) {
        res.render(`restore.2fa.${res.locals.language}.js`);
    }
];

exports.restore2fa.submit = [
    createCSRFValidator(
        function(req, res, next) {
            return next();
        },
        function(req, res) {
            return res.redirect('/restoration');
        }
    ),
    setup,
    function(req, res, next) {
        const controller = req._controller;

        req.api.restore2faCommit(req.body).then(
            function(data) {
                if (data.body.cookies) {
                    // эта ветка - удачная смена пароля
                    req.api.statboxLogger({
                        ignoreMissedTrack: true,
                        host: req.hostname,
                        ip: req.headers['x-real-ip'],
                        user_agent: req.headers['user-agent'],
                        yandexuid: req.cookies.yandexuid,
                        track_id: res.locals.track_id,

                        action: 'submitted',
                        mode: 'restore_2fa_successful_password_change',
                        experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                        experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
                    });

                    controller.augmentResponse(data.body);

                    return res.redirect(
                        url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: 'restoration/2fa/check',
                            query: {
                                auth_track_id: data.body.auth_track_id,
                                track_id: req.body['track_id']
                            }
                        })
                    );
                }

                next(data);
            },
            function(errors) {
                next(errors);
            }
        );
    },
    errorProcess
];

exports.restore2fa.checkSession = [
    apiSetup,
    function(req, res, next) {
        var session = req.cookies['Session_id'];
        var isRu = /.*(.yandex\.ru.*)/.test(req.hostname);
        var retpath;
        var mdaQuery;
        var finishURL;

        if (!session) {
            next(['ErrorsTexts.nocki']);
        }

        req.api.sessionCheck({session, track_id: req.body['track_id']}).then(
            function(data) {
                if (data.body && data.body.session_is_correct) {
                    retpath = url.format({
                        protocol: req.headers['x-real-scheme'],
                        hostname: req.hostname,
                        pathname: 'restoration/2fa',
                        query: {
                            track_id: req.nquery && req.nquery['track_id'],
                            auth_track_id: req.nquery && req.nquery['auth_track_id'],
                            retpath:
                                data.body.retpath ||
                                url.format({
                                    protocol: req.headers['x-real-scheme'],
                                    hostname: req.hostname,
                                    pathname: 'passport',
                                    query: {
                                        mode: 'passport'
                                    }
                                })
                        }
                    });

                    mdaQuery = {
                        retpath,
                        ncrnd: String(Math.round(Math.random() * 1000000))
                    };

                    if (isRu) {
                        finishURL = url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: config.paths.mda,
                            pathname: 'login',
                            query: mdaQuery
                        });
                    } else {
                        finishURL = retpath;
                    }

                    return res.redirect(finishURL);
                }

                next(data);
            },
            function(error) {
                next(error);
            }
        );
    }
];

exports.restore2fa.link2restore = [
    setup,
    function(req, res, next) {
        if (!req.nquery['track_id']) {
            return next('no_track');
        }

        when.resolve()
            .then(function() {
                return req.api.restore2faGetState();
            })
            .then(
                function(getStateData) {
                    if (getStateData.body.account && getStateData.body.account.login) {
                        req.api.statboxLogger({
                            ignoreMissedTrack: true,
                            host: req.hostname,
                            ip: req.headers['x-real-ip'],
                            user_agent: req.headers['user-agent'],
                            yandexuid: req.cookies.yandexuid,
                            track_id: req.nquery['track_id'],
                            process_uuid: req.query.process_uuid || null,
                            action: 'opened',
                            mode: `restore_2fa_to_restore_semi_auto_from_${getStep(getStateData.body.account)}`,
                            experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                            experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
                        });

                        return req.api.restoreGetTrack({
                            login: getStateData.body.account.login,
                            request_source: 'restore2fa'
                        });
                    }

                    return next('no_login');
                },
                function(errors) {
                    return next(errors);
                }
            )
            .then(
                function(getTrackData) {
                    if (getTrackData.body.track_id) {
                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: '/restoration/semi_auto',
                                query: {
                                    track_id: getTrackData.body.track_id
                                }
                            })
                        );
                    }

                    return next('no_semi_auto_track');
                },
                function(errors) {
                    return next(errors);
                }
            );
    }
];
