var apiSetup = require('./common/apiSetup');
var getMDAOptions = require('./common/getMDAOptions.js');
var PForm = require('pform');
var PLog = require('plog');
var PUtils = require('putils');
var _ = require('lodash');
var url = require('url');
var when = require('when');
var locs = require('../loc/auth.json');
var config = require('../configs/current');
var getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var FieldPhoneConfirm = require('../blocks/control/phone-confirm/phone-confirm.field');
var FieldSubmit = require('../blocks/control/submit/submit.field');
var FieldCaptcha = require('../blocks/control/captcha/captcha.field');
var FieldPin = require('../blocks/control/pin/pin.field');
var FieldPassword = require('../blocks/control/password/password.field');
var FieldPasswordConfirm = require('../blocks/control/password-confirm/password-confirm.field');
var FieldQuestion = require('../blocks/control/question/question.field');
var FieldUserQuestion = require('../blocks/control/user-question/user-question.field');
var FieldAnswer = require('../blocks/control/answer/answer.field');
var FieldFirstname = require('../blocks/control/firstname/firstname.field');
var FieldLastname = require('../blocks/control/lastname/lastname.field');
var FieldGender = require('../blocks/control/gender/gender.field');
var FieldBirthday = require('../blocks/control/birthday/birthday.field');
var FieldLastPasswords = require('../blocks/control/restore/last-passwords/last-passwords.field');

var restorePagesPaths = {
    start: '/restoration',
    phone: '/restoration/phone',
    phone_and_2fa_factor: '/restoration/twofa',
    hint: '/restoration/hint',
    email: '/restoration/email',
    semi_auto: '/restoration/semi_auto',
    changepassword: '/restoration/changepassword',
    finish: '/restoration/finish',
    check: '/restoration/check',
    fallback: '/restoration/fallback',
    twofaFallback: '/restoration/twofa-form?track_id=%1'
};

exports.settings = {
    paths: restorePagesPaths
};

var stateMessages = {
    password_change_forbidden: 'pdd_password_change_forbidden',
    complete_social: 'social_without_password',
    complete_pdd: 'request_pdd_admin',
    complete_autoregistered: 'restore_complete_autoregistered',
    domain_not_served: 'restore_domain_not_served',
    rate_limit_exceeded: 'restore_fio_limit_exceeded',
    sessionoverflowerror: 'MendTexts.sessionidoverflow_restore',
    twofa_form_not_matched: 'twofa_form_not_matched',
    phone_alias_prohibited: 'restore.phone.alias'
};

var errorIds = {
    'account.disabled': 'ErrorsTexts.badlog_blocked',
    'account.not_found': 'ErrorsTexts.deleted',
    'account.disabled_on_deletion': 'ErrorsTexts.deleted',
    'compare.not_matched': 'import.account_not_found',
    'backend.historydb_api_failed': 'AutoLogin.dberror',
    'backend.oauth_failed': 'AutoLogin.dberror',
    password_change_forbidden: 'pdd_password_change_forbidden',
    'account.without_password': 'without_password',
    complete_social: 'social_without_password',
    complete_pdd: 'request_pdd_admin',
    internal: 'ErrorsTexts.internal',
    'account.global_logout': 'ErrorsTexts.restart_restore',
    'email.changed': 'restore.email.changed'
};

var mendIds = {
    'account.disabled': 'disabled',
    'account.not_found': 'deleted',
    'account.disabled_on_deletion': 'deleted',
    internal: 'crap',
    'track.not_found': 'crap',
    'compare.not_matched': 'account_not_found',
    'account.global_logout': 'restart_restore_process',
    'email.changed': 'mend.restore.email.changed'
};

var internalErrIds = ['exception.unhandled', 'track_id.invalid_state', 'track_id.invalid', 'track_id.not_found'];

var skipErrors = [
    'password.likephonenumber',
    'password.found_in_history',
    'password.equals_previous',
    'secret_key.invalid',
    'email.not_matched',
    'phone.compromised',
    'password.likelogin',
    'password.week',
    'user.not_verified',
    'answer.required',
    'answer.incorrect',
    'answer.captchalocate'
];

var errorsAliases = {
    equals_previous: 'likeoldpassword',
    found_in_history: 'foundinhistory',
    not_verified: 'missingvalue'
};

var fieldsMap = {
    phone: 'phone_number',
    user: 'answer'
};

var exitErrors = ['track.invalid_state', 'track_id.invalid', 'track.not_found', 'method.not_allowed'];

var init = [
    function(req, res, next) {
        res.locals.track_id = (req.body && req.body.track_id) || (req.nquery && req.nquery.track_id) || '';
        res.locals.process_uuid = req.query.process_uuid || '';

        var query = {};

        if (res.locals.track_id) {
            query.track_id = res.locals.track_id;
        }

        if (res.locals.process_uuid) {
            query.process_uuid = res.locals.process_uuid;
        }

        var fallbackUrl = {
            pathname: restorePagesPaths.fallback,
            search: null,
            query
        };
        var restorationChooseUrl = {
            pathname: '/restoration/choose',
            search: null,
            query
        };
        var restoreLoginUrl = {
            pathname: '/restoration/login',
            search: null,
            query
        };

        res.locals.restoreLoginUrl = url.format(_.extend({}, restoreLoginUrl));
        res.locals.restorationChooseUrl = url.format(_.extend({}, restorationChooseUrl));
        res.locals.fallbackUrl = url.format(_.extend({}, fallbackUrl));

        req._controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('restore.base')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    apiSetup
];

function getForm(locals) {
    var method = locals.method;
    var methodsToBind = locals.methodsToBind;
    var fields = [];
    const exps = (locals.experiments && locals.experiments.flags) || [];

    var submitField = new FieldSubmit()
        .setRequired()
        .setLabel('%next')
        .setOption('theme', 'action');

    var phoneField = new FieldPhoneConfirm()
        .setPlaceholder('%field_phone_enter_number')
        .setLabel('%field_phone_enter_number')
        .setHint('%phone-confirm_code_label')
        .setOptions({
            mode: 'restore',
            hideExample: true,
            hideFakeEntry: true,
            showCodePlaceholder: true,
            resendTimer: true,
            tryToCall: exps.includes('restore_pwd_calls_on')
        });

    if (locals.phoneConfirmed && locals.newPhone && !locals.phoneCompromised) {
        phoneField
            .setValue(locals.newPhone)
            .setOption('state', 'confirmed')
            .setOption('hasConfirmedPhone', true);
    }

    if (method === 'hint') {
        submitField.setLabel('%send');

        fields.push(
            new FieldCaptcha().setOptions({
                asyncCheck: true
            })
        );
    }

    fields.push(submitField);

    if (['phone', 'twofa'].indexOf(method) !== -1) {
        fields.push(phoneField);

        if (method === 'twofa') {
            fields.push(
                new FieldPin()
                    .setOptions({maxlength: 16, placeholderLabel: true, fallbackLink: locals.twofaFallbackLink})
                    .setHint('')
            );
        }
    }

    if (method === 'changepassword') {
        phoneField.setOptions({mode: 'restore', hideExample: false, hideFakeEntry: false});

        fields = [
            new FieldPassword().setPlaceholder('%field_password'),
            new FieldPasswordConfirm().setPlaceholder('%field_password_confirm')
        ].concat(fields);
    }

    if (methodsToBind && methodsToBind.length) {
        methodsToBind.forEach(function(field) {
            if (field === 'phone') {
                fields.push(phoneField.setOption('mode', 'restoreBind'));
            }

            if (field === 'hint') {
                fields = [
                    new FieldQuestion(),
                    new FieldUserQuestion().setPlaceholder('%field_hint_question'),
                    new FieldAnswer().setPlaceholder('%field_hint_answer')
                ].concat(fields);
            }
        });
    }

    if (locals.isSocial) {
        fields = [
            new FieldFirstname().setPlaceholder('%field_firstname'),
            new FieldLastname().setPlaceholder('%field_lastname'),
            new FieldGender(),
            new FieldBirthday()
        ].concat(fields);
    }

    if (method === 'twofa-form') {
        fields = [
            new FieldFirstname().setOption('placeholderLabel'),
            new FieldLastname().setOption('placeholderLabel'),
            new FieldLastPasswords().setPlaceholder('%field_last_password_before_enabling_2fa')
        ].concat(fields);
    }

    return new PForm(fields);
}

function getUrl(req, mode, track_id) {
    // eslint-disable-line camelcase
    if (!mode || !restorePagesPaths[mode]) {
        return '';
    }

    var restoreUrl = {
        pathname: restorePagesPaths[mode],
        search: null,
        query: {}
    };

    if (mode === 'start') {
        // https://st.yandex-team.ru/MOBDEVAUTH-7635
        // нужен любой параметр, чтобы в АМ работала логика работы
        // с query-параметрами
        restoreUrl.query.restart = '1';
    }

    if (track_id) {
        // eslint-disable-line camelcase
        restoreUrl.query.track_id = track_id; // eslint-disable-line camelcase
    }

    if (req.query.process_uuid) {
        restoreUrl.query.process_uuid = req.query.process_uuid;
    }

    return url.format(_.extend({}, req._controller.getModePassportUrl(), restoreUrl));
}

function formCompile(req, res, next) {
    var form = getForm(res.locals).setApi(req.api);

    if (res.locals.formErrors && res.locals.formErrors.length) {
        res.locals.formErrors.forEach(function(error) {
            var _error = error.split('.');
            var field = form.getField(fieldsMap[_error[0]] || _error[0]);

            if (field) {
                var fieldError = field.getErrorByCode(_error[1]) || field.getErrorByCode(errorsAliases[_error[1]]);

                if (fieldError && typeof fieldError.setActive === 'function') {
                    fieldError.setActive();
                }
            }
        });
    }

    when.resolve()
        .then(function() {
            return form.validate(req.body);
        })
        .then(function() {
            return form.compile(res.locals.language);
        })
        .then(function(data) {
            res.locals.restoreMethod = req.body['restore-method'];
            res.locals.form = {
                control: data
            };

            return next();
        });
}

function checkState(req, res, next) {
    req.api.restoreGetState().then(
        function(response) {
            var body = (response && response.body) || {};

            if (body.track_id && !res.locals.track_id) {
                res.locals.track_id = body.track_id;
            }

            if (res.locals.method !== 'changepassword' && body.restore_method_passed && !res.locals.ignorMethodPassed) {
                return res.redirect(getUrl(req, 'changepassword', res.locals.track_id));
            }

            res.locals = _.extend(res.locals, {
                // all methods ['phone', 'phone_and_pin', 'hint', 'email', 'semi_auto']
                login: body.user_entered_login,
                methods: body.suitable_restore_methods || [],
                currentRestoreMethod: body.current_restore_method,
                restoreMethodPassed: body.restore_method_passed
            });

            if (body && body.method_state) {
                res.locals = _.extend(res.locals, {
                    hint: body.method_state.question,
                    requiredCaptcha: body.method_state.is_captcha_required,
                    isPhoneConfirmed: body.method_state.is_phone_confirmed,
                    emailEtered: body.method_state.email_entered,
                    emailIsCorrect: body.method_state.is_email_suitable_for_restore
                });
            }

            if (body && body.new_auth_state) {
                var newState = body.new_auth_state;

                res.locals = _.extend(res.locals, {
                    // сделать опционаленое добавление средств восстановления
                    newPhone: (newState.new_phone && newState.new_phone.masked_international) || '',
                    methodsToBind: (newState.is_method_binding_required && newState.allowed_methods_to_bind) || [],
                    phoneConfirmed: newState.is_new_phone_confirmed,
                    isSocial: newState.is_social_completion_required
                });

                if (newState.revokers && newState.revokers.allow_select) {
                    res.locals.allowSelectRevokers = true;
                }
            }

            if (res.locals.methods.indexOf('semi_auto') !== -1) {
                res.locals.semiAutoAvailable = true;
                res.locals.methods = res.locals.methods.filter(function(method) {
                    return method !== 'semi_auto';
                });
            }

            if (body.state) {
                return next([body.state]);
            }

            return next();
        },
        function(error) {
            var exit = false;

            if (Array.isArray(error)) {
                exit = error.some(function(err) {
                    return exitErrors.indexOf(err) !== -1;
                });
            }

            if (exit || exitErrors.indexOf(error) !== -1) {
                return res.redirect(getUrl(req, 'start'));
            }

            return next(error);
        }
    );
}

function errorProcess(errors, req, res, next) {
    var locErr = locs[res.locals.language].Errors;
    var locMend = locs[res.locals.language].Mend;

    res.locals.errors = [];
    res.locals.formErrors = [];

    function getErrorMessage(code) {
        var errId = code;

        if (internalErrIds.indexOf(code) !== -1) {
            errId = 'internal';
        }

        if (!errorIds[errId]) {
            errId = 'internal';
        }

        var error = {
            code
        };

        var errorResult = [];

        [].concat(errorIds[errId]).forEach(function(id) {
            errorResult.push(locErr[id]);
        });

        error.msg = errorResult.join(' ');

        if (mendIds[errId]) {
            var mendResult = [];

            [].concat(mendIds[errId]).forEach(function(id) {
                mendResult.push(locMend[id]);
            });

            error.mend = mendResult.join(' ');
        }

        return error;
    }

    [].concat(errors).forEach(function(code) {
        // eslint-disable-line consistent-return
        // {
        //     field: null,
        //     message: 'No more users allowed',
        //     code: 'sessionoverflowerror'
        // }

        code = code.code || code;

        if (exitErrors.indexOf(code) !== -1) {
            return res.redirect(getUrl(req, 'start'));
        }

        if (errors.indexOf('phone.compromised') !== -1) {
            res.locals.phoneCompromised = true;
        }

        if (skipErrors.indexOf(code) !== -1) {
            res.locals.formErrors.push(code);
        } else if (code in stateMessages) {
            res.locals.state = locErr[stateMessages[code]] || locMend[stateMessages[code]] || '';

            if (code === 'sessionoverflowerror') {
                res.locals.state = res.locals.state.replace('%1', url.format(req._controller.getAuthUrl()) || '');
            } else {
                res.locals.state = res.locals.state.replace('%1', res.locals.track_id || '');
            }
        } else {
            res.locals.errors.push(getErrorMessage(code));
        }
    });

    return next();
}

function checkCaptcha(req, res, next) {
    if (req.body.key && req.body.answer) {
        var data = {
            key: req.body.key,
            answer: req.body.answer
        };

        delete req.body.key;
        delete req.body.answer;

        return req.api.captchaCheck(data).then(
            function(result) {
                if (result.body.correct) {
                    return next();
                }

                return next(['answer.incorrect']);
            },
            function() {
                return next(['answer.captchalocate']);
            }
        );
    }

    return next();
}

function render(req, res) {
    if (res.locals.delTrack) {
        req.api.delTrack();
    }

    res.render(`restore.base.${res.locals.language}.js`);
}

exports.route = function(app) {
    app.post('/restoration/choose', this.choose.commit)
        .all('/restoration/choose', this.choose.entry)

        .post('/restoration/hint', this.hint.commit)
        .all('/restoration/hint', this.hint.entry)

        .get('/restoration/phone', this.phone.entry)

        .get('/restoration/2fa', this.twofa.entry)
        .get('/restoration/twofa', this.twofa.entry)
        .get('/restoration/twofa-form', this.twofa.formSubmit)
        .post('/restoration/twofa-form', this.twofa.formCommit)

        .post('/restoration/email', this.email.commit)
        .all('/restoration/email', this.email.entry)

        .post('/restoration/changepassword', this.changepassword.commit)
        .all('/restoration/changepassword', this.changepassword.entry)

        .get('/restoration/check', this.check.entry)

        .get('/restoration/fallback', this.fallback.entry)

        .get('/restoration/finish', this.finish.entry);
};

function chooseMethod(req, res, next) {
    var method = res.locals.newMethod;

    if (!method) {
        return next();
    }

    return req.api
        .restoreSetRestoreMethod({
            method
        })
        .then(
            function(response) {
                if (response && response.body.status === 'ok' && restorePagesPaths[method]) {
                    return res.redirect(getUrl(req, method, res.locals.track_id));
                }

                return next();
            },
            function(error) {
                return next(error);
            }
        );
}

exports.choose = {
    entry: [
        init,
        checkState,
        function(req, res, next) {
            if (res.locals.methods.length > 1) {
                return next();
            }

            res.locals.newMethod = res.locals.methods[0] || (res.locals.semiAutoAvailable && 'semi_auto');
            return next();
        },
        chooseMethod,
        errorProcess,
        render
    ],
    commit: [
        init,
        function(req, res, next) {
            if (!req.body || !req.body.method) {
                return next('route');
            }

            res.locals.newMethod = req.body.method;
            return next();
        },
        chooseMethod,
        errorProcess,
        render
    ]
};

exports.hint = {
    entry: [
        function(req, res, next) {
            res.locals.method = 'hint';
            return next();
        },
        init,
        checkState,
        errorProcess,
        formCompile,
        render
    ],
    commit: [
        function(req, res, next) {
            res.locals.method = 'hint';
            return next();
        },
        init,
        checkCaptcha,
        function(req, res, next) {
            if (!req.body || !req.body.hint_answer) {
                return next('route');
            }

            return req.api
                .restoreCheckHintAnswer({
                    answer: req.body.hint_answer
                })
                .then(
                    function(response) {
                        var body = response && response.body;

                        if (body.status === 'ok') {
                            return res.redirect(getUrl(req, 'changepassword', res.locals.track_id));
                        }

                        return next();
                    },
                    function(error) {
                        if (error.indexOf('answer.not_matched') !== -1) {
                            res.locals.error = 'not_matched';
                            res.locals.value = req.body.hint_answer;

                            return next();
                        }

                        return next(error);
                    }
                );
        },
        errorProcess,
        checkState,
        errorProcess,
        formCompile,
        render
    ]
};

exports.phone = {
    entry: [
        function(req, res, next) {
            res.locals.method = 'phone';
            return next();
        },
        init,
        checkState,
        errorProcess,
        getYaExperimentsFlags,
        formCompile,
        render
    ]
};

exports.twofa = {
    entry: [
        init,
        function(req, res, next) {
            res.locals.method = 'twofa';
            res.locals.twofaFallbackLink = restorePagesPaths.twofaFallback.replace('%1', res.locals.track_id);
            return next();
        },
        checkState,
        function checkValidMethodHasChosen(req, res, next) {
            const validMethods = ['phone_and_2fa_factor', 'phone'];

            if (validMethods.includes(res.locals.currentRestoreMethod)) {
                return next();
            }

            return next(['method.not_allowed']); // restoration process will be restarted
        },
        errorProcess,
        getYaExperimentsFlags,
        formCompile,
        render
    ],
    formSubmit: [
        function(req, res, next) {
            res.locals.method = 'twofa-form';
            return next();
        },
        init,
        checkState,
        errorProcess,
        getYaExperimentsFlags,
        formCompile,
        function(req, res, next) {
            req.api.statboxLogger({
                action: 'opened',
                mode: 'restore_2fa_short_form',
                track_id: res.locals.track_id || null,
                process_uuid: req.query.process_uuid || null,
                ip: req.headers['x-real-ip'],
                host: req.hostname,
                user_agent: req.headers['user-agent'],
                yandexuid: req.cookies.yandexuid,
                experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
            });

            return next();
        },
        render
    ],
    formCommit: [
        function(req, res, next) {
            res.locals.method = 'twofa-form';
            return next();
        },
        init,
        function(req, res, next) {
            var statBoxLog = {
                action: 'commit',
                track_id: res.locals.track_id || null,
                ip: req.headers['x-real-ip'],
                host: req.hostname,
                user_agent: req.headers['user-agent'],
                yandexuid: req.cookies.yandexuid,
                experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
            };

            req.api
                .restoreCheckTwoFaShortForm({
                    firstname: req.body.firstname,
                    lastname: req.body.lastname,
                    password: req.body['last-passwords'],
                    answer: req.body.answer
                })
                .then(
                    function(response) {
                        var body = response.body || {};

                        if (body && body.status === 'ok') {
                            req.api.statboxLogger(
                                _.extend(statBoxLog, {
                                    mode: 'restore_2fa_short_form_success'
                                })
                            );

                            return res.redirect(getUrl(req, 'changepassword', res.locals.track_id));
                        }

                        return next();
                    },
                    function(error) {
                        if (
                            (Array.isArray(error) && error.indexOf('compare.not_matched') !== -1) ||
                            error === 'compare.not_matched'
                        ) {
                            error = ['twofa_form_not_matched'];

                            req.api.statboxLogger(
                                _.extend(statBoxLog, {
                                    mode: 'restore_2fa_short_form_fail'
                                })
                            );

                            res.locals.delTrack = true;
                        }

                        return next(error);
                    }
                );
        },
        errorProcess,
        checkState,
        errorProcess,
        getYaExperimentsFlags,
        formCompile,
        render
    ]
};

exports.fallback = {
    entry: [
        init,
        checkState,
        function(req, res, next) {
            res.locals.newMethod = 'semi_auto';
            return next();
        },
        function(req, _res, next) {
            const {track_id, app_id} = req.query;

            if (track_id && app_id) {
                req.api
                    .writeTrack({
                        track_id,
                        app_id
                    })
                    .catch((err) => {
                        PLog.warn()
                            .logId(req.logID)
                            .type('restore.base.fallback')
                            .write(err);
                    });
            }

            return next();
        },
        chooseMethod,
        errorProcess,
        render
    ]
};

exports.email = {
    entry: [
        function(req, res, next) {
            res.locals.method = 'email';
            return next();
        },
        init,
        checkState,
        errorProcess,
        formCompile,
        render
    ],
    commit: [
        function(req, res, next) {
            res.locals.method = 'email';
            return next();
        },
        init,
        function(req, res, next) {
            var email = req.body.email;
            var key = req.body.key;

            if (email) {
                return req.api
                    .restoreCheckEmail({
                        email,
                        simple: req.body.simple || 'no'
                    })
                    .then(
                        function() {
                            // do nothing
                            return next();
                        },
                        function(error) {
                            if (
                                error.indexOf('email.not_matched') !== -1 ||
                                error.indexOf('email.invalid') !== -1 ||
                                error.indexOf('email.check_limit_exceeded') !== 1
                            ) {
                                res.locals.error = error[0];
                                res.locals.value = email;

                                return next();
                            }

                            return next(error);
                        }
                    );
            } else if (key) {
                return req.api
                    .restoreCheckEmailKey({
                        key,
                        track_id: res.locals.track_id
                    })
                    .then(
                        function() {
                            return res.redirect(getUrl(req, 'changepassword', res.locals.track_id));
                        },
                        function(error) {
                            if (error.indexOf('secret_key.invalid') !== -1) {
                                res.locals.error = 'not_matched';
                                res.locals.value = key;

                                return next();
                            }

                            return next(error);
                        }
                    );
            }

            return next();
        },
        errorProcess,
        checkState,
        errorProcess,
        formCompile,
        render
    ],
    check: [
        init,
        function(req, res, next) {
            var key = req.nquery && req.nquery.key;

            if (!key) {
                return next('route');
            }

            return req.api
                .restoreCheckEmailKey({
                    key
                })
                .then(
                    function() {
                        return next();
                    },
                    function() {
                        return next('route');
                    }
                );
        },
        checkState,
        function(req, res) {
            if (!res.locals.restoreMethodPassed && res.locals.currentRestoreMethod) {
                return res.redirect(getUrl(req, res.locals.currentRestoreMethod, res.locals.track_id));
            }

            return res.redirect(getUrl(req, 'changepassword', res.locals.track_id));
        }
    ]
};

exports.changepassword = {
    entry: [
        function(req, res, next) {
            res.locals.method = 'changepassword';
            res.locals.ignorMethodPassed = true;
            return next();
        },
        init,
        getYaExperimentsFlags,
        checkState,
        function(req, res, next) {
            req.api.statboxLogger({
                action: 'opened',
                mode: 'restoration_change_password',
                track_id: res.locals.track_id || null,
                ip: req.headers['x-real-ip'],
                host: req.hostname,
                user_agent: req.headers['user-agent'],
                yandexuid: req.cookies.yandexuid,
                experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
            });
            return next();
        },
        errorProcess,
        formCompile,
        render
    ],
    commit: [
        function(req, res, next) {
            res.locals.method = 'changepassword';
            return next();
        },
        init,
        getYaExperimentsFlags,
        function(req, res, next) {
            var body = req.body || {};
            var statBoxLog = {
                action: 'commit',
                mode: 'restoration_change_password',
                track_id: res.locals.track_id || null,
                ip: req.headers['x-real-ip'],
                host: req.hostname,
                user_agent: req.headers['user-agent'],
                yandexuid: req.cookies.yandexuid,
                experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
                experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
            };

            if (body.is_able_to_select_revoke) {
                statBoxLog.is_revoke_checkbox_checked = body.revoke_tokens ? 'yes' : 'no';
            }

            req.api.statboxLogger(statBoxLog);
            return next();
        },
        function(req, res, next) {
            const controller = req._controller;
            var body = req.body || {};

            if (!body.password) {
                return next('route');
            }

            var data = {
                password: body.password,
                question_id: body.hint_question_id,
                question: body.hint_question,
                answer: body.hint_answer,
                new_method: body['restore-method'],
                firstname: body.firstname,
                lastname: body.lastname,
                gender: body.sex,
                birthday: PUtils.formatDate(body.byear, body.bmonth, body.bday)
            };

            if (req.body['restore-method'] !== 'hint') {
                delete data.answer;
                delete data.question;
                delete data.question_id;
            }

            if (data.question_id === '99') {
                delete data.question_id;
            }

            if (body.is_able_to_select_revoke && !body.revoke_tokens) {
                data.revoke_app_passwords = 'no';
                data.revoke_tokens = 'no';
                data.revoke_web_sessions = 'no';
            }

            return when
                .resolve()
                .then(function() {
                    return req.api.restoreChangePassword(data);
                })
                .catch(function(error) {
                    return when.reject(error);
                })
                .then(function(response) {
                    var body = response && response.body;

                    if (!body || body.status === 'error') {
                        return when.reject(body.errors || ['internal']);
                    }

                    return req.api.sessionCreate();
                })
                .then(
                    function(response) {
                        var body = (response && response.body) || {};

                        controller.augmentResponse(body);

                        if (body.cookies && body.cookies.length) {
                            return res.redirect(getUrl(req, 'check', res.locals.track_id));
                        }

                        return next();
                    },
                    function(error) {
                        return next(error);
                    }
                );
        },
        errorProcess,
        checkState,
        errorProcess,
        formCompile,
        render
    ]
};

exports.check = {
    entry: [
        init,
        function(req, res, next) {
            req.api
                .sessionCheck({
                    session: req.cookies.Session_id || '',
                    track_id: res.locals.track_id
                })
                .then(
                    function(response) {
                        var body = response && response.body;

                        if (body.session_is_correct) {
                            var protocol = req.headers['x-real-scheme'];
                            var mdaCookie = req.cookies.mda;
                            var needMDA = false;
                            var mdaPathname = '/';
                            var retpath = getUrl(req, 'finish', res.locals.track_id);
                            var clean = body.clean;
                            var mdaQuery;

                            if (mdaCookie === '1') {
                                needMDA = true;
                            }

                            mdaQuery = {
                                retpath,
                                ncrnd: String(Math.round(Math.random() * 1000000))
                            };

                            if (clean) {
                                mdaQuery.clean = clean;
                            }

                            if (needMDA) {
                                req.exitURL = url.format({
                                    protocol,
                                    hostname: config.paths.mda,
                                    pathname: mdaPathname,
                                    query: mdaQuery
                                });
                            } else {
                                req.exitURL = retpath;
                            }

                            return res.redirect(req.exitURL);
                        }

                        return next(['no_cookie']);
                    },
                    function(error) {
                        return next(error);
                    }
                );
        }
    ]
};

exports.finish = {
    entry: [
        function(req, res, next) {
            res.locals.method = 'finish';
            return next();
        },
        init,
        function(req, res, next) {
            req.api.restoreGetState().then(
                function(response) {
                    var body = response.body || {};
                    var notRestoredByPhone =
                        body.current_restore_method === 'email' || body.current_restore_method === 'hint';
                    var caseToSuggestPhone =
                        !body.is_restore_passed_by_support_link && !body.has_secure_phone_number && notRestoredByPhone;

                    if (body.restore_finished && caseToSuggestPhone) {
                        res.locals.suggestBindPhone = 1;
                        res.locals.restoreMethod = body.current_restore_method;
                    }

                    if (body.next_track_id) {
                        res.locals.auth_track_id = body.next_track_id;
                    }

                    if (body.state) {
                        res.locals.promo = body.state;
                    }

                    if (body.retpath) {
                        res.locals.exitLink = body.retpath;
                    } else {
                        res.locals.exitLink = url.format(req._controller.getModePassportUrl());
                    }

                    const mdaOptions = getMDAOptions(res.locals.exitLink);

                    if (mdaOptions.needMDA) {
                        res.locals.exitLink = mdaOptions.redirectUrl;
                    }

                    return next();
                },
                function(error) {
                    if (Array.isArray(error) && error.indexOf('track.not_found') !== -1) {
                        return res.redirect(url.format(req._controller.getModePassportUrl()));
                    }

                    return next(error);
                }
            );
        },
        function(req, res, next) {
            req.api.delTrack();
            next();
        },
        function(req, res, next) {
            if (res.locals.suggestBindPhone) {
                req.api
                    .getTrack()
                    .then(function(result) {
                        res.locals.track_id = result.body.id;
                        var form = new PForm(
                            new FieldPhoneConfirm().setRequired().setOptions({
                                mode: 'confirmAndBindSecure'
                            })
                        ).setApi(req.api);

                        form.compile(res.locals.language)
                            .then(function(data) {
                                res.locals.form = {control: data};
                                next();
                            })
                            .catch(function() {
                                next();
                            });
                    })
                    .catch(function(error) {
                        next(error);
                    });
            } else {
                next();
            }
        },
        errorProcess,
        render
    ]
};
