'use strict';

var url = require('url');
var _ = require('lodash');
var when = require('when');
var express = require('express');
var PLog = require('plog');
var config = require('../configs/current');
var apiSetup = require('./common/apiSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
var LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const getMetrics = require('./common/getMetrics');
const getUatraitsData = require('./common/getUatraitsData');
const rumCounterSetup = require('./common/rumCounterSetup');
const isUnsupportedBro = require('./common/isUnsupportedBro');

var router = express.Router();
var redirectMap = {};
var setup;

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then(function(lang) {
            res.locals.language = lang;
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);
        })
        .done(function() {
            next();
        });
}

function renderPage(req, res) {
    var lang = res.locals.language;

    req.api.statboxLogger({
        track_id: res.locals.track_id || null,
        process_uuid: req.query.process_uuid || null,
        action: 'opened',
        mode: 'restore.login',
        host: req.hostname,
        pathname: req.path,
        referer: req.headers.referer || null,
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: req.cookies.yandexuid,
        uid: (res.locals.store && res.locals.store.person && res.locals.store.person.uid) || null,
        origin: (req.query && req.query.origin) || null,
        experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
        experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
    });

    res.render(`restore.login.${lang}.jsx`);
}

const createStore = [
    (req, res, next) => {
        // eslint-disable-line
        var langSwitcher = new LangSwitcherView(req._controller);
        var locals = res.locals;
        var tld = req._controller.getTld();
        var retpath = (req.query && (req.query.url || req.query.retpath)) || null;
        var handlers = [langSwitcher._compile(), req.api.restoreLoginsSuggest()];
        const processUUID = req.query.process_uuid || null;
        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}, account = {}} = locals;
        const {metricsUserType} = userType;
        const {blackbox: {connection_id: ci = ''} = {}} = req;

        if (retpath) {
            handlers.push(req.api.validateRetpath({retpath}));
        } else {
            retpath = url.format(
                Object.assign({}, req._controller.getUrl(), {
                    pathname: 'auth'
                })
            );
        }

        const processUUIDQuery = {};

        if (processUUID) {
            processUUIDQuery.process_uuid = processUUID;
        }

        locals.store = {
            settings: {
                avatar: config.paths.avatar || {},
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.ru || {},
                help: config.paths.help || {},
                tld,
                language: locals.language,
                ua: {},
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType,
                staticPath
            },
            common: {
                actionForRepeat: null,
                uid: account.uid,
                yandexuid: req.cookies.yandexuid,
                process_uuid: processUUID,
                track_id: locals.track_id,
                experiments: locals.experiments,
                retpath: null,
                restorationLoginUrl: url.format({
                    pathname: '/restoration/login',
                    query: processUUIDQuery
                }),
                authUrl: url.format({
                    pathname: '/auth',
                    query: processUUIDQuery
                }),
                addUserUrl: url.format({
                    pathname: '/auth',
                    query: Object.assign(
                        {},
                        {
                            mode: 'add-user',
                            login: 'login'
                        },
                        processUUIDQuery
                    )
                }),
                restorationUrl: url.format({
                    pathname: '/restoration',
                    query: Object.assign(
                        {},
                        {
                            login: 'login'
                        },
                        processUUIDQuery
                    )
                }),
                currentPage: req._controller.getUrl().href,
                defaultPage: url.format({
                    pathname: '/restoration/login',
                    query: req.query
                }),
                isPDD: account.domain !== undefined,
                isWSUser: locals.isWSUser
            },
            restoreLogin: {
                suggestedLogins: [],
                step: ['suggested_logins'],
                loading: false,
                selectedLogin: '',
                errors: {}
            },
            captcha: {
                loading: false,
                playing: false,
                type: 'text',
                key: null,
                imageUrl: null,
                introSound: null,
                captchaSound: null
            },
            person: _.extend({}, account.person || {}, {
                uid: account.uid,
                login: account.login,
                displayLogin: account.display_login,
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayNames: account.display_names || {},
                displayName: (account.display_name && account.display_name.name) || '',
                escapedDisplayName: (account.display_name && _.escape(account.display_name.name)) || '',
                errors: {}
            }),
            header: {
                accounts: (locals.accounts && locals.accounts.accounts) || [],
                defaultAccount: (locals.accounts && locals.accounts.defaultAccount) || {},
                canAddMore: locals.accounts && locals.accounts['can-add-more'],
                ci
            },
            footer: {
                langlist: []
            },
            monitoring: {
                page: 'restore.login'
            }
        };

        when.all(handlers)
            .then(function(response) {
                var uatraits = res.locals.ua || {};
                var langlist = (response[0] && response[0].langlist) || {};
                var suggestedLogins = response[1].body.suggested_logins || [];
                var validatedRetpath = response[2] || {};

                locals.store.settings.ua = uatraits;
                locals.store.footer.langlist = langlist;
                locals.store.restoreLogin.suggestedLogins = suggestedLogins.filter(
                    (item = {}) => item.login && !item.login.startsWith(config.neoPhonishPrefix)
                );

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                }

                if (!uatraits.isMobile && !uatraits.isTouch) {
                    Object.keys(locals.store.person.displayNames).forEach(function(key) {
                        var value = locals.store.person.displayNames[key];

                        delete locals.store.person.displayNames[key];

                        locals.store.person.displayNames[_.escape(key)] = value;
                    });
                }
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport')
                    .write(err);
            })
            .done(function() {
                delete locals.account;
                delete locals.accounts;

                return next();
            });
    },
    getMetrics({
        header: 'Страница восстановления логина'
    }),
    rumCounterSetup
];

function getState(req, res, next) {
    var isNotLoggedIn = !req._controller.getAuth().isLoggedIn();

    if (isNotLoggedIn) {
        res.locals.account = {};
        return next();
    }

    req.api
        .profileGetState()
        .then(function(response) {
            var profile = response.body;

            res.locals.account = profile.account || {};

            return next();
        })
        .catch(function(errors) {
            res.locals.account = {};
            PLog.warn()
                .logId(req.logID)
                .type('restore.login, getState')
                .write(errors);
            return next();
        });
}

setup = [
    getLanguage,
    apiSetup,
    function(req, res, next) {
        return req.api
            .getRestoreLoginTrack()
            .then(function(response) {
                res.locals.track_id = (response.body && response.body.track_id) || '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport')
                    .write(errors);

                return next();
            });
    },
    multiAuthAccountsSetup,
    getUatraitsData,
    getYaExperimentsFlags,
    function(req, res, next) {
        if (!isUnsupportedBro(res.locals.ua)) {
            return req._controller.redirectToLocalUrl({
                pathname: 'auth/restore/login'
            });
        }

        if (res.locals.isWSUser && req.path === redirectMap.changereg) {
            return res.redirect(url.format(_.extend({}, req._controller.getModePassportUrl(), {query: req.query})));
        }

        return next();
    }
];

function submitPhone(req, res) {
    var body = req.body;
    var captchaPromise;

    if ('captcha' in body) {
        captchaPromise = req.api.captchaCheck({
            track_id: body.track_id,
            answer: body.captcha
        });
    } else {
        captchaPromise = when.resolve({body: {}});
    }

    captchaPromise
        .then(function(response) {
            var captcha = response.body;
            var promise;

            if (Array.isArray(response)) {
                captcha.forEach(function(item) {
                    if (item.code === 'captchalocate') {
                        promise = when.reject(['captcha.cannot_locate']);
                    }
                });
            } else if (captcha.correct === false) {
                promise = when.reject(['captcha.not_matched']);
            } else {
                promise = req.api.getRestoreLoginCode({
                    track_id: body.track_id,
                    phone_number: body.phone_number,
                    display_language: body.display_language,
                    country: body.country
                });
            }

            return promise;
        })
        .then(function() {
            res.json({
                status: 'ok'
            });
        })
        .catch(function(errors) {
            res.json({
                status: 'error',
                errors
            });
        });
}

function submitCode(req, res) {
    var body = req.body;

    return req.api
        .checkRestoreLoginCode({
            track_id: body.track_id,
            code: body.code
        })
        .then(function() {
            res.json({
                status: 'ok'
            });
        })
        .catch(function(errors) {
            res.json({
                status: 'error',
                errors
            });
        });
}

function submitName(req, res) {
    var body = req.body;

    return req.api
        .restoreLoginByName({
            track_id: body.track_id,
            firstname: body.firstname,
            lastname: body.lastname
        })
        .then(function(response) {
            res.json(response.body);
        })
        .catch(function(errors) {
            res.json({
                status: 'error',
                errors
            });
        });
}

var enter = [setup, getState, createStore, renderPage];

router.get('/login', enter);
router.get('/select', enter);

exports.submitPhone = submitPhone;
exports.submitCode = submitCode;
exports.submitName = submitName;
exports.router = router;
