var apiSetup = require('./common/apiSetup');
var locs = require('../loc/auth.json');
var url = require('url');
var _ = require('lodash');
var when = require('when');
var PLog = require('plog');
var PForm = require('pform');
var trimSlashesRegexp = /^\/+|\/+$/g;
var emailCheck = require('./restore.base').email.check;
var FieldLoginSimple = require('../blocks/control/login-simple/login-simple.field');
var FieldCaptcha = require('../blocks/control/captcha/captcha.field');
var FieldSubmit = require('../blocks/control/submit/submit.field');
var getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var getUatraitsData = require('./common/getUatraitsData');
var isUnsupportedBro = require('./common/isUnsupportedBro');

var helper = {
    errorIds: {
        'account.disabled': 'ErrorsTexts.badlog_blocked',
        'account.not_found': 'ErrorsTexts.deleted',
        'account.disabled_on_deletion': 'ErrorsTexts.deleted',
        'login.invalid': 'import.login_not_found',
        'account.without_password': 'without_password',
        internal: 'ErrorsTexts.internal'
    },

    mendIds: {
        'account.disabled': 'disabled',
        'account.not_found': 'deleted',
        'account.disabled_on_deletion': 'deleted',
        internal: 'crap'
    },

    stateMessage: {
        password_change_forbidden: 'pdd_password_change_forbidden',
        complete_social: 'social_without_password',
        complete_pdd: 'request_pdd_admin',
        complete_autoregistered: 'restore_complete_autoregistered',
        domain_not_served: 'restore_domain_not_served',
        phone_alias_prohibited: 'restore.phone.alias'
    },

    internalErrIds: ['blackboxfailed', 'track_id.invalid_state', 'track_id.invalid', 'track_id.not_found'],

    skipErrIds: ['user.not_verified', 'login.empty', 'answer.incorrect'],

    fieldsNameMap: {
        login: 'login-simple',
        user: 'answer'
    },

    aliases: {
        'login-simple': {
            empty: 'missingvalue'
        },
        answer: {
            not_verified: 'missingvalue'
        }
    }
};

var authRestoreFilter = [
    getUatraitsData,
    function(req, res, next) {
        var isUnsupportedBrowser = isUnsupportedBro(res.locals.ua);
        var query = req.nquery || {};
        var queryToPass = {};

        if (query.retpath) {
            queryToPass.retpath = query.retpath;
        }

        if (!isUnsupportedBrowser) {
            if (query.key) {
                queryToPass.key = query.key;

                return req._controller.redirectToLocalUrl({
                    pathname: '/auth/restore',
                    query: queryToPass
                });
            }

            if (query.login) {
                queryToPass.login = query.login;
                queryToPass.app_id = query.app_id;

                return req._controller.redirectToLocalUrl({
                    pathname: '/auth/restore/password/captcha',
                    query: queryToPass
                });
            }

            return req._controller.redirectToLocalUrl({
                pathname: '/auth/restore/password',
                query: queryToPass
            });
        }

        return next('route');
    }
];

function getForm() {
    return new PForm(
        new FieldLoginSimple()
            .setPlaceholder('%restoration.login.placeholder')
            .setLabel('')
            .setRequired(),
        new FieldCaptcha()
            .setOption('asyncCheck', true)
            .setPlaceholder('%field_answer_without_colon')
            .setLabel(''),
        new FieldSubmit().setOption('theme', 'action').setLabel('%next')
    );
}

function errorProcess(errors, req, res, next) {
    var locErr = locs[res.locals.language]['Errors'];
    var locMend = locs[res.locals.language]['Mend'];

    res.locals.errors = [];
    res.locals.fieldsErrors = [];

    function getMessage(code) {
        var id = code;
        var isState = false;

        if (helper.skipErrIds.indexOf(code) >= 0) {
            res.locals.fieldsErrors.push(code);
            return;
        }

        if (helper.stateMessage[code]) {
            isState = true;
        }

        if (!isState) {
            if (helper.internalErrIds.indexOf(code) >= 0) {
                id = 'internal';
            }

            if (!helper.errorIds[id]) {
                id = 'internal';
            }
        }

        var message = {
            code: id
        };

        message.msg = []
            .concat(helper.errorIds[id])
            .map(function(errorId) {
                return locErr[errorId];
            })
            .join(' ');

        if (helper.mendIds[id] || isState) {
            message.mend = []
                .concat(helper[isState ? 'stateMessage' : 'mendIds'][id])
                .map(function(msgId) {
                    return isState ? locErr[msgId] : locMend[msgId];
                })
                .join(' ');
        }

        if (isState) {
            res.locals.state = message;
            return;
        }

        res.locals.errors.push(message);
    }

    [].concat(errors).forEach(function(code) {
        getMessage(code);
    });

    next();
}

function pddFilter(req, res, next) {
    res.locals.pddDomain = (req.nquery && req.nquery['pdd_domain']) || (req.params && req.params.domain) || null;
    next();
}

function modeFilter(req, res, next) {
    var mode = req.nquery && req.nquery['mode'];

    if (!mode || ['spprtrstr', 'restore'].indexOf(mode) === -1) {
        return next('route');
    }

    next();
}

exports.route = function(app) {
    var self = this;

    app.get('/restoration', authRestoreFilter, self.enter);

    ['/restoration', '/restoration/form', '/restoration/changehint'].forEach(function(route) {
        app.get(route, emailCheck)
            .post(route, pddFilter, route === '/restoration' ? self.restore : self.submit)
            .get(route, pddFilter, logMiddleware, self.enter)
            .post(route, pddFilter, logMiddleware, self.enter);
    });

    ['/', '/passport', '/for/:domain', '/for/:domain/passport'].forEach(function(route) {
        app.get(route, modeFilter, emailCheck)
            .post(route, modeFilter, pddFilter, self.restore)
            .get(route, modeFilter, pddFilter, logMiddleware, self.enter)
            .post(route, modeFilter, pddFilter, logMiddleware, self.enter);
    });
};

var setup = [
    function(req, res, next) {
        var path = req.path.replace(trimSlashesRegexp, '');

        if (path === 'restoration/changehint') {
            res.locals.isChangeHint = true;
        }

        if (path === 'passport') {
            res.locals.forceUnconditional = true;
        }

        next();
    },
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('prerestore')
                    .write(err);
            })
            .then(function() {
                return next();
            });
    },
    apiSetup,
    function getTrack(req, res, next) {
        if (res.locals.track_id) {
            return next();
        }

        if (req.body['track_id']) {
            res.locals.track_id = req.body['track_id'];
            return next();
        }

        req.api.getTrack({type: 'restore', process: 'restore'}).then(
            function(data) {
                if (data.body && data.body.id) {
                    res.locals.track_id = data.body.id;
                    req.api.track(data.body.id);
                }

                next();
            },
            function(errors) {
                next(errors);
            }
        );
    },
    function(req, res, next) {
        const {app_id} = req.query;
        const track_id = res.locals.track_id;

        if (track_id && app_id) {
            req.api
                .writeTrack({
                    track_id,
                    app_id
                })
                .catch((err) => {
                    PLog.warn()
                        .logId(req.logID)
                        .type('prerestore.setup.writeTrack')
                        .write(err);
                });
        }

        return next();
    }
];

function checkCaptcha(req, res, next) {
    if (req.body.key && req.body.answer) {
        var data = {
            key: req.body.key,
            answer: req.body.answer
        };

        delete req.body.key;
        delete req.body.answer;

        req.api.captchaCheck(data).then(
            function(result) {
                if (result.body.correct) {
                    next();
                } else {
                    return next(['answer.incorrect']);
                }
            },
            function() {
                return next(['answer.captchalocate']);
            }
        );
    } else {
        return next();
    }
}

exports.enter = [
    setup,
    function(req, res, next) {
        var form = getForm().setApi(req.api);

        if (res.locals.fieldsErrors && res.locals.fieldsErrors.length) {
            res.locals.fieldsErrors.forEach(function(code) {
                var c = code.split('.');
                var fieldName = helper.fieldsNameMap[c[0]] || c[0];
                var field = form.getField(fieldName);

                if (field) {
                    var err = field.getErrorByCode(
                        (helper.aliases[fieldName] && helper.aliases[fieldName][c[1]]) || c[1]
                    );

                    if (err) {
                        err.setActive();
                    }
                }
            });
        }

        if (req.nquery && req.nquery['login']) {
            form.getField('login-simple').setValue(req.nquery['login']);
        }

        when.resolve()
            .then(function() {
                return form.validate(req.body);
            })
            .then(function() {
                return form.compile(res.locals.language, req.api);
            })
            .then(function(data) {
                res.locals.form = {
                    control: data
                };

                return next();
            });
    },
    function(req, res) {
        res.render(`restore.prerestore.${res.locals.language}.js`);
    }
];

exports.submit = [
    setup,
    checkCaptcha,
    function(req, res, next) {
        var body = {
            body: req.body
        };

        if (res.locals.isChangeHint) {
            body.requestSource = 'changehint';
        } else {
            body.unconditional = req.query['unconditional'];
        }

        if (res.locals.forceUnconditional) {
            body.unconditional = true;
        }

        req.api.restoreGetTrackWithCaptcha(body).then(
            function(data) {
                var body = data.body || {};

                if (body.user_entered_login) {
                    res.locals.login = body.user_entered_login;
                }

                if (body.state) {
                    return next([body.state]);
                }

                return res.redirect(
                    url.format(
                        _.extend(req._controller.getUrl(), {
                            pathname: '/restoration/semi_auto',
                            query: {
                                track_id: req.body.track_id
                            },
                            search: null
                        })
                    )
                );
            },
            function(errors) {
                next(errors);
            }
        );
    },
    errorProcess
];

exports.restore = [
    setup,
    checkCaptcha,
    function(req, res, next) {
        var login = req.body['login-simple'];
        var data = {};

        if (res.locals.pddDomain && !/@/.test(login)) {
            login += `@${res.locals.pddDomain}`;
        }

        data.login = login;

        if (req.query && req.query.retpath) {
            data.retpath = req.query.retpath;
        }

        if (req.query && req.query && req.query.gps_package_name) {
            data.gps_package_name = req.query.gps_package_name;
        }

        req.api.restoreCheckLogin(data).then(
            function(response) {
                var body = response.body || {};

                if (body.user_entered_login) {
                    res.locals.login = body.user_entered_login;
                }

                if (body.state) {
                    return next([response.body.state]);
                }

                if (!body.track_id) {
                    return next('no_track');
                }

                var query = {
                    track_id: body.track_id
                };

                if (req.query.process_uuid) {
                    query.process_uuid = req.query.process_uuid;
                }

                return res.redirect(
                    url.format(
                        _.extend(req._controller.getUrl(), {
                            pathname: '/restoration/choose',
                            query,
                            search: null
                        })
                    )
                );
            },
            function(errors) {
                next(errors);
            }
        );
    },
    errorProcess
];

const logMiddleware = [
    apiSetup,
    getYaExperimentsFlags,
    function(req, res, next) {
        const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
        const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

        req.api.statboxLogger({
            ignoreMissedTrack: true,
            action: 'opened',
            mode: 'prerestore',
            path: req.path,
            track_id: res.locals.track_id || null,
            process_uuid: req.query.process_uuid || null,
            ip: req.headers['x-real-ip'],
            host: req.hostname,
            user_agent: req.headers['user-agent'],
            yandexuid: req.cookies.yandexuid,
            origin: (req.nquery && req.nquery.origin) || null,
            experiment_flags: experimentFlags,
            experiment_boxes: experimentBoxes
        });

        next();
    }
];
