const url = require('url');
const PLog = require('plog');
const securityLevels = require('../../configs/common/').securityLevels;

module.exports = function getAccountInfo(req, res, next) {
    req.api
        .profileGetState({need_display_name_variants: false, need_social_profiles: false})
        .then(function(result) {
            const retpath = req.query && req.query.retpath;
            const storeDraft = res.locals.store || (res.locals.store = {});
            const {body = {}} = result;
            const {account = {}} = body;
            const {emails = {}, phones = {}, password_info = {}} = account;
            const isSocialUser = password_info.strength === -1 && !account.is_2fa_enabled;
            const securityLevel = securityLevels[account.security_level];
            const passwordAge = getPasswordAgeStatus(password_info.last_update);
            const getSecuredPhoneData = () => {
                for (const p in phones) {
                    if (!phones.hasOwnProperty(p)) {
                        continue;
                    }

                    if (phones[p].secured && phones[p].bound && phones[p].number) {
                        return phones[p].number.masked_international;
                    }
                }
            };

            if (isSocialUser) {
                if (retpath) {
                    return req.api
                        .validateRetpath({retpath})
                        .then((response = {}) => {
                            const {body} = response;

                            if (!body.retpath) {
                                return req._controller.redirectToFrontpage();
                            }

                            return res.redirect(body.retpath);
                        })
                        .catch((error) => {
                            PLog.info()
                                .logId(req.logID)
                                .type('getAccountInfo')
                                .write('Redirect to profile', error);
                            return req._controller.redirectToFrontpage();
                        });
                } else {
                    return req._controller.redirectToFrontpage();
                }
            }

            if (req.body.mode && req.body.mode === 'updateAccountInfo') {
                return res.json({
                    status: 'ok',
                    info: {
                        securityLevel: securityLevel === 'middle' ? 'low' : securityLevel,
                        is2faEnabled: account.is_2fa_enabled,
                        isAppPasswordsEnabled: account.app_passwords_enabled,
                        phone: getSecuredPhoneData(),
                        email: emails.suitable_for_restore || [],
                        password: {
                            status: password_info.strength && passwordAge === 'normal' ? 'ok' : 'warning',
                            updated: password_info.last_update,
                            age: passwordAge
                        }
                    }
                });
            }

            storeDraft.securityInfo = {
                step: 'welcome',
                securityLevel: securityLevel === 'middle' ? 'low' : securityLevel,
                is2faEnabled: account.is_2fa_enabled,
                isAppPasswordsEnabled: account.app_passwords_enabled,
                phone: getSecuredPhoneData(),
                email: emails.suitable_for_restore,
                password: {
                    status: password_info.strength && passwordAge === 'normal' ? 'ok' : 'warning',
                    updated: password_info.last_update,
                    age: passwordAge
                },
                twoFAUrl: url.format({
                    pathname: '/profile/access/2fa',
                    query: req.query
                })
            };

            return next();
        })
        .catch(function(error) {
            const errorCode = error[0];
            const retpath = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: '/profile/security/'
            });

            PLog.warn()
                .logId(req.logID)
                .type('getAccountInfo')
                .write(error);

            if (req.body.updateAccountInfo) {
                return res.json({status: 'error'});
            }

            if (errorCode === 'sslsession.required') {
                return res.redirect(
                    url.format({
                        protocol: req.headers['x-real-scheme'],
                        hostname: req.hostname,
                        pathname: '/auth/secure/',
                        query: {
                            retpath
                        }
                    })
                );
            }

            return res.redirect(
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: '/auth/',
                    query: {
                        retpath
                    }
                })
            );
        });
};

function getPasswordAgeStatus(timestamp) {
    const PASSWORD_CREATION_DATE_IN_MS = timestamp * 1000; // Sec to ms
    const YEAR_IN_MS = 60 * 60 * 24 * 365 * 1000;
    const TIME_NOW = Date.now();

    const yearAgo = TIME_NOW - YEAR_IN_MS;
    const isPasswordOutdated = yearAgo - PASSWORD_CREATION_DATE_IN_MS > 0;

    let age = 'normal';

    if (isPasswordOutdated) {
        const CURRENT_YEAR = new Date(TIME_NOW).getFullYear();
        const CREATION_YEAR = new Date(PASSWORD_CREATION_DATE_IN_MS).getFullYear();
        const isPasswordOlderThanFive = CURRENT_YEAR - CREATION_YEAR >= 5;

        age = isPasswordOlderThanFive ? 'too_old' : 'old';
    }

    return age;
}
