const PLog = require('plog');
const he = require('he');
const _ = require('lodash');
const config = require('../configs/current');
const apiSetup = require('./common/apiSetup');
const langSetup = require('./common/langSetup');
const createState = require('./common/createState');
const createFormState = require('./common/createFormState');
const getUatraitsData = require('./common/getUatraitsData');
const isUnsupportedBro = require('./common/isUnsupportedBro');
const getCapthca = require('./common/getCaptcha');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const prepareRetpath = require('./common/prepareSocialRetpath');
const rumCounterSetup = require('./common/rumCounterSetup');
const getMetrics = require('./common/getMetrics');
const loc = require('../loc/social');
const locAuth = require('../loc/auth.json');
const closerPath = '/auth/i-social__closer.html';
const getMDAOptions = require('./common/getMDAOptions.js');
const urlFormat = require('./common/urlFormat.js').urlFormat;
const getCustomConfig = require('./common/getCustomConfig');
const customsConfig = getCustomConfig('customs.config.json');
const customs = getCustomConfig('customs.js');

function checkPassportSubsEulaExp(_req, res, next) {
    const experiments = res.locals.experiments && res.locals.experiments.flags;

    if (experiments.includes('passport-subs-eula-check-on')) {
        res.locals.keepUnsubscribedShown = true;
    }

    return next();
}

const getConfig = (origin) => {
    if (customsConfig[origin]) {
        return customsConfig[origin];
    }

    const originMatch = customs.originMatcher.exec(origin);
    const matchedOrigin = originMatch && originMatch[1];

    if (matchedOrigin) {
        if (matchedOrigin === 'cmnt') {
            const originMatchCmnt = customs.originMatcherCmnt.exec(origin);

            if (originMatchCmnt !== null) {
                return customsConfig[`cmnt_${originMatchCmnt[1]}`];
            }
        }

        if (customsConfig[matchedOrigin]) {
            return customsConfig[matchedOrigin];
        }
    }

    return {};
};

const PASSPORT_CONSUMERS = ['com.edadeal.android', 'com.s-g-i.Edadeal', 'ru.beru.android', 'ru.kinopoisk', 'passport'];
const PASSPORT_CONSUMERS_MASK = ['com.yandex.', 'ru.yandex.'];

const isAvailableConsumer = (consumer) => {
    const isAvailableConsumerMask =
        consumer && Boolean(PASSPORT_CONSUMERS_MASK.filter((mask) => consumer.startsWith(mask)).length);

    return PASSPORT_CONSUMERS.includes(consumer) || isAvailableConsumerMask;
};

const isAM = (consumer) => isAvailableConsumer(consumer) && consumer !== 'passport';

const publicException = {
    'consumer.empty': 'consumer-unknown',
    'broker_consumer.empty': 'consumer-unknown',
    'consumer.invalid': 'consumer-unknown',
    'access.denied': 'consumer-unknown',
    'application.empty': 'application-unknown',
    'application.invalid': 'application-unknown',
    'provider.empty': 'provider-unknown',
    'provider.invalid': 'provider-unknown',
    'provider_token.invalid': 'provider-token-invalid',
    'retpath.empty': 'retpath-invalid',
    'retpath.invalid': 'retpath-invalid',
    'track.invalid_state': 'integrity-data',
    'track_id.empty': 'integrity-data',
    'track_id.invalid': 'integrity-data',
    'track.not_found': 'integrity-data',
    'task_id.empty': 'integrity-data',
    'ip.empty': 'integrity-data',
    'host.empty': 'integrity-data',
    'useragent.empty': 'integrity-data',
    'exception.unhandled': 'integrity-data',
    'form.invalid': 'integrity-data',
    'status.empty': 'integrity-data',
    'social.broker_auth_error': 'integrity-data',
    'account.has_no_profiles': 'integrity-data',
    'account.not_found': 'integrity-data',
    'captcha.required': 'integrity-data',
    'frontend_url.empty': 'integrity-data',
    'login_creation.failed': 'internal-exception',
    'backend.database_failed': 'internal-exception',
    'backend.blackbox_failed': 'internal-exception',
    'backend.oauth_failed': 'internal-exception',
    'backend.social_api_failed': 'internal-exception',
    'backend.social_broker_failed': 'internal-exception',
    'backend.redis_failed': 'internal-exception',
    ETIMEDOUT: 'internal-exception',
    ECONNREFUSED: 'internal-exception',
    'name.required': 'name-required',
    'uid.rejected': 'uid-rejected',
    'account.already_registered': 'account-found',
    'sessionid.overflow': 'sessionid-overflow',
    'account.2fa_enabled': '2fa_enabled',
    // 'account.required_change_password': 'required-change-password',
    'account.required_change_password': 'force-change-password',
    'account.strong_password_policy_enabled': 'strong-password-policy-enabled',
    'sessionid.expired': 'sessionid-expired',
    'account.sms_2fa_enabled': 'sms-2fa-enabled'
};

exports.route = (app) => {
    app.get('/auth/social/start', this.social.start)
        .get('/auth/social/start_restoration', this.social.startForRestoration)
        .get('/auth/social/start_secure', this.social.startForSecure)
        .post('/auth/social/native_start', this.social.nativeStart)
        .post('/auth/social/third_party_native_start', this.social.thirdPartyNativeStart)
        .all('/auth/social/callback', this.social.callback)
        .all('/auth/social/callback_secure', this.social.callbackSecure)
        .all('/auth/social/choose', this.social.choose)
        .all('/auth/social/third_party_choose', this.social.thirdPartyNativeChoose)
        .post('/auth/social/register', this.social.register)
        .post('/auth/social/third_party_register', this.social.thirdPartyRegister)
        .get('/auth/social/register', this.social.registerGet)
        .get('/auth/social/register', this.social.thirdPartyRegisterGet)
        .get('/auth/social/finish', this.social.issueCredentials)
        .get(closerPath, (req, res) =>
            res
                .status(200)
                .set('X-Frame-Options', 'DENY')
                .sendFile('i-social__closer.html', {root: 'pages/public_html/'})
        );
};

exports.social = {};

const setup = [
    apiSetup,
    getYaExperimentsFlags,
    checkPassportSubsEulaExp,
    langSetup,
    getUatraitsData,
    createState,
    createFormState,
    getCapthca(),
    rumCounterSetup,
    getMetrics({header: 'Социальная авторизация'}),
    (req, res, next) => {
        if (loc[res.locals.language] === undefined) {
            res.locals.language = 'en'; // если языка нет в кейсетах из танкера, то фолбэчимся на английский
        }

        const storeDraft = res.locals.store || (res.locals.store = {});
        const resultStore = storeDraft.metrics || (storeDraft.metrics = {});

        Object.assign(resultStore, {
            header: 'Социализм'
        });

        res.set({
            'X-Content-Type-Options': 'nosniff',
            'X-Frame-Options': 'Deny',
            'Strict-Transport-Security': 'max-age=31536000'
        });

        next();
    }
];

const getCancelRetpath = (data) => {
    const {retpath, place} = data;

    if (!retpath) {
        return '';
    }

    try {
        const parsedRetpath = new URL(retpath);

        if (place === 'fragment') {
            parsedRetpath.hash = parsedRetpath.hash ? `${parsedRetpath.hash}&status=error` : 'status=error';
        } else {
            parsedRetpath.searchParams.set('status', 'error');
        }
        return parsedRetpath.toString();
    } catch {
        return '';
    }
};

const render = (req, res) => {
    const {language} = res.locals;

    res.render(`social.app.${language}.jsx`);
};

const renderJSXError = (req, res, error, data = {}) => {
    const {language} = res.locals;
    const {profile_link: profileLink, is_native: isNative, retpath, account = {}, provider = {}} = data;
    const {login = ''} = account;
    const {name} = provider;
    const code = error.code || Array.isArray(error) ? error[0] : error;
    const _loc = loc[language];
    const _locAuth = locAuth[language];
    const store = {
        title: _loc.common['errror.title'],
        description: _loc.error[publicException[code] || 'internal-exception'] || '',
        buttonText: _loc.common['button.close'],
        retpath: getCancelRetpath(data),
        isAccountDisabled: ['account.disabled', 'account.disabled_on_deletion', 'account.disabled_with_money'].includes(
            code
        ),
        isInsufficientData: code === 'name.required' && profileLink,
        needChangePass: code === 'account.required_change_password',
        screen: 'error'
    };

    if (store.isAccountDisabled) {
        store.title =
            _locAuth['Errors'][code === 'account.disabled' ? 'ErrorsTexts.badlog_blocked' : 'ErrorsTexts.deleted'];
        store.description = _locAuth['Mend'][code === 'account.disabled' ? 'disabled' : 'deleted'];
    }

    if (['account.strong_password_policy_enabled', 'account.2fa_enabled'].includes(code)) {
        store.title = _loc.common['need.password'];

        if (code === 'account.2fa_enabled') {
            if (isNative) {
                try {
                    const linkToMagic = new URL(retpath);

                    linkToMagic.search = new URLSearchParams({
                        type: 'auth/social',
                        status: 'fail'
                    });

                    store.linkToMagic = linkToMagic.toString();
                } catch {
                    store.linkToMagic = '';
                }
            } else {
                store.goToMagic = true;
                store.login = login;
            }

            store.description = store.description.replace('%1', login);
            store.buttonText = _loc.common['button.continue'];
        }
    }

    if (store.needChangePass) {
        store.title = _loc.common['need.changepassword'];
    }

    if (code === 'name.required') {
        const provName = _loc.common[name] || '';

        store.description = store.description
            .replace('%provider%', provName)
            .replace('%profile.addresses%', profileLink);
    }

    // В случае ошибки в секурном домике в треке есть retpath, также как для нативного
    // AM, но он нам не нужен
    if (res.secure) {
        delete store.retpath;
    }

    res.locals.store.social = store;

    render(req, res);
};

const renderError = (req, res, error, data = {}) => {
    if (req._controller.hasExp('socializm-new-exp')) {
        return renderJSXError(req, res, error, (data = {}));
    }

    const err = error.code || Array.isArray(error) ? error[0] : error;
    const {language} = res.locals;

    var errTmplData = {
        language: res.locals.language,
        loc: loc[language],
        errorTitle: loc[language].common['errror.title'],
        errorCode: err,
        errorMessage: loc[language].error[publicException[err]] || '',
        close: loc[language].common['button.close'],
        retpath: data.retpath,
        paths: config.paths
    };

    if (data.retpath) {
        const cancelRetpath = getCancelRetpath(data);

        errTmplData.cancelRetpath = cancelRetpath.toString();
    }

    if (err === 'account.disabled' || err === 'account.disabled_on_deletion' || err === 'account.disabled_with_money') {
        errTmplData.accountDisabled = 1;
        errTmplData.register = (data && !data.has_enabled_accounts) || '';
        errTmplData.errorTitle =
            locAuth[language]['Errors'][
                err === 'account.disabled' ? 'ErrorsTexts.badlog_blocked' : 'ErrorsTexts.deleted'
            ];
        errTmplData.errorMessage = locAuth[language]['Mend'][err === 'account.disabled' ? 'disabled' : 'deleted'];
    }

    if (err === 'account.strong_password_policy_enabled' || err === 'account.2fa_enabled') {
        errTmplData.errorTitle = loc[language].common['need.password'];
    }

    if (err === 'account.2fa_enabled') {
        if (data.is_native) {
            try {
                const linkToMagic = new URL(data.retpath);

                linkToMagic.search = new URLSearchParams({
                    type: 'auth/social',
                    status: 'fail'
                });

                errTmplData.linkToMagic = linkToMagic.toString();
            } catch {
                errTmplData.linkToMagic = '';
            }
        } else {
            errTmplData.goToMagic = true;
            errTmplData.login = data.account.login;
        }

        errTmplData.close = loc[language].common['button.continue'];
        errTmplData.errorMessage = errTmplData.errorMessage.replace('%1', data.account.login || '');
    }

    if (err === 'account.required_change_password') {
        errTmplData.errorTitle = loc[language].common['need.changepassword'];
        errTmplData.needChangePass = true;
    }

    if (err === 'name.required') {
        if (data.profile_link) {
            errTmplData.insufficientData = 1;
            errTmplData.errorMessage = errTmplData.errorMessage.replace('%profile.addresses%', data.profile_link);
        }

        var provName = loc[language].common[data.provider.name] || '';

        errTmplData.errorMessage = errTmplData.errorMessage.replace('%provider%', provName);
    }

    // В случае ошибки в секурном домике в треке есть retpath, также как для нативного
    // AM, но он нам не нужен
    if (res.secure) {
        delete errTmplData.retpath;
    }

    res.render(`social.error.${language}.js`, errTmplData);
};

const processAccounts = (accounts) =>
    accounts.map((account) => {
        const {host, pathname} = config.paths.avatar;
        const {display_login, login, default_email, display_name, uid} = account;

        return {
            avatar_url: `https://${host}${pathname}`
                .replace('%uid%', display_name.default_avatar)
                .replace('%size%', '200')
                .replace('%login%', login),
            login: default_email || display_login || login,
            originalLogin: login,
            default_avatar: display_name.default_avatar,
            display_name,
            uid
        };
    });

const getSuggestData = (data = {}) => {
    const {
        profile,
        suggested_accounts: suggestedAccounts,
        auth_retpath: authRetpath,
        auth_track_id: authTrackId,
        register_lite_track_id: registerLiteTrackId,
        register_social_track_id: registerSocialTrackId,
        process_uuid: processUUID,
        can_register_social: canRegisterSocial,
        can_register_lite: canRegisterLite,
        track_id: trackId,
        state
    } = data;
    const {email, provider = {}, lastname, firstname, username} = profile || {};
    const socialProfile = {};

    if (profile) {
        socialProfile.login = email;
        socialProfile.provider = he.escape(provider.code);

        let name = '';

        if (lastname || firstname) {
            if (lastname) {
                name += `${lastname} `;
            }
            if (firstname) {
                name += firstname;
            }
        } else if (username) {
            name = username;
        }

        socialProfile.name = he.escape(name);
    }

    return {
        state:
            suggestedAccounts !== undefined
                ? 'auth'
                : canRegisterLite
                ? 'register-lite'
                : canRegisterSocial || state === 'register'
                ? 'register-social'
                : '',
        accounts: suggestedAccounts && processAccounts(suggestedAccounts),
        authRetpath,
        authTrackId,
        registerLiteTrackId,
        registerSocialTrackId: state === 'register' ? trackId : registerSocialTrackId,
        processUUID,
        canRegisterSocial,
        profile: socialProfile
    };
};

const renderJSXPage = (req, res, data) => {
    const {language, page} = res.locals;
    const {location, profile = {}, state, provider = {}, accounts = [], track_id: trackId} = data;
    const {email, firstname, lastname, username} = profile;
    const _loc = loc[language];

    let storeLocation = '';

    try {
        if (location) {
            storeLocation = new URL(location).toString();
        }
        // eslint-disable-next-line
    } catch {}

    const store = {
        profile: {
            login: email,
            provider: he.escape(provider.code || req.body['provider'] || req.query['provider'])
        },
        retpath: getCancelRetpath(data),
        location: storeLocation,
        consumer: config.brokerParams.consumer
    };

    const customConfig = getConfig(req.query.origin);

    if (['callback_third_party', 'callback', 'register_third_party', 'register'].includes(page)) {
        store.mode = state;
        let name = '';

        if (lastname || firstname) {
            name = `${lastname} ${firstname}`;
        } else if (username) {
            name = username;
        }

        store.profile.name = he.escape(name);
    }

    if (['callback_third_party', 'callback'].includes(page) && state === 'choose') {
        const provName = loc[language].common[provider.name] || '';

        accounts.forEach((profile) => {
            if (profile.display_name.name) {
                profile.display_name.name = he.escape(profile.display_name.name);
                profile.name = he.escape(profile.display_name.name);
            }

            if (!profile.display_name.social) {
                profile.display_name.social = {
                    provider: 'ya'
                };
            }
        });

        store.accounts = accounts;
        store.action = '/auth/social/choose';

        if (page === 'callback_third_party') {
            store.action = '/auth/social/third_party_choose';
        }

        store.chooseIntro = _loc.choose[customConfig.isWhiteLabel ? 'intro.whitelabel' : 'intro']
            .replace('%username%', store.userName)
            .replace('%provider%', `<b>${provName}</b>`);
    }

    if (
        (page === 'callback' && state === 'register') ||
        (page === 'callback_third_party' && state === 'register') ||
        ['register', 'register_third_party'].includes(page)
    ) {
        store.action = '/auth/social/register';

        if (['callback_third_party', 'register_third_party'].includes(page)) {
            store.action = '/auth/social/third_party_register';
        }
    }

    if (['register', 'register_third_party'].includes(page) && req.captchaRequired) {
        store.isCaptchaRequired = true;
    }

    if (page === 'callback_third_party') {
        res.locals.page = 'callback';
    }

    if (['callback_third_party', 'callback'].includes(page) && state === 'register') {
        res.locals.page = 'register';
    }

    if (page === 'register_third_party') {
        res.locals.page = 'register';
    }

    if (
        res.locals.page === 'callback' &&
        (data.state === 'suggest' ||
            (data.state === 'register' &&
                isAvailableConsumer(data.broker_consumer) &&
                !isUnsupportedBro(res.locals.ua)))
    ) {
        res.locals.page = 'suggest';
        store.suggest = getSuggestData(data);
        store.suggest.isCaptchaRequired = store.isCaptchaRequired;
        store.suggest.trackId = _.escape(data.track_id || req.body['track_id'] || req.query['track_id']);
        store.suggest.taskId = _.escape(data.task_id || req.body['task_id'] || req.query['task_id']);
        store.suggest.isAM = isAM(data.broker_consumer);
    }

    const track = he.escape(trackId || req.body['track_id'] || req.query['track_id']);

    store.screen = res.locals.page;
    res.locals.store.social = store;
    res.locals.store.tracks = Object.assign({}, res.locals.store.tracks, {
        socialTrackId: track,
        commonTrackId: track
    });

    render(req, res);
};

const renderPage = function(req, res, data) {
    if (req._controller.hasExp('socializm-new-exp')) {
        return renderJSXPage(req, res, data);
    }

    const {language} = res.locals;

    var tmplData = {
        language: language,
        loc: loc[language],
        provider: {
            name: _.escape(data.provider && data.provider.name),
            code: _.escape((data.provider && data.provider.code) || req.body['provider'] || req.query['provider'])
        },
        close: loc[language].common['button.close'],
        confirm: loc[language].common['button.confirm'],
        continue: loc[language].common['button.continue'],
        retpath: data.retpath
    };

    const customConfig = getConfig(req.query.origin);

    if (data.retpath) {
        const cancelRetpath = getCancelRetpath(data);

        tmplData.cancelRetpath = cancelRetpath;
    }

    if (data.state === 'suggest' && isUnsupportedBro(res.locals.ua)) {
        res.locals.page = 'callback';
        data.state = 'register';
    }

    if (res.locals.page === 'spin') {
        tmplData.mode = 'simple';
        try {
            tmplData.location = new URL(data.location).toString();
        } catch {
            tmplData.location = '';
        }
    }

    if (
        res.locals.page === 'callback_third_party' ||
        res.locals.page === 'callback' ||
        res.locals.page === 'register' ||
        res.locals.page === 'register_third_party'
    ) {
        tmplData.consumer = config.brokerParams.consumer;
        tmplData.track_id = _.escape(data.track_id || req.body['track_id'] || req.query['track_id']);
        tmplData.mode = data.state;
        if (data.profile) {
            if (data.profile.lastname || data.profile.firstname) {
                tmplData.userName = '';
                if (data.profile.lastname) {
                    tmplData.userName += `${data.profile.lastname} `;
                }
                if (data.profile.firstname) {
                    tmplData.userName += data.profile.firstname;
                }
            } else if (data.profile.username) {
                tmplData.userName = data.profile.username;
            }

            tmplData.userName = _.escape(tmplData.userName);
        }
    }

    if ((res.locals.page === 'callback_third_party' || res.locals.page === 'callback') && data.state === 'choose') {
        const provName = loc[language].common[tmplData.provider.name] || '';

        _.forEach(data.accounts, function(profile) {
            if (profile.display_name.name) {
                profile.display_name.name = _.escape(profile.display_name.name);
            }

            if (!profile.display_name.social) {
                profile.display_name.social = {
                    provider: 'ya'
                };
            }
        });

        tmplData.accounts = data.accounts;
        tmplData.action = '/auth/social/choose';

        if (res.locals.page === 'callback_third_party') {
            tmplData.action = '/auth/social/third_party_choose';
        }
        tmplData.chooseIntro = loc[language].choose[customConfig.isWhiteLabel ? 'intro.whitelabel' : 'intro']
            .replace('%username%', tmplData.userName)
            .replace('%provider%', `<b>${provName}</b>`);
    }

    if (
        (res.locals.page === 'callback' && data.state === 'register') ||
        (res.locals.page === 'callback_third_party' && data.state === 'register') ||
        res.locals.page === 'register' ||
        res.locals.page === 'register_third_party'
    ) {
        tmplData.action = '/auth/social/register';

        if (res.locals.page === 'callback_third_party' || res.locals.page === 'register_third_party') {
            tmplData.action = '/auth/social/third_party_register';
        }

        tmplData.userNew = loc[language].greet['user.new'];
        tmplData.eula = loc[language].common.eula_gdpr.replace('%button%', tmplData.userNew);
        tmplData.eulaError = loc[language].greet['error.eula'];
    }

    if ((res.locals.page === 'register' || res.locals.page === 'register_third_party') && req.captchaRequired) {
        tmplData.mode = 'captcha';
        tmplData.captcha_failed = req.captchaFailed;

        var captcha = new (require('../blocks/control/captcha/captcha.field'))();

        captcha
            .setMode(req.body.captcha_mode)
            .compile(language, req.api)
            .then(
                function(compiled) {
                    tmplData.captcha = compiled;
                    tmplData.captcha_required = true;
                },
                function() {
                    tmplData.captcha_required = false;
                }
            )
            .then(function() {
                res.render(`social.register.${language}.js`, tmplData);
            });

        return;
    }

    if (res.locals.page === 'callback_third_party') {
        res.locals.page = 'callback';
    }

    if (res.locals.page === 'register_third_party') {
        res.locals.page = 'register';
    }

    if (
        res.locals.page === 'callback' &&
        (data.state === 'suggest' ||
            (data.state === 'register' &&
                isAvailableConsumer(data.broker_consumer) &&
                !isUnsupportedBro(res.locals.ua)))
    ) {
        res.locals.page = 'suggest';
        tmplData.suggest = JSON.stringify(
            Object.assign({}, getSuggestData(data), {
                isCaptchaRequired: tmplData.captcha_required,
                trackId: _.escape(data.track_id || req.body['track_id'] || req.query['track_id']),
                taskId: _.escape(data.task_id || req.body['task_id'] || req.query['task_id']),
                isAM: isAM(data.broker_consumer)
            })
        );
    }

    if (customConfig.isWhiteLabel) {
        tmplData.isWhiteLabel = true;
    }

    res.render(`social.${res.locals.page}.${language}.js`, tmplData);
};

const closer = (req, res, data) => {
    var params = {
        status: data.status
    };
    var parsedLink;
    var link;

    if (data.noauth) {
        return res.redirect(
            urlFormat({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: closerPath,
                query: {
                    task_id: data.task_id,
                    status: data.status
                }
            })
        );
    }

    if (data.yandex_authorization_code) {
        params.yandex_authorization_code = data.yandex_authorization_code;
    }

    // безусловно отправялем на закрывашку, потому что в секьюрном коллбэке
    // есть полноценный трек, в котором есть провалидированный retpath, который
    // нам совсем не нужен в попапе
    if (res.secure) {
        data.place = 'fragment';
        data.retpath = urlFormat({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: closerPath,
            query: {
                status: data.status
            }
        });
    }

    if (data.profile_id && data.return_brief_profile) {
        params['profile.profile_id'] = data.profile_id;
    }

    if (data['x_token']) {
        params['x_token'] = data['x_token'];

        if (data.account) {
            if (data.account.uid) {
                params['uid'] = data.account.uid;
            }

            if (data.account.login) {
                params['login'] = data.account.login;
            }

            if (data.account.display_name && data.account.display_name.name) {
                params['display_name'] = data.account.display_name.name;
            }
        }
    }

    if (data['token']) {
        params['token'] = data['token'];

        if (data.account) {
            if (data.account.uid) {
                params['uid'] = data.account.uid;
            }

            if (data.account.login) {
                params['login'] = data.account.login;
            }

            if (data.account.display_name && data.account.display_name.name) {
                params['display_name'] = data.account.display_name.name;
            }
        }
    }

    try {
        parsedLink = new URL(data.retpath);

        if (data.place && data.place === 'fragment') {
            var args = new URLSearchParams(params).toString();

            parsedLink.hash = parsedLink.hash ? [parsedLink.hash, '&', args].join('') : args;
        } else {
            // считаем, что place = query
            if (typeof parsedLink.search === 'string') {
                const parsed = new URLSearchParams(parsedLink.search);

                Object.entries(params).forEach((pair) => parsed.append(...pair));
                parsedLink.search = parsed.toString();
            } else {
                parsedLink.search = new URLSearchParams(params).toString();
            }
        }

        link = parsedLink.toString();
    } catch {
        link = '';
    }

    // MDA

    const mdaOptions = getMDAOptions(link, {});

    if (mdaOptions.needMDA && data.status === 'ok' && !data.x_token && !data.token && !data.yandex_authorization_code) {
        data.location = mdaOptions.redirectUrl;
    } else {
        data.location = link;
    }

    if (data.is_native) {
        return res.redirect(303, data.location);
    }

    res.locals.page = req._controller.hasExp('socializm-new-exp') ? 'redirector' : 'spin';
    renderPage(req, res, data);
};

const urlGenerate = (req, res, data, err) => {
    const controller = req._controller;

    controller.augmentResponse(data);

    if (err || data.errors) {
        const logger = new PLog(req.logID, 'passport', 'social', 'urlGenerate');

        if (data && data.errors[0] === 'broker.failed') {
            logger.warn('Broker failed');
            closer(req, res, data);
        } else if (data && data.errors[0] === 'captcha.required') {
            logger.warn('Captcha required');
            res.locals.page = 'register';
            req.captchaRequired = true;
            renderPage(req, res, data);
        } else {
            logger.warn(err || data.errors);
            renderError(req, res, err || data.errors, data || null);
        }
        return;
    }

    const trackId = data.track_id || req.body['track_id'] || req.query['track_id'];
    const origin = req.body['origin'] || req.query['origin'];

    if (data.state === 'auth_challenge') {
        return res.redirect(
            urlFormat({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: '/auth/challenge',
                query: {
                    track_id: trackId,
                    display: 'touch',
                    origin,
                    from: 'social_auth'
                }
            })
        );
    }

    if (data.state === 'auth' || data.state === 'broker_error') {
        return closer(req, res, data);
    }

    if (['startForSecure', 'startForRestoration', 'start'].indexOf(res.locals.page) >= 0) {
        var link;
        var params = [
            'application',
            'display',
            'force_prompt',
            'passthrough_errors',
            'provider',
            'scope',
            'sid',
            'consumer'
        ];
        var retpath = Object.assign({}, req._controller.getUrl(), {
            pathname: '/auth/social/callback',
            query: {
                track_id: data.track_id,
                origin
            },
            search: null
        });

        try {
            const brokerUrl = new URL(config.paths.broker.replace('%tld%', req._controller.getTld()));
            const brokerUrlQuery = {};

            if (res.locals.page === 'startForRestoration') {
                params.push('place');
            }

            if (res.locals.page === 'start' && data.state === 'broker.with_auth') {
                params.push('return_brief_profile', 'place');
                brokerUrlQuery['require_auth'] = 1;
                // TODO: тут retpath внезапно строка
                retpath = he.escape(req.body['retpath'] || req.query['retpath']);
            }

            if (res.locals.page === 'startForRestoration') {
                retpath.query['noauth'] = 1;
            }

            if (res.locals.page === 'startForSecure') {
                retpath.pathname = '/auth/social/callback_secure';
            }

            brokerUrlQuery['retpath'] = urlFormat(retpath);

            params.forEach(function(id) {
                var param = req.body[id] || req.query[id] || null;

                if (param) {
                    brokerUrlQuery[id] = he.escape(param);
                }
            });

            if (typeof data.force_prompt !== 'undefined') {
                brokerUrlQuery.force_prompt = data.force_prompt ? '1' : '0';
            }

            const origin = req.query.origin || req.body.origin || null;

            if (origin) {
                const customConfig = getConfig(origin);
                const {provider, application} = brokerUrlQuery;

                if (
                    !application &&
                    provider &&
                    customConfig.socialApplicationMap &&
                    customConfig.socialApplicationMap[provider]
                ) {
                    brokerUrlQuery.application = customConfig.socialApplicationMap[provider];
                }
            }

            brokerUrl.search = new URLSearchParams(brokerUrlQuery);

            link = brokerUrl.toString();
        } catch {
            link = '';
        }

        if (link) {
            data.location = link;
            res.locals.page = req._controller.hasExp('socializm-new-exp') ? 'redirector' : 'spin';
            renderPage(req, res, data);
        }
    }

    if (
        (res.locals.page === 'callback' || res.locals.page === 'callback_third_party') &&
        ['register', 'choose', 'suggest'].includes(data.state)
    ) {
        renderPage(req, res, data);
    }

    if (res.locals.page === 'register' || res.locals.page === 'register_third_party') {
        renderPage(req, res, data);
    }
};

const getParams = (req, res, next) => {
    const data = {};
    const {params = []} = res.locals;

    params.forEach((id) => {
        const param = req.body[id] || req.query[id] || null;

        if (param) {
            let field = id;

            if (id === 'form') {
                field = 'service';
            }

            if (id === 'consumer') {
                field = 'broker_consumer';
            }

            data[field] = param;
        }
    });

    if (data.origin) {
        const customConfig = getConfig(data.origin);

        if (
            !data.application &&
            data.provider &&
            customConfig.socialApplicationMap &&
            customConfig.socialApplicationMap[data.provider]
        ) {
            data.application = customConfig.socialApplicationMap[data.provider];
        }
    }

    res.locals.params = data;

    return next();
};

const doRequest = (req, res, next) => {
    const {method, params} = res.locals;
    const data = method
        ? {
              form: params,
              method
          }
        : params;

    req.api[res.locals.handle](data, {hostname: req.hostname})
        .then((response) => urlGenerate(req, res, response.body))
        .catch((err) => {
            PLog.warn()
                .logId(req.logID)
                .type('social')
                .write(err);

            return next(err);
        });
};

const checkCaptcha = (req, res, next) => {
    const {key, captcha} = req.body;

    if (!key) {
        return next();
    }

    req.captchaRequired = true;
    req.api
        .captchaCheck({answer: captcha})
        .then((result = {body: {}}) => {
            delete req.body.captcha;

            req.captchaFailed = !result.body.correct;

            next();
        })
        .catch(next);
};

exports.social.start = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'start',
            params: [
                'application',
                'code_challenge',
                'code_challenge_method',
                'consumer',
                'domik',
                'force_prompt',
                'from',
                'origin',
                'passthrough_errors',
                'place',
                'provider',
                'process_uuid',
                'retpath',
                'return_brief_profile',
                'scope',
                'am_version',
                'am_version_name',
                'app_id',
                'app_platform',
                'app_version',
                'app_version_name',
                'deviceid',
                'device_id',
                'device_name',
                'ifv',
                'manufacturer',
                'model',
                'os_version',
                'uuid'
            ],
            handle: 'socialStart'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.startForRestoration = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'startForRestoration',
            params: [
                'retpath',
                'place',
                'return_brief_profile',
                'from',
                'domik',
                'process_uuid',
                'provider',
                'application',
                'origin',
                'consumer'
            ],
            handle: 'socialStart'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.startForSecure = [
    setup,
    (req, res) => {
        res.locals.page = 'startForSecure';

        return urlGenerate(req, res, {
            track_id: req.query.track_id
        });
    }
];

exports.social.nativeStart = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'callback',
            params: [
                'retpath',
                'place',
                'application',
                'provider',
                'provider_token',
                'provider_token_secret',
                'scope',
                'provider',
                'application',
                'origin',
                'app_id',
                'app_platform',
                'app_version',
                'manufacturer',
                'model',
                'uuid',
                'device_name',
                'os_version',
                'process_uuid',
                'ifv',
                'deviceid',
                'device_id',
                'consumer',
                'am_version',
                'am_version_name',
                'app_version_name'
            ],
            handle: 'socialStartNative'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.thirdPartyNativeStart = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'callback_third_party',
            params: [
                'retpath',
                'place',
                'application',
                'provider',
                'provider_token',
                'provider_token_secret',
                'scope',
                'provider',
                'application',
                'origin',
                'app_id',
                'app_platform',
                'app_version',
                'manufacturer',
                'model',
                'uuid',
                'device_name',
                'os_version',
                'process_uuid',
                'ifv',
                'deviceid',
                'device_id',
                'consumer',
                'am_version',
                'am_version_name',
                'app_version_name'
            ],
            handle: 'socialStartNativeThirdParty'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.callback = [
    setup,
    (req, res, next) => {
        if (req.query['noauth']) {
            return closer(req, res, Object.assign({}, req.query));
        }

        next();
    },
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'callback',
            params: ['code', 'track_id', 'task_id', 'status'],
            handle: 'socialCallback'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.callbackSecure = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'callback',
            params: ['track_id', 'task_id', 'status'],
            handle: 'socialCallbackSecure'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.choose = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'choose',
            params: ['track_id', 'uid'],
            handle: 'socialChoose'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.thirdPartyNativeChoose = [
    setup,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'choose',
            params: ['track_id', 'uid'],
            handle: 'socialChooseNativeThirdParty'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.register = [
    setup,
    checkCaptcha,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'register',
            params: ['track_id', 'eula_accepted', 'unsubscribe_from_maillists'],
            method: 'POST',
            handle: 'socialRegister'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.asyncRegister = [
    setup,
    (req, res, next) => {
        const {key, captcha} = req.body;

        if (!key) {
            return next();
        }

        req.api
            .captchaCheck({answer: captcha})
            .then((result = {body: {}}) => {
                delete req.body.captcha;
                delete req.body.key;

                if (result.body.correct) {
                    return next();
                }

                throw [{code: 'captcha.not_matched'}];
            })
            .catch((error) =>
                res.json({
                    status: 'error',
                    errors: error
                })
            );
    },
    (req, res) => {
        const controller = req._controller;
        const {track_id, eula_accepted, retpath, firstname, lastname, unsubscribe_from_maillists, origin} = req.body;
        const params = {form: {track_id, eula_accepted, firstname, lastname}};

        if (unsubscribe_from_maillists) {
            params.form.unsubscribe_from_maillists = unsubscribe_from_maillists;
        }

        if (origin) {
            params.form.origin = origin;
        }

        req.api
            .socialRegister(params)
            .then((response = {}) => {
                const {body = {}} = response;
                const {errors, status} = body;

                if (status === 'ok') {
                    controller.augmentResponse(body);
                    return res.json({status, retpath: prepareRetpath(retpath, body)});
                }

                throw errors;
            })
            .catch((error) =>
                res.json({
                    status: 'error',
                    errors: error
                })
            );
    }
];

const trackRe = /^\w{16,34}$/;

exports.social.asyncCallback = [
    setup,
    (req, res) => {
        const trackId = req.cookies.socialParams || '';

        if (!trackId) {
            return res.json({
                status: 'error',
                errors: ['track_id.empty']
            });
        }

        if (!trackRe.test(trackId)) {
            return res.json({
                status: 'error',
                errors: ['track_id.invalid']
            });
        }

        req.api
            .readTrack(trackId)
            .then((response = {}) => {
                const {body = {}} = response;
                const {social_task_id: taskId} = body;

                const params = {
                    track_id: trackId,
                    task_id: taskId,
                    status: 'ok'
                };

                req.api.socialCallback(params).then((response = {}) => {
                    const {body = {}} = response;
                    const {errors, status} = body;

                    if (status === 'ok') {
                        return res.json({status, suggest: getSuggestData(body)});
                    }

                    throw errors;
                });
            })
            .catch((error) =>
                res.json({
                    status: 'error',
                    errors: error
                })
            );
    }
];

exports.social.thirdPartyRegister = [
    setup,
    checkCaptcha,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'register_third_party',
            params: ['track_id', 'eula_accepted'],
            method: 'POST',
            handle: 'socialRegisterThirdParty'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.registerGet = [
    setup,
    checkCaptcha,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'register',
            params: ['track_id'],
            method: 'GET',
            handle: 'socialRegister'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.thirdPartyRegisterGet = [
    setup,
    checkCaptcha,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'register_third_party',
            params: ['track_id'],
            method: 'GET',
            handle: 'socialRegisterThirdParty'
        });

        next();
    },
    getParams,
    doRequest
];

exports.social.issueCredentials = [
    setup,
    checkCaptcha,
    (req, res, next) => {
        res.locals = Object.assign({}, res.locals, {
            page: 'close',
            params: ['track_id'],
            method: 'POST',
            handle: 'socialIssueCredentials'
        });

        next();
    },
    getParams,
    doRequest
];
