var url = require('url');
var apiSetup = require('./common/apiSetup');
var config = require('../configs/current');
var langs = config.langs;
var locs = require('../loc/auth.json');
var _ = require('lodash');
var locTitles = require('../loc/common.json');
var locMessages = require('../loc/subscriptions.json');
var PLog = require('plog');
var PForm = require('pform');
var PCaptcha = require('../blocks/control/captcha/captcha.field');
var PPhoneConfirm = require('../blocks/control/phone-confirm/phone-confirm.field');
var PCurrentPasswd = require('../blocks/control/password-current/password-current.field');
var CurrentAnswer = require('../blocks/control/current-answer/current-answer.field');
var dheaderSetup = require('./common/dheaderSetup');
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

var langSetup = [
    function(req, res, next) {
        if (!req.blackbox && req.body['language']) {
            req.blackbox = {
                dbfields: {
                    'userinfo.lang.uid': req.body['language']
                }
            };
        }
        next();
    },
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('unsubscribe service')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    function(req, res, next) {
        var language = langs[langs.indexOf(res.locals.language)] || 'ru';

        res.loc = locs[language];

        if (req.api) {
            req.api.language(language);
        }

        next();
    }
];

var buildForm = {
    phone() {
        return new PForm(
            new PCaptcha().setOptions({asyncCheck: true, popup: true}).setRequired(),
            new PPhoneConfirm().setRequired().setOptions({
                hasConfirmedPhone: true,
                prevCheckCaptcha: true,
                mode: 'tracked'
            }),
            new PCurrentPasswd()
                .setLabel('%subscriptions.field_password_current')
                .setOption('showEntry', true)
                .setRequired()
        );
    },
    question() {
        return new PForm(
            new PCaptcha().setOptions({asyncCheck: true, popup: true}).setRequired(),
            new CurrentAnswer().setRequired(),
            new PCurrentPasswd()
                .setLabel('%subscriptions.field_password_current')
                .setOption('showEntry', true)
                .setRequired()
        );
    }
};

var initUnsubscribe = [
    function(req, res, next) {
        if (!req.api) {
            apiSetup(req, res, unsubscribeSubmit);
        } else {
            unsubscribeSubmit(req, res, next);
        }
        function unsubscribeSubmit(req, res, next) {
            var service = req.nquery && req.nquery.from;
            var controller = req._controller;
            var defaultRetpath = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: 'passport',
                query: {
                    mode: 'passport'
                }
            });
            var altRetpath = req.nquery && req.nquery.retpath;

            if (!service) {
                res.redirect(defaultRetpath);
            }

            req.api
                .submitUnsubscribe({
                    service,
                    retpath: altRetpath || defaultRetpath
                })
                .then(function(result) {
                    var body = (result && result.body) || null;
                    var phoneNotAvailable = req.nquery.number;
                    var link;

                    if (!body) {
                        processError(['exception.unhandled'], req, res);
                        renderPage(req, res);
                    }

                    res.locals.uid = body.account.uid;
                    res.locals.track_id = body.track_id || '';

                    if (body.state && result.body.state === 'complete_social') {
                        link = url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: 'passport',
                            query: {
                                mode: 'postregistration',
                                retpath: controller.getUrl().href
                            }
                        });

                        res.redirect(link);
                    }

                    if (body.number && !phoneNotAvailable) {
                        res.locals.case = 'user_phone';
                        res.locals.phone = body.number;
                    }
                    next();
                })
                .catch(function(error) {
                    new PLog(req.logID, 'passport', 'unsubscribe', 'initUnsubscribe').warn(`Error:${error}`);
                    processError(error, req, res);
                    renderPage(req, res);
                });
        }
    },
    function(req, res, next) {
        if (!res.locals.phone || req.nquery.number) {
            req.api.getCurrentQuestion().then(
                function(result) {
                    var body = (result && result.body) || null;
                    var bodyQuestion = result.body.question;

                    if (!body) {
                        processError(['exception.unhandled'], req, res);
                        renderPage(req, res);
                    }

                    if (bodyQuestion) {
                        var question = {};

                        question.id = `${bodyQuestion.id}:${bodyQuestion.text}`;
                        question.text = bodyQuestion.text;

                        res.locals.case = 'user_questions';
                        res.locals.question = question;
                        if (req.nquery.number) {
                            res.locals.phone = '';
                        }
                        next();
                    } else {
                        res.locals.case = 'restoration';
                        res.locals.phone = '';
                        res.locals.question = '';
                        next();
                    }
                },
                function(error) {
                    processError(error, req, res);
                    res.locals.track_id = req.body.track_id;
                    res.locals.service = req.body.service;
                    renderPage(req, res);
                }
            );
        } else {
            next();
        }
    }
];

var renderCaptcha = function(req, res) {
    var captcha = new PCaptcha().setOption('asyncCheck', true).setRequired();

    if (res.captchaError) {
        captcha.getErrorByCode(res.captchaError).setActive();
    }
    if (res.locals.captcha_required) {
        captcha
            .setMode(req.body.captcha_mode)
            .compile(res.locals.language, req.api)
            .then(
                function(compiled) {
                    res.locals.captcha = {control: compiled};
                    res.locals.captcha_required = true;
                },
                function() {
                    res.locals.captcha_required = false;
                }
            )
            .then(function() {
                res.render(`unsubscribe.${res.locals.language}.js`);
            });
    }
};

exports.routes = {};

exports.route = function(app) {
    var routes = this.routes;

    app.all(
        '/passport/?',
        function(req, res, next) {
            if (req.nquery && req.nquery.mode === 'unsubscribe') {
                next();
            } else {
                next('route');
            }
        },
        function(err, req, res, next) {
            if (req.nquery && req.nquery.mode === 'unsubscribe') {
                next();
            } else {
                next('route');
            }
        },
        routes.main
    );

    app.get('/profile/unsubscribe', routes.main);
    app.post('/profile/unsubscribe', routes.unsubscribe);
};

exports.routes.main = [
    apiSetup,
    langSetup,
    initUnsubscribe,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    writeStatbox,
    renderPage
];

exports.routes.unsubscribe = [
    apiSetup,
    langSetup,
    initUnsubscribe,
    checkCaptcha,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function(req, res, next) {
        var data = {
            track_id: req.body.track_id,
            password: req.body.current_password
        };

        const controller = req._controller;

        req.api.commitUnsubscribe(data).done(function(result) {
            controller.augmentResponse(result.body);

            if (result.body && result.body.cookies) {
                var cookies = result.body.cookies;

                if (Array.isArray(cookies) && cookies.length) {
                    res.redirect(result.body.retpath);
                }
            }
        }, next);
    },
    function(error, req, res, next) {
        processError(error, req, res, next);
        res.locals.track_id = req.body.track_id;
        res.locals.service = req.body.service;
        renderPage(req, res);
    },
    renderPage
];

function renderPage(req, res) {
    var form;
    var lang = res.locals.language;
    var serviceName = (req.nquery && req.nquery.from) || '';
    var loc = locTitles[lang];
    var title = `serviceTitles.${serviceName}`;

    if (res.locals.phone && res.locals.case !== 'user_questions') {
        form = buildForm['phone']().setApi(req.api);
        var phoneField = res.locals.phone.masked_international;
        var phone = form.getField('phone_number');

        if (phone) {
            phone.setValue(phoneField);
        }
    } else {
        form = buildForm['question']().setApi(req.api);
    }

    if (res.locals.isWrong) {
        form.getField('current_password')
            .getErrorByCode('not_matched')
            .setActive();
    }

    if (res.locals.isEmpty) {
        form.getField('current_password')
            .getErrorByCode('missingvalue')
            .setActive();
    }

    return form.compile(res.locals.language).then(function(data) {
        res.locals.form = {control: data};
        res.locals.service = serviceName;
        res.locals.serviceTitle = loc['common'][title];

        if (res.locals.captcha_required) {
            renderCaptcha(req, res);
        } else {
            res.render(`unsubscribe.${res.locals.language}.js`);
        }
    });
}

function processError(errors, req, res) {
    var lang = res.locals.language;
    var errorCode = errors[0];
    var basicUrlObj = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: '/profile/unsubscribe/',
        query: {
            from: req.nquery.from
        }
    };

    var errorsWithMessages = {
        'exception.unhandled': locs[lang]['Errors']['ErrorsTexts.subscr_err'],
        'account.disabled': locMessages[lang]['ServicesSubscriptions']['subscriptions.account-disabled'],
        'account.disabled_on_deletion': locMessages[lang]['ServicesSubscriptions']['subscriptions.account-disabled'],
        'user.not_verified': locMessages[lang]['ServicesSubscriptions']['subscriptions.user-not_verified']
    };

    var retpath = url.format(basicUrlObj);

    if (!Array.isArray(errors)) {
        return;
    }

    if (errorCode === 'account.no_question') {
        res.locals.case = 'restoration';
    }

    if (errorCode === 'subscription.blocked') {
        res.locals.subscriptionBlocked = true;
    }

    if (errorCode === 'session.invalid' || errorCode === 'sessionid.invalid' || errorCode === 'is_pdd.invalid') {
        return res.redirect(
            url.format(
                _.extend(basicUrlObj, {
                    pathname: '/auth/',
                    query: {
                        retpath
                    }
                })
            )
        );
    }

    if (errorCode === 'sslsession.required') {
        return res.redirect(
            url.format(
                _.extend(basicUrlObj, {
                    pathname: '/auth/secure/',
                    query: {
                        retpath
                    }
                })
            )
        );
    }

    if (errors.includes('password.not_matched')) {
        res.locals.isWrong = true;
    }

    if (errorCode === 'password.empty') {
        res.locals.isEmpty = true;
    }

    if (
        errorCode === 'account.not_subscribed' ||
        errorCode === 'service.invalid' ||
        errorCode === 'account.not_found'
    ) {
        res.redirect(config.paths.retpath);
    }

    if (errorCode === 'captcha.required') {
        res.locals.captcha_required = true;
    }

    if (
        errorCode === 'backend.yasms_failed' ||
        errorCode === 'access.denied' ||
        errorCode === 'backend.blackbox_failed'
    ) {
        res.locals.errorMessage = errorsWithMessages['exception.unhandled'];
    }

    if (errorCode === 'track_id.invalid' || errorCode === 'track_id.empty' || errorCode === 'track.not_found') {
        return res.redirect(url.format(basicUrlObj));
    }

    _.forIn(errorsWithMessages, function(value, key) {
        if (errorCode === key) {
            res.locals.errorMessage = errorsWithMessages[key];
        }
    });
}

function checkCaptcha(req, res, next) {
    if (!req.body.key) {
        return next();
    }
    req.api.captchaCheckStatus(req.body).then(
        function(result) {
            var body = (result && result.body) || null;

            if (body && body.is_recognized && body.is_checked) {
                return next();
            }

            req.api.captchaCheck(req.body).then(
                function(result) {
                    delete req.body.answer;
                    if (result.body.correct) {
                        next();
                    } else {
                        res.captchaError = 'incorrect';
                        next();
                    }
                },
                function() {
                    res.captchaError = 'missingvalue';
                    next();
                }
            );
        },
        function() {
            next();
        }
    );
}

function writeStatbox(req, res, next) {
    const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
    const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

    var statBoxData = {
        mode: 'unsubscribe',
        ignoreMissedTrack: true,
        action: 'opened',
        service: req.nquery && req.nquery.from,
        track_id: res.locals.track_id || null,
        uid: res.locals.uid,
        yandexuid: req.cookies['yandexuid'],
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        experiment_flags: experimentFlags,
        experiment_boxes: experimentBoxes
    };

    req.api.statboxLogger(statBoxData).catch(function(err) {
        PLog.warn()
            .logId(req.logID)
            .type('unsubscribe service')
            .write(err);
    });

    next();
}
