const PLog = require('plog');
const config = require('../configs/current');
const {urlFormat, urlParse} = require('./common/urlFormat');
const {getFirstSymbolForRetpathQuery} = require('./common/getFirstSymbolForRetpathQuery');
const langSetup = require('./common/langSetup');
const apiSetup = require('./common/apiSetup');
const getUatraitsData = require('./common/getUatraitsData');
const validateCSRF = require('./common/validateCSRF.js');
const writeStatbox = require('./common/writeStatbox');
const checkAuth = require('./common/checkAuth');
const rumCounterSetup = require('./common/rumCounterSetup');
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const validateRetpath = require('./common/validateRetpath');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const getMetrics = require('./common/getMetrics');
const createFormState = require('./common/createFormState');
const createCustomsState = require('./common/createCustomsState');
const express = require('express');
const router = express.Router();
const {env = {}} = process;
const {INTRANET, NODE_ENV} = env;
const SAFETY_REGEXP = /[^0-9]/g;
const ORIGINS_CAN_USE_ONE_STEP_3DS = ['family'];
const ORIGINS_WITHOUT_FOOTER_ON_3DS = ['taxi_familycard'];
const ORIGINS_ONLY_MOBILE_3DS = ['taxi_familycard'];
const ONE_STEP_CHALLENGE_ORIGINS = ['pay'];
const ONE_STEP_CHALLENGE_ALLOWED_TYPES = ['phone_confirmation'];

const checkIsOneStepChallenge = (origin, challengeType) =>
    ONE_STEP_CHALLENGE_ORIGINS.includes(origin) && ONE_STEP_CHALLENGE_ALLOWED_TYPES.includes(challengeType);

const getReturnPath = (error = 'track.not_found', retpath) => {
    if (!retpath) {
        return;
    }

    const queryStatus = ['action.impossible', 'challenge.not_enabled'].includes(error) ? 'not_available' : 'error';

    return `${retpath}${getFirstSymbolForRetpathQuery(retpath)}status=${queryStatus}`;
};

const enter = [
    apiSetup,
    checkAuth,
    getYaExperimentsFlags,
    getUatraitsData,
    langSetup,
    rumCounterSetup,
    validateRetpath,
    (req, res, next) => {
        const {locals = {}} = res;
        const {query = {}, hostname} = req;
        const {
            language,
            ua = {},
            track_id,
            userType = {},
            regionId,
            countryId,
            experiments,
            validRetpath: retpath,
            csrf,
            country
        } = locals;
        const {uid} = query;
        const {paths = {}, version} = config;
        const langSwitcher = new LangSwitcherView(req._controller);
        const protocol = req.headers['x-real-scheme'];
        const {static: staticPath} = paths;
        const {metricsUserType} = userType;

        locals.store = {
            auth: {
                loading: false
            },
            settings: {
                version,
                metricsUserType,
                staticPath,
                host: urlFormat({
                    protocol,
                    hostname
                }),
                embeddedauth: paths.embeddedauth || '',
                soundManagerPath: paths.soundManager,
                help: paths.help || {},
                tld: req._controller.getTld(),
                language,
                country,
                ua,
                env: {
                    type: NODE_ENV,
                    name: INTRANET
                },
                regionId,
                countryId,
                isTouch: ua.isMobile || ua.isTouch
            },
            common: {
                uid: (uid || '').replace(SAFETY_REGEXP, ''),
                yandexuid: res._yandexuid && res._yandexuid.replace(SAFETY_REGEXP, ''),
                track_id,
                retpath,
                experiments,
                currentPage: req._controller.getUrl().href,
                profilePage: urlFormat({...req._controller.getUrl(), pathname: '/profile'}),
                actionForRepeat: null,
                dev: config.dev,
                csrf,
                loadCustomsStyles: true
            },
            isLoading: false,
            footer: {
                langlist: []
            }
        };

        langSwitcher
            ._compile()
            .then((response = {}) => {
                const {langlist = {}} = response;

                locals.store.footer.langlist = langlist;
                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('common.createState, langSwitcher')
                    .write(error);
            });
    },
    createCustomsState,
    function getInitialData(req, res, next) {
        const {query: {track_id: trackId, origin} = {}} = req;
        const state = res.locals.store;
        const retpath =
            res.locals.validRetpath ||
            urlFormat({
                hostname: req.hostname,
                pathname: '/profile'
            });

        if (!trackId) {
            const finalPath = `${retpath}${getFirstSymbolForRetpathQuery(retpath)}status=error`;

            PLog.warn()
                .logId(req.logID)
                .type('user-validate, getInitialData')
                .write('no track provided');
            return req._controller.redirect(finalPath);
        }

        req.api
            .userValidateSubmit(trackId)
            .then((result) => {
                const {
                    body: {
                        status,
                        challenges_enabled: challengeInfo = {},
                        default_challenge: challengeType,
                        track_id: trackId,
                        retpath: trackRetpath,
                        errors
                    }
                } = result;

                const isOneStepChallenge = checkIsOneStepChallenge(origin, challengeType);

                if (challengeType === '3ds') {
                    const retpathQueryParams = urlParse(trackRetpath).query || {};
                    const {origin: retpathOrigin = ''} = retpathQueryParams;
                    const hideFooter = ORIGINS_WITHOUT_FOOTER_ON_3DS.includes(retpathOrigin);
                    const onlyMobile3ds = ORIGINS_ONLY_MOBILE_3DS.includes(retpathOrigin);
                    const isOneStep3dsExp =
                        ORIGINS_CAN_USE_ONE_STEP_3DS.includes(origin) &&
                        (res.locals.experiments.flags || []).includes('chaas-3ds-one-step-exp');

                    if (onlyMobile3ds) {
                        state.settings.ua.isTablet = false;
                        state.settings.ua.isMobile = true;
                        state.settings.isTouch = true;
                    }

                    state.challenge = {
                        challengeType,
                        fieldValue: '',
                        errorText: '',
                        errorCode: '',
                        is3dsLoading: isOneStep3dsExp,
                        isOneStep3dsExp,
                        isOneStepChallenge
                    };
                    state.auth.loading = false;
                    state.common.track_id = trackId;
                    state.pagePopup = {
                        isShowPagePopup: false,
                        pagePopupType: null
                    };
                    state.footer.hideFooter = hideFooter;

                    return next();
                }

                if (status !== 'ok') {
                    const errorCode = Array.isArray(errors) && errors[0];

                    if (errorCode === 'challenge.limit_exceeded') {
                        state.challenge = {
                            errorCode: 'challenge.limit_exceeded'
                        };

                        return next();
                    }
                    return req._controller.redirect(getReturnPath(errorCode, trackRetpath) || retpath);
                }

                state.pagePopup = {
                    isShowPagePopup: isOneStepChallenge,
                    pagePopupType: isOneStepChallenge ? 'challengeCall' : null
                };
                state.auth.loading = isOneStepChallenge;
                state.challenge = {
                    challengeType,
                    hint: challengeInfo[challengeType].hint,
                    fieldValue: '',
                    errorText: '',
                    errorCode: '',
                    isSms2faChallenge: res.locals.isSms2faChallenge,
                    isStandAloneChallenge: true,
                    isOneStepChallenge
                };

                state.common.track_id = trackId;

                if (challengeType === 'phone_confirmation' || challengeType === 'dictation') {
                    const {phone_id: phoneId} = challengeInfo[challengeType];

                    state.challenge.phoneId = phoneId;
                }

                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('user-validate, getInitialData')
                    .write(error);

                return next();
            });
    },
    createFormState,
    writeStatbox({
        action: 'opened',
        mode: 'user-validate'
    }),
    getMetrics({
        header: 'Страница challenge-as-a-service'
    }),
    function renderPage(req, res) {
        res.render(`react.user-validate.${res.locals.language}.jsx`);
    }
];

const commitUserValidate = [
    apiSetup,
    validateCSRF,
    function validateCommit(req, res, next) {
        req.api
            .userValidateCommit(req.body)
            .then(function(response = {}) {
                const {body = {}} = response;

                if (body.status === 'ok') {
                    return next();
                }

                return res.json({
                    status: 'error',
                    errors: response.errors || []
                });
            })
            .catch(function(error = {}) {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    },
    function validateSave(req, res) {
        req.api
            .userValidateSave({track_id: req.body.track_id})
            .then((response = {}) => {
                const {body = {}} = response;

                if (body.status === 'ok' && body.retpath) {
                    return res.json({status: 'ok', retpath: body.retpath});
                }
            })
            .catch((error = {}) => {
                return res.json({
                    status: 'error',
                    errors: error.errors || []
                });
            });
    }
];

router.get('/', enter);
exports.router = router;
exports.commitUserValidate = commitUserValidate;
