const url = require('url');
const _ = require('lodash');
const when = require('when');
const express = require('express');
const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');
const getTrack = require('./common/getTrack');
const config = require('../configs/current');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const userapprovePage = express.Router();
const userapproveAjax = express.Router();
const assign = Object.assign;

userapprovePage.get('/', [
    function(req, res, next) {
        const mode = req.query.mode;

        if (mode === 'userapprove') {
            res.locals.passportHost = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname
            });
            return next();
        }

        return next('route');
    },
    function getLanguage(req, res, next) {
        req._controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('userapprove')
                    .write(err);
            })
            .done(function() {
                next();
            });
    },
    apiSetup,
    getTrack(),
    getYaExperimentsFlags,
    multiAuthAccountsSetup,
    function getUserKarma(req, res, next) {
        const retpath = req.query.retpath;

        req.api
            .userApproveSubmit({
                retpath
            })
            .then(function(response) {
                const body = response && response.body;

                if (body.state === 'karma_washed') {
                    if (body.retpath) {
                        return req._controller.redirect(body.retpath);
                    }

                    return req._controller.redirectToYandexTld();
                }

                res.locals.track_id = body.track_id;
                return next();
            })
            .catch(function(errors) {
                if (errors.includes('sessionid.invalid')) {
                    return req._controller.redirectToAuth();
                }

                if (errors.includes('action.not_required')) {
                    return req._controller.followRetpathOrYandexTld();
                }

                return req._controller.redirectToYandexTld();
            });
    },
    function getProfile(req, res, next) {
        req.api
            .profileGetState()
            .then(function(response) {
                const profile = response.body;

                res.locals.account = profile.account || {};

                return next();
            })
            .catch(function(errors) {
                let query;

                PLog.warn()
                    .logId(req.logID)
                    .type('passport.userapprove')
                    .write(errors);

                if (Array.isArray(errors)) {
                    if (errors.includes('sessionid.invalid') || errors.includes('account.disabled')) {
                        return req._controller.redirectToAuth();
                    }

                    if (errors[0].code && errors[0].code === 'unknowntrack') {
                        query = assign({}, req.query);

                        delete query.track_id;

                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: req.path,
                                query
                            })
                        );
                    }
                }

                return next(errors);
            });
    },
    function getStore(req, res, next) {
        const langSwitcher = new LangSwitcherView(req._controller);
        const locals = res.locals;
        const {account = {}, userType = {}} = locals;
        const tld = req._controller.getTld();
        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {metricsUserType} = userType;
        const {blackbox: {connection_id: ci = ''} = {}} = req;

        let retpath = req.query.retpath;
        const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
        const currentUrl = req._controller.getUrl();

        if (retpath) {
            handlers.push(req.api.validateRetpath({retpath}));
        } else {
            retpath = url.format(
                assign({}, currentUrl, {
                    pathname: 'profile',
                    search: null,
                    query: null
                })
            );
        }

        locals.store = {
            settings: {
                host: req.hostname,
                avatar: config.paths.avatar || {},
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.ru || {},
                help: config.paths.help || {},
                tld,
                phonesLink: url.format(
                    assign({}, currentUrl, {
                        pathname: 'profile/phones',
                        search: null,
                        query: {
                            retpath: url.format(currentUrl)
                        }
                    })
                ),
                language: locals.language,
                ua: {},
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType,
                staticPath
            },
            common: {
                actionForRepeat: null,
                uid: account.uid,
                yandexuid: res._yandexuid && res._yandexuid.replace(/[^0-9]/g, ''),
                track_id: locals.track_id,
                retpath: null,
                historyOnPassport: 1,
                experiments: res.locals.experiments,
                currentPage: req._controller.getUrl().href,
                defaultPage: url.format({
                    pathname: '/profile',
                    query: req.query
                }),
                isPDD: account.domain !== undefined,
                isWSUser: locals.isWSUser
            },
            person: assign({}, account.person || {}, {
                uid: account.uid,
                login: account.login,
                displayLogin: account.display_login,
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayNames: account.display_names || {},
                displayName: (account.display_name && account.display_name.name) || '',
                escapedDisplayName: (account.display_name && _.escape(account.display_name.name)) || '',
                errors: {}
            }),
            captcha: {
                loading: false,
                loadingAudio: false,
                playing: false,
                type: 'text',
                key: null,
                imageUrl: null,
                introSound: null,
                captchaSound: null
            },
            header: {
                accounts: (locals.accounts && locals.accounts.accounts) || [],
                defaultAccount: (locals.accounts && locals.accounts.defaultAccount) || {},
                canAddMore: locals.accounts && locals.accounts['can-add-more'],
                ci
            },
            footer: {
                langlist: []
            },
            metrics: {
                header: 'Страница userapprove'
            },
            monitoring: {
                page: 'userapprove'
            }
        };

        when.all(handlers)
            .then(function(response) {
                const uatraits = response[0] || {};
                const langlist = (response[1] && response[1].langlist) || {};
                const validatedRetpath = response[2] || retpath;

                locals.store.settings.ua = uatraits;
                locals.store.footer.langlist = langlist;

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                }
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('userapprove')
                    .write(err);
            })
            .done(function() {
                delete locals.account;
                delete locals.accounts;

                return next();
            });
    },
    getMetrics({
        header: 'Страница userapprove'
    }),
    rumCounterSetup,
    function renderPage(req, res) {
        const lang = res.locals.language;

        res.render(`mode.userapprove.${lang}.jsx`);
    }
]);

userapproveAjax.post('/commit', [
    apiSetup,
    function captcha(req, res, next) {
        const {captcha, track_id} = req.body;

        req.api
            .captchaCheck({
                track_id,
                answer: captcha
            })
            .then(function(response) {
                const captcha = response.body;

                let promise;

                if (Array.isArray(response)) {
                    captcha.forEach(function(item) {
                        if (item.code === 'captchalocate') {
                            promise = when.reject(['captcha.cannot_locate']);
                        }
                    });
                } else if (captcha.correct === false) {
                    promise = when.reject(['captcha.not_matched']);
                } else {
                    promise = when.resolve({});
                }

                return promise;
            })
            .then(function() {
                next();
            })
            .catch(function(errors) {
                res.json({
                    status: 'error',
                    errors
                });
            });
    },
    function commit(req, res) {
        const {track_id, text, retpath} = req.body;

        req.api
            .userApproveCommit({
                track_id,
                text,
                retpath
            })
            .then(function(response) {
                res.json(response.body);
            })
            .catch(function(errors) {
                res.json({
                    status: 'error',
                    errors
                });
            });
    }
]);

exports.router = {
    userapprovePage,
    userapproveAjax
};
