const errorHandler = (req, res) => (errors) =>
    res.json({status: 'error', code: errors && errors[0] ? errors[0] : 'webauthn.error'});

const okHandler = (req, res) => ({body = {}}) => res.json(body);

module.exports.webauthnRegSubmit = (req, res) =>
    req.api
        .webauthnRegSubmit(req.body.track_id)
        .then(({body = {}}) =>
            res.json({
                status: 'ok',
                makeCredOptions: body.make_credential_options || {}
            })
        )
        .catch(errorHandler(req, res));

module.exports.webauthnRegCommit = (req, res) => {
    const {track_id: trackId, client, attestation, deviceName} = req.body;

    return req.api
        .webauthnRegCommit({
            track_id: trackId,
            origin: `https://${req.hostname}`,
            client_data: client,
            attestation_object: attestation,
            device_name: deviceName
        })
        .then(() => res.json({status: 'ok'}))
        .catch(errorHandler(req, res));
};

module.exports.webauthnList = (req, res) =>
    req.api
        .webauthnList(req.body.track_id)
        .then(({body: {webauthn_credentials: credentials = []} = {}}) =>
            res.json({
                status: 'ok',
                list: credentials.map((i) => ({
                    browser: i.browser_name,
                    created: i.created_at,
                    deviceName: i.device_name,
                    extId: i.external_id,
                    os: i.os_family_name,
                    isSuggested: i.is_suggested,
                    deviceType: i.is_device_tablet ? 'tablet' : i.is_device_mobile ? 'mobile' : 'desktop'
                }))
            })
        )
        .catch(errorHandler(req, res));

module.exports.webauthnRemove = (req, res) => {
    const {track_id: trackId, extId} = req.body;

    return req.api
        .webauthnRemove({
            track_id: trackId,
            credential_external_id: extId
        })
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));
};

module.exports.webauthnAuthSubmit = (req, res) => {
    const {track_id: trackId, extId} = req.body;

    return req.api
        .webauthnAuthSubmit({
            track_id: trackId,
            credential_external_id: extId
        })
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));
};

module.exports.webauthnAuthCommit = (req, res) => {
    const {track_id: trackId, extId, client, auth, signature} = req.body;

    return req.api
        .webauthnAuthCommit({
            track_id: trackId,
            credential_external_id: extId,
            origin: `https://${req.hostname}`,
            client_data: client,
            auth_data: auth,
            signature
        })
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));
};
