'use strict';

var express = require('express');
var when = require('when');
var crypto = require('crypto');
var router = express.Router();
var assign = require('lodash').assign;
var get = require('lodash').get;
var set = require('lodash').set;
var apiSetup = require('./common/apiSetup');
var config = require('../configs/current');

var ROBOTNESS_LIMIT = 0.95;
var YSA_PERCENT = 50;

function getEsecId(namespace) {
    return function(req, res, next) {
        res.locals.ysa = {
            namespace: namespace
        };

        req.api
            .ysaGetEsecId(namespace)
            .then(function(result) {
                var md5Hash = crypto.createHash('md5');
                var esecId = result.body.esec_id;

                md5Hash.update(namespace);
                res.locals.ysa.esecId = esecId;
                res.locals.ysa.static = config.paths.ysa.static;
                res.locals.ysa.namespaceMd5 = md5Hash.digest('hex');
                next();
            })
            .catch(function() {
                next();
            });
    };
}

function writeUserEnv(req, res, next) {
    var ysa = res.locals.ysa;
    var data = {
        namespace: ysa.namespace,
        esecId: ysa.esecId,
        events: [
            {
                name: 'env',
                ts: Math.floor(Date.now() / 1000),
                payload: {
                    track_id: res.locals.track,
                    ip: req.headers['x-real-ip'],
                    cookies: req.cookies,
                    yandexuid: res._yandexuid
                }
            }
        ]
    };

    req.api
        .ysaWriteUserEvent(data)
        .then(function() {
            next();
        })
        .catch(function() {
            next();
        });
}

function checkRobot(req, res, next) {
    const handlers = [req.api.captchaCheckStatus()];
    const ysa = {
        namespace: req.body.ysaNamespace,
        esecId: req.body.ysaEsecId
    };

    if (ysa.namespace && ysa.esecId) {
        handlers.push(req.api.ysaGetRobotness(ysa));
    }

    return when
        .all(handlers)
        .then(function(results) {
            const captchaStatus = results[0].body;
            const robotness = results[1] && results[1].body;

            if (!robotness || robotness.robotness > ROBOTNESS_LIMIT) {
                // maybe change to robotness_1 in the future
                if (!captchaStatus.is_recognized && !captchaStatus.is_checked) {
                    return res.json({
                        status: 'error',
                        errors: ['captcha.required']
                    });
                }

                if (!captchaStatus.is_recognized && captchaStatus.is_checked) {
                    if (ysa.namespace && ysa.esecId) {
                        req.api.ysaWriteUserEvent(
                            assign({}, ysa, {
                                events: [
                                    {
                                        name: 'mark',
                                        ts: Math.floor(Date.now() / 1000),
                                        payload: {
                                            value: 1
                                        }
                                    }
                                ]
                            })
                        );
                    }

                    return res.json({
                        status: 'error',
                        errors: ['captcha.required']
                    });
                }

                if (ysa.namespace && ysa.esecId) {
                    req.api.ysaWriteUserEvent(
                        assign({}, ysa, {
                            events: [
                                {
                                    name: 'mark',
                                    ts: Math.floor(Date.now() / 1000),
                                    payload: {
                                        value: 0
                                    }
                                }
                            ]
                        })
                    );
                }

                return next();
            }

            return next();
        })
        .catch(function(errors) {
            if (!(errors instanceof Array)) {
                errors = ['internal'];
            }

            res.json({
                status: 'error',
                errors: errors
            });
        });
}

function writeUserEvent(req, res) {
    var data = {
        namespace: req.body.ysaNamespace,
        esecId: req.body.ysaEsecId
    };
    var promise;

    try {
        data.events = JSON.parse(req.body.events);
    } catch (e) {
        return res.json({
            status: 'error',
            errors: ['internal']
        });
    }
    if (get(data, 'events[0].name', null) === 'provide-phone-number') {
        promise = req.api.params('country').then(function(result) {
            set(data, 'events[0].payload.country', result.body.country[0]);
        });
    } else {
        promise = when.resolve();
    }

    return promise
        .then(function() {
            return req.api.ysaWriteUserEvent(data);
        })
        .then(function(result) {
            res.json(result.body);
        })
        .catch(function(errors) {
            if (!(errors instanceof Array)) {
                errors = ['internal'];
            }

            res.json({
                status: 'error',
                errors: errors
            });
        });
}

router.post('/event', apiSetup, writeUserEvent);

module.exports.getEsecId = getEsecId;
module.exports.writeUserEnv = writeUserEnv;
module.exports.writeUserEvent = writeUserEvent;
module.exports.checkRobot = checkRobot;
module.exports.router = router;
module.exports.YSA_PERCENT = YSA_PERCENT;
