const path = require('path');
const argv = require('yargs').argv;
const webpack = require('webpack');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const MomentLocalesPlugin = require('moment-locales-webpack-plugin');
const TerserJSPlugin = require('terser-webpack-plugin');
const OptimizeCSSAssetsPlugin = require('optimize-css-assets-webpack-plugin');
const CircularDependencyPlugin = require('circular-dependency-plugin');
const CleanUpStatsPlugin = require('./clean-up-stats-plugin');
const LoadablePlugin = require('@loadable/webpack-plugin');
const resolveConfig = require('./webpack.paths');
const I18n = require('./i18n-webpack');
const locs = require('../loc/locs.json');
const packageJSON = require('../package.json');

const bundleName = argv.bundle;
const file = argv.file;
const ENV = argv.env || 'production';
const langs = argv.langs.split(' ');
const publicPath = argv.publicPath;
const {version} = packageJSON;
const sourceMap = true;

const IS_AUTH_PAGE = bundleName === 'react.authv2';
const IS_FAMILY_LAZY_PAGE = bundleName === 'react.family-lazy';
const configs = [];

langs.forEach((lang) => {
    const config = {
        name: 'clientside',
        target: 'web',
        profile: true,
        context: path.resolve(__dirname, '..'),
        mode: ENV,
        entry: {
            [bundleName]: [
                'regenerator-runtime',
                'core-js/features/array/find',
                'core-js/features/array/find-index',
                'core-js/features/array/for-each',
                'core-js/features/array/includes',
                'core-js/features/array/map',
                'core-js/features/array/reduce',
                'core-js/features/array/every',
                'core-js/features/array/some',
                'core-js/features/array/fill',
                'core-js/features/object/values',
                'core-js/features/object/entries',
                'core-js/features/object/assign',
                'core-js/features/string/includes',
                'core-js/es/map',
                'core-js/es/set',
                'core-js/es/promise',
                'raf/polyfill',
                'core-js/features/string/starts-with',
                'resize-observer-polyfill',
                file
            ]
        },
        output: {
            path: path.resolve(__dirname, '..', 'public/js'),
            filename: `${bundleName}.react.${lang}.js`,
            chunkFilename: `chunk-${bundleName}-[name].${lang}.js`,
            publicPath: `${publicPath}/js/`
        },
        module: {
            rules: [
                {
                    test: /\.json$/,
                    use: ['json-loader'],
                    type: 'javascript/auto'
                },
                {
                    test: /\.jsx?$/,
                    use: [
                        {
                            loader: 'babel-loader',
                            options: {
                                cacheDirectory: true,
                                caller: {target: 'clientside', name: 'babel-loader'}
                            }
                        }
                    ],
                    include: /(pages|blocks|plibs|node_modules)/,
                    exclude: /(plibs\/greed)/
                },
                {
                    include: /node_modules/,
                    use: 'base64-inline-loader',
                    test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/
                },
                {
                    test: /\.styl$/,
                    use: [
                        MiniCssExtractPlugin.loader,
                        {
                            loader: 'css-loader',
                            options: {
                                sourceMap,
                                url: false
                            }
                        },
                        {
                            loader: 'postcss-loader',
                            options: {sourceMap}
                        },
                        {
                            loader: 'stylus-loader',
                            options: {sourceMap}
                        }
                    ]
                },
                {
                    test: /\.css$/,
                    include: /node_modules/,
                    use: [
                        MiniCssExtractPlugin.loader,
                        {
                            loader: 'css-loader',
                            options: {sourceMap}
                        },
                        {
                            loader: 'postcss-loader',
                            options: {sourceMap}
                        }
                    ]
                }
            ],
            noParse: [/lib\/geo\//, /plibs\/greed\/greed/]
        },
        resolve: resolveConfig.resolve,
        plugins: [
            new webpack.DefinePlugin({
                'process.env.NODE_ENV': ENV === 'development' ? '"development"' : '"production"',
                'typeof window': 'object',
                'process.env.BROWSER': true,
                IS_AUTH_PAGE
            }),
            new MomentLocalesPlugin({
                localesToKeep: ['en', 'ru', 'uk', 'tr', 'fr', 'fi']
            }),
            new webpack.HashedModuleIdsPlugin({
                hashFunction: 'sha1',
                hashDigest: 'base64',
                hashDigestLength: 20
            }),
            new I18n(locs[lang], null, lang, bundleName),
            new CleanUpStatsPlugin('mini-css-extract-plugin'),
            new MiniCssExtractPlugin({
                filename: `../css/${bundleName}.cij.css`,
                chunkFilename: `../css/[name].${bundleName}.cij.css`
            }),
            new CircularDependencyPlugin({
                exclude: /node_modules/,
                failOnError: true,
                allowAsyncCycles: false,
                cwd: process.cwd()
            })
        ],
        optimization: {
            minimizer:
                ENV === 'development'
                    ? []
                    : [
                          new TerserJSPlugin({
                              parallel: false,
                              extractComments: false,
                              sourceMap: true
                          }),
                          new OptimizeCSSAssetsPlugin({})
                      ],
            splitChunks: {
                minSize: 0,
                chunks: 'initial',
                cacheGroups: {
                    default: false,
                    vendors: {
                        test: /node_modules[\\/]((?!\.css).)*$/,
                        filename: `${bundleName}.vendors.${lang}.js`,
                        priority: -10
                    },
                    lego: {
                        test: /(plibs[/])?node_modules[\\/](?:lego-on-react|dheader|@yandex-lego)[\\/]((?!\.css).)*$/,
                        filename: `${bundleName}.lego.${lang}.js`
                    }
                }
            }
        },
        externals: {
            react: 'React',
            'react-dom': 'ReactDOM'
        },
        devtool: ENV === 'development' ? '#inline-source-map' : false
    };

    if (IS_AUTH_PAGE || IS_FAMILY_LAZY_PAGE) {
        config.plugins.push(
            new LoadablePlugin({
                writeToDisk: {
                    filename: path.resolve(__dirname, `../loadable/${lang}/${bundleName}`)
                }
            })
        );
    }

    if (ENV !== 'development') {
        config.plugins.push(
            new webpack.SourceMapDevToolPlugin({
                // eslint-disable-next-line max-len
                append: `\n//# sourceMappingURL=https://s3.mds.yandex.net/passport-source-maps/core/v${version}/[url]`,
                filename: '[file].map'
            })
        );
    }

    configs.push(config);
});

module.exports = configs;
