const path = require('path');
const argv = require('yargs').argv;
const webpack = require('webpack');
const MomentLocalesPlugin = require('moment-locales-webpack-plugin');
const CircularDependencyPlugin = require('circular-dependency-plugin');
const resolveConfig = require('./webpack.paths');
const I18n = require('./i18n-webpack');
const locs = require('../loc/locs.json');

const bundleName = argv.bundle;
const file = argv.file;
const ENV = argv.env || 'production';
const langs = argv.langs.split(' ');

const IS_AUTH_PAGE = bundleName === 'react.authv2';
const configs = [];

langs.forEach((lang) => {
    const config = {
        name: 'serverside',
        target: 'node',
        context: path.resolve(__dirname, '..'),
        mode: ENV,
        entry: {
            [bundleName]: file
        },
        output: {
            libraryTarget: 'commonjs2',
            filename: `${bundleName}.${lang}.js`,
            path: path.resolve(__dirname, '..', 'templates')
        },
        module: {
            rules: [
                {
                    test: /\.json$/,
                    use: ['json-loader'],
                    type: 'javascript/auto'
                },
                {
                    test: /\.jsx?$/,
                    use: [
                        {
                            loader: 'babel-loader',
                            options: {
                                cacheDirectory: true,
                                caller: {target: 'serverside', name: 'babel-loader'}
                            }
                        }
                    ],
                    include: /(pages|blocks|plibs)/
                },
                {
                    test: /\.js$/,
                    include: /pclientjs|broker|Metrika|plibs\/greed/,
                    loader: 'null-loader'
                },
                {
                    test: /\.styl$/,
                    loader: 'null-loader'
                },
                {
                    test: /\.css$/,
                    loader: 'null-loader'
                },
                {
                    include: /node_modules/,
                    use: 'base64-inline-loader',
                    test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/
                }
            ],
            noParse: [/lib\/geo\//]
        },
        resolve: resolveConfig.resolve,
        plugins: [
            new webpack.DefinePlugin({
                'process.env.NODE_ENV': ENV === 'development' ? '"development"' : '"production"',
                'process.env.BROWSER': false,
                IS_AUTH_PAGE
            }),
            new MomentLocalesPlugin({
                localesToKeep: ['en', 'ru', 'uk', 'tr', 'fr', 'fi']
            }),
            new I18n(locs[lang], null, lang, bundleName),
            new CircularDependencyPlugin({
                exclude: /node_modules/,
                failOnError: true,
                allowAsyncCycles: false,
                cwd: process.cwd()
            })
        ]
    };

    configs.push(config);
});

module.exports = configs;
