var fs = require('fs');
var path = require('path');
var crypto = require('crypto');

var data = {};
var cnt = 0;
var OPTIONS = {};

const extensions = {
    '.css': function(absPath) {
        return path.basename(absPath).split('.')[0].replace(/^_/, '');
    },
    '.js': function(absPath){
        return path.basename(absPath).split('.')[1];
    },
    '.yate': function(absPath){
        return path.basename(absPath).split('.')[1];
    }
};

function incCnt() {
    cnt++;
}

function decCnt() {
    cnt--;
    if (cnt === 0) {
        thatsAll();
    }
}

function addToResult(info, cb) {
    var newFileName = info.hash + info.ext;
    data[info.path] = {
        url: newFileName,
        size: info.size
    };

    var freezeFile = OPTIONS.freeze + '/' + newFileName;

    if (OPTIONS.copy) {
        copy(info.absPath, freezeFile, function() {
            cb();
        });

    } else if (OPTIONS.move) {
        fs.rename(info.absPath, freezeFile, function() {

            if (OPTIONS['move-with-symlink']) {
                var fileDir = path.dirname(info.absPath);
                var relativePath = path.relative(fileDir, freezeFile);
                fs.symlinkSync(relativePath, info.absPath);
            }

            cb();
        });

    } else {
        cb();
    }
}

function copy(src, dst, cb) {
    fs.createReadStream(src).pipe(fs.createWriteStream(dst).on('close', cb));
}

function md5(absPath, cb) {
    var shasum = crypto.createHash('md5');

    var s = fs.ReadStream(absPath);
    s.on('data', function(d) {
        shasum.update(d);
    });

    s.on('end', function() {
        var d = shasum.digest('hex');

        // считаем количество символов в файле, чтобы потом проверять в JS
        var file = fs.readFileSync(absPath, 'utf-8');
        cb(d, file.length);
    });
}

/**
 * Обрабатывает файл из папки. Считает его md5 и добавляет в результат.
 * @param absPath
 * @param basePath
 */
function processFile(absPath, basePath) {
    var extension = path.extname(absPath);
    var processFn = extensions[extension];
    if (processFn) {
        incCnt();
        md5(absPath, function(hash, size) {
            var relativePath = path.relative(basePath, absPath);
            var flag = processFn(relativePath);
            addToResult({
                ext: extension,
                flag: flag,
                path: relativePath,
                hash: hash,
                absPath: absPath,
                size: size
            }, function() {
                decCnt();
            });
        });
    }
}

/**
 * Рекурсивно читает содержимое папки.
 * @param dir
 * @param basePath
 */
function readDir(dir, basePath) {
    incCnt();
    fs.readdirSync(dir).forEach(function(file) {
        file = path.resolve(dir, file);
        var stat = fs.statSync(file);
        if (stat.isFile()) {
            processFile(file, basePath);

        } else if (stat.isDirectory()) {
            readDir(file, basePath);
        }
    });
    decCnt();
}

function error() {
    var args = Array.prototype.slice.call(arguments, 0);
    args.unshift('  *** ERROR *** ');
    console.error.apply(console, args);
    printUsage();
}

/**
 * Проверяет существование папки.
 * @param dir
 * @return {*}
 */
function checkIfDirExists(dir) {
    if (!fs.existsSync(dir)) {
        error('Director "'+ dir +'" is not exists!');
    }
}

/**
 * Выводит подготовленный JSON в stdout
 */
function thatsAll() {
    // скобки нужны, чтобы работал System.include
    process.stdout.write('(' + JSON.stringify(data) + ')')
}

function printUsage(){
    console.error('\
        \n\
        d2-freeze-static builds hashes for css/js/yate files as JSON objects\n\
          --hubs - path to static files\n\
          --freeze - path to freeze path\n\
          --copy - flag indicates that files should be copied to freeze path. Default: false\n\
          --move - flag indicates that files should be moved to freeze path. Default: false\n\
          --move-with-symlink - Works with --move flag only! Flag indicates that symlinks will be created for moved files. Default: false\n\
        Example: d2-freeze-static --hubs mydir/hubs --hubs mydir/another_hubs --freeze path/to/freeze > freeze.json\n\
');
    process.exit(1);
}

module.exports = function(options) {
    if (!options.hubs) {
        error('Give me dir!')
    }
    if (!options.freeze) {
        error('Give me freezePath!')
    }

    OPTIONS = options;
    checkIfDirExists(options.freeze);

    options.hubs.forEach(function(hub) {
        var basePath = path.dirname(hub);
        var stat = fs.statSync(hub);
        if (stat.isFile()) {
            // берем dirname еще раз, чтобы оставить папку
            processFile(hub, path.dirname(basePath));

        } else if (stat.isDirectory()) {
            readDir(hub, basePath);
        }
    });
};
