/**
 * Регулярка для разбива строки "loc_key    loc_value" на [key,value]
 * @type {RegExp}
 */
const LOC_STRING_SPLITTER = /^\s*(.*?)\s+(.*)/;

var UglifyJS = require("uglify-js");

/**
 * @param {String} [namespace=Jane] Неймспейс для функций <namespace>.i18n.convert
 * @constructor
 */
function LocParser(namespace) {
    this.namespace = namespace || 'Jane';
    this.KEYS = {};
    this._AST_KEYS = {};
    this.ORIGINAl_KEYS = {};
}

LocParser.prototype = {

    /**
     * Возвращает ключи.
     * @param [keys] Если передан, то записыват ключи.
     * @return {Object}
     */
    keys: function(keys) {
        if (keys) {
            this.KEYS = keys;
        }

        return this.KEYS;
    },

    getKeyAST: function(key) {
        if (key in this._AST_KEYS) {
            return this._AST_KEYS[key];

        } else if (key in this.ORIGINAl_KEYS) {
            this._AST_KEYS[key] = this._parseKeyAST(key);
            return this._AST_KEYS[key];

        } else {
            throw 'Key "' + key + '" is undefined';
        }
    },

    stringify: function() {
        var props = [];
        for (var key in this.ORIGINAl_KEYS) {
            var keyAST = this.getKeyAST(key);

            var args = [];
            for (var arg in keyAST.params) {
                args.push(new UglifyJS.AST_SymbolRef({name: '_' + arg}));
            }
            var ast;
            if (args.length) {
                ast = new UglifyJS.AST_Function({
                    argnames: args,
                    body: [
                        new UglifyJS.AST_Return({
                            value: keyAST.ast
                        })
                    ]
                });

            } else {
                ast = keyAST.ast;
            }

            props.push(
                new UglifyJS.AST_ObjectKeyVal({
                    key: key,
                    value: ast
                })
            );
        }

        return new UglifyJS.AST_Object({properties: props}).print_to_string({
            quote_keys: true
        });
    },

    /**
     * Парсит файл с локализационными строчками
     * @param {String} content
     */
    parseFile: function (content) {
        content
            .split('\n')
            .filter(function(str) {
                // не пустая строка и не комментарий
                return str && str.charAt(0) != '#'
            })
            .forEach(function(str) {
                var match = str.match(LOC_STRING_SPLITTER);
                if (match && match.length === 3 && match[1] && match[2]) {
                    var key = match[1].trim();
                    var value = match[2].trim();
                    if (key in this.ORIGINAl_KEYS) {
                        throw 'Duplicate key "' + key + '" declaration';
                    }

                    value = value
                        // удаляем ненужное экранирование \%
                        .replace(/\\%/g, '%')
                        // удаляем ненужное экранирование \)
                        .replace(/\\\)/g, ')')
                        // удаляем ненужное экранирование \(
                        .replace(/\\\(/g, '(');

                    //сохраняем оригинальные ключи для того, чтобы обрабатыват %convert(%@key | 1)
                    this.ORIGINAl_KEYS[key] = value;

                } else {
                    throw 'Invalid string "' + str + '"';
                }
            }.bind(this));
    },

    _parseKeyAST: function(key) {
        var value = this.ORIGINAl_KEYS[key];
        var processResult = this._value2Array(value).map(this._processStringAST.bind(this));
        if (processResult.length == 1) {
            return processResult[0];

        } else {
            var params = {};
            var elements = [];
            processResult.forEach(function(item){
                for (var i in item.params) {
                    if (i in params) {
                        params[i] = params[i].concat(item.params[i]);
                    } else {
                        params[i] = item.params[i];
                    }
                }
                elements.push(item.ast);
            });

            return {
                params: params,
                ast: new UglifyJS.AST_Array({
                    elements: elements
                })
            };
        }
    },

    /**
     * Парсит массив в строке
     * @example
     * "str" -> ["str"]
     * "[foo | bar | foo1]" -> ["foo", "bar", "foo1"]
     * @param {String} val
     * @return {Array}
     * @private
     */
    _value2Array: function(val) {
        if (/^\[.*\]$/.test(val)) {
            val = val.substring(1, val.length - 1);
            return val.split('|').map(function(val) {
                return val.trim()
            });
        }
        return [val];
    },

    _processStringAST: function(str) {
        var params = {};
        var result = [];
        var lastIndex = 0;

        /**
         * Регулярка для нахождения параметров (%1), других строк (%some_string), функция %convert и %index.
         * @type {RegExp}
         */
        var PARSE_PARAMS = /%((convert|index)\(%(.*?)\s*\|\s*%(\d+)\s*\)|(plural|if)\(\s*%(\d+)\s*\|\s*(.*?)\s*\|\s*(.*?)\s*\)|\d+|[а-яА-Яa-zA-Z0-9][а-яА-Яa-zA-Z0-9_-]*[а-яА-Яa-zA-Z0-9])/g;

        var match;
        while (match = PARSE_PARAMS.exec(str)) {
            var keyAST;
            var leftString = str.substring(lastIndex, match.index);
            if (leftString) {
                result.push(new UglifyJS.AST_String({value: leftString}));
            }
            lastIndex = PARSE_PARAMS.lastIndex;

            if (match[5] == 'plural' || match[5] == 'if') {
                argNode = new UglifyJS.AST_SymbolRef({name: '_' + match[6]});
                addParam(match[6], argNode);
                result.push(new UglifyJS.AST_Call({
                    args: [
                        argNode,
                        new UglifyJS.AST_String({value: match[7]}),
                        new UglifyJS.AST_String({value: match[8]})
                    ],
                    // "if" - зарезервированное слово, поэтому вместо AST_Dot (i18n.if), надо использовать AST_Sub (i18n["if"])
                    expression: new UglifyJS.AST_Sub({
                        property: new UglifyJS.AST_String({value: match[5]}),
                        expression: new UglifyJS.AST_Dot({
                            property: 'i18n',
                            expression: new UglifyJS.AST_SymbolRef({
                                name: this.namespace
                            })
                        })
                    })
                }));
                continue;

            } else if (match[2]) { // convert/index
                try {
                    keyAST = this.getKeyAST(match[3]);
                } catch (e) {
                    console.error('Cant parse key "' + match[3] + '" in token "' + match[0] + '"');
                    console.error('Full string', str);
                    throw e;
                }

                argNode = new UglifyJS.AST_SymbolRef({name: '_' + match[4]});

                // копируем ast, чтобы правильно преобразовывать такой вариант
                // "Поставленая метка %1 на %convert(%@N_файлов | %2)"
                // в N_файлов будет %1, но значения туда должны попасть из %2
                var keyASTCopy = this.deepASTCopy(keyAST.ast, [argNode]);

                addParam(match[4], argNode);
                addParams(keyAST.params);

                result.push(new UglifyJS.AST_Call({
                    args: [
                        keyASTCopy,
                        argNode
                    ],
                    expression: new UglifyJS.AST_Dot({
                        property: match[2],
                        expression: new UglifyJS.AST_Dot({
                            property: 'i18n',
                            expression: new UglifyJS.AST_SymbolRef({
                                name: this.namespace
                            })
                        })
                    })
                }));
                continue;
            }

            var key = match[1];

            // игноруем url-encode
            if (/^[A-Z0-9]{2}$/.test(key)) {
                continue;
            }

            // параметр %1
            if (/^\d+$/.test(key)) {
                var argNode = new UglifyJS.AST_SymbolRef({name: '_' + key});
                addParam(key, argNode);
                result.push(argNode);

            } else {
                // ссылка на другую строку
                try {
                    keyAST = this.getKeyAST(key);
                } catch (e) {
                    console.error('Cant parse key "' + key + '" in token "' + match[0] + '"');
                    console.error('Full string', str);
                    throw e;
                }
                addParams(keyAST.params);
                result.push(keyAST.ast);
            }
        }

        leftString = str.substring(lastIndex);
        if (leftString) {
            result.push(new UglifyJS.AST_String({value: leftString}));
        }

        if (result.length == 1) {
            return {
                ast: result[0],
                params: params
            };

        } else {
            // добавляем выражения с конца
            var ast = new UglifyJS.AST_Binary({
                operator: '+',
                left: result.shift(),
                right: result.shift()
            });
            while(result.length > 0) {
                ast = new UglifyJS.AST_Binary({
                    operator: '+',
                    right: result.shift(),
                    left: ast
                });
            }

            return {
                ast: ast,
                params: params
            };
        }

        function addParam(key, node) {
            if (key in params) {
                params[key].push(node);
            } else {
                params[key] = [ node ];
            }
        }

        function addParams(paramsToMerge) {
            for (var i in paramsToMerge) {
                if (i in params) {
                    params[i] = params[i].concat(paramsToMerge[i])
                } else {
                    params[i] = paramsToMerge[i];
                }
            }
        }
    },

    /**
     * Регулярка для тестирования, того что название переменной - параметр.
     * @constant
     * @private
     */
    _REGEXP_LOC_PARAM: /_(\d+)/,

    /**
     * Создает копию AST, заменяя переменные "_\d+" на переменные из params
     * @param {String} astCode
     * @param {Array} params
     */
    deepASTCopy: function(astCode, params) {
        return astCode.transform(
            new UglifyJS.TreeTransformer(null, function(node){
                var paramMatch;
                if (node.TYPE == 'SymbolRef' && (paramMatch = node.name.match(this._REGEXP_LOC_PARAM))) {
                    var paramNum = paramMatch[1] - 1;
                    if (paramNum in params) {
                        return params[paramNum].clone();

                    } else {
                        throw 'Missed required argument ' + paramNum;
                    }
                }
                return node;
            }.bind(this))
        );
    }
};

module.exports = LocParser;
