#!/usr/bin/env node

var fs = require('fs');
var path = require('path');
var LocParser = require('./d2-localize-loc-parser.js');
var LocProcessor = require('./d2-localize-processor.js');

/**
 * Кодировка файлов для чтения.
 * @type {String}
 */
const ENCODING = 'utf8';

function localizeFile(fileName, opts, processor) {
    var extension = processor || path.extname(fileName);
    var res = localizeString(
        readFile(fileName),
        extension,
        opts
    );
    if (typeof res == 'string') {
        return res;
    }
    return require('..').uglify(res, opts).replace(/;$/, '');
}

function localizeString(str, extension, opts) {
    var parser = createParser(opts['locale'], opts['path'], opts['namespace']);

    var entitites = false;
    if (opts['entity']) {
        entitites = {};
        opts['entity'].forEach(function(file) {
            var data = readEntityAndParse(file);
            for (var entity in data) {
                entitites[entity] = data[entity];
            }
        });
    }

    // создаем локализатор
    var localizer = new LocProcessor(parser, str, extension, {
        'entity': entitites,
        'ignore-undefined-entity': opts['ignore-undefined-entity']
    });
    // процессим
    return localizer.process();
}

function dump(opts) {
    var parser = createParser(opts['locale'], opts['path'], opts['namespace']);

    // дампим все в json
    return parser.stringify();
}

function createParser(locale, path, namespace) {
    // создаем парсер локализаций
    var parser = new LocParser(namespace);
    // парсим локализации
    readLocFiles(path, locale, parser);

    return parser;
}

/**
 * Читает файл и вызывает callback
 * @param filename
 */
function readFile(filename) {
    try {
        var content = fs.readFileSync(filename, ENCODING);
    } catch (err) {
        error('Fail to read file "' + filename + '"');
        error(err.message);
        process.exit(1);
    }
    return content;
}

function readLocFiles(locPaths, locale, parser) {
    /**
     * Регулярка для поиска файлов с локализацией.
     * /\.ru\.loc$/
     * @type {RegExp}
     */
    var rFILE_NAME = new RegExp('\\.' + locale + '\\.loc$');

//    console.error('cwd', );

    // читаем локализации
    locPaths.forEach(function(dirName) {
        var absPath = fs.realpathSync(dirName);
        var stat = fs.statSync(absPath);
        // если передали файл
        if (stat.isFile()) {
            dirContent = [absPath];
        } else {
            try {
                var dirContent = fs.readdirSync(absPath);
            } catch (err) {
                if (err) {
                    error('Fail to read dir "' + dirName + '"("' + absPath + '")');
                    error(err);
                    process.exit(1);
                }
            }
        }

        dirContent = dirContent
            .map(function(file) {
                // преобразуем в абсолютный путь
                return path.resolve(dirName, file)
            })
            .filter(function(fileName) {
                // фильтруем только файлы с текущей локалью
                return rFILE_NAME.test(fileName)
            });

        // читаем и парсим каждый файл
        dirContent.forEach(function(file) {
            parser.parseFile(readFile(file), file);
        });
    });
}

function readEntityAndParse(file) {
    var content = readFile(file);
    try {
        var data = JSON.parse(content);
    } catch (e) {
        console.error(e);
        process.exit(1);
    }
    return data;
}

function error() {
    var args = Array.prototype.slice.call(arguments, 0);
    args.unshift('  *** ERROR *** ');
    console.error.apply(console, args);
}

module.exports = {
    localizeFile: localizeFile,
    localizeString: localizeString,
    dump: dump
};
