var fs = require('fs');
var path = require('path');
var http = require('http');
var url_ = require('url');
var diffable = require('./vcdiff.js');

var OPTIONS = {};
var STATIC_URL = 'http://yandex.st/mail';
var VERSIONS = [];

const pid2dir = {
    'jane': 'neo2',
    'mail': 'mail'
};

const extensions = {
    '.css': function(basename) {
        if (basename.indexOf('.ie.') > -1) {
            return false;
        } else if (basename.indexOf('_') == 0) {
            return basename.slice(1);
        } else {
            return basename;
        }
    },
    '.js': function(basename, ext, pid) {
        return getKey(basename, ext, pid, true);
    },
    '.yate': function(basename, ext, pid) {
        return getKey(basename, ext, pid, true);
    }
};

function get(url, cb, retry) {
    http.get(url_.parse(url), function(res) {
        var body = '';
        res.setEncoding('utf8');

        res.on('data', function(chunk) {
            body += chunk;
        });

        res.on('end', function() {
            cb(body);
        });
    }).on('error', function(err) {
        if (retry) {
            get(url, cb, retry--);
        } else {
            console.error(err);
            process.exit(1);
        }
    });
}

function generatePatch(oldStr, newStr) {
    var vcd = new diffable.Vcdiff();
    vcd.blockSize = 10;
    return vcd.encode(oldStr, newStr);
}

/**
 * Вычисляет чексумму данных по алгоритму Флетчера.
 * @param {String} data
 * @return {Number}
 */
function calcFletcherSum(data) {
    var length = data.length, i = 0, sum1 = 0xFF, sum2 = 0xFF;
    while (length) {
        var tlen = length > 21 ? 21 : length;
        length -= tlen;
        do {
            var ch = data.charCodeAt(i++);
            if (ch > 255) {
                var ch2 = ch >> 8;
                ch &= 0xFF;
                ch ^= ch2;
            }
            sum1 += ch;
            sum2 += sum1;
        } while (--tlen);
        sum1 = (sum1 & 0xFF) + (sum1 >> 8);
        sum2 = (sum2 & 0xFF) + (sum2 >> 8);
    }
    var result = (((sum1 & 0xFF) + (sum1 >> 8)) << 8) | ((sum2 & 0xFF) + (sum2 >> 8));
    return result == 0xFFFF ? 0 : result;
}

/**
 * Проверяет существование папки.
 * @param dir
 * @return {*}
 */
function checkIfDirExists(dir) {
    if (!fs.existsSync(dir)) {
        error('Director "'+ dir +'" is not exists!');
    }
}

function printUsage(){
    console.error('\
        \n\
        d2-make-patch builds patches for css/js/yate files as JSON objects\n\
          --versions - path to old versions JSON file\n\
          --project - project name\n\
          --resource - path to css/js/yate file\n\
          --diff - path to output diff dir\n\
        Example: d2-make-patch --diff path/to/diff --project jane --versions _versions.json --resource hubs/jane/_jane.ru.js\n\
');
    process.exit(1);
}

function error() {
    var args = Array.prototype.slice.call(arguments, 0);
    args.unshift('  *** ERROR *** ');
    console.error.apply(console, args);
    printUsage();
}

function makePatches() {
    VERSIONS = require(OPTIONS.versions).versions;
    OPTIONS.resource.forEach(function(res) {
        var ext = path.extname(res);
        var stat = fs.statSync(res);

        if (stat && extensions[ext]) {
            writePatchFile(res, ext);
        }
    });
}

function writePatchFile(res, ext) {
    var pid = OPTIONS.project;
    var basename = path.basename(res);
    var key = getKey(basename, ext, pid);
    var stat = fs.statSync(res);

    if (stat) {
        var newFile = fs.readFileSync(res, 'utf-8');
        var name = extensions[ext](basename, ext, pid);

        if (!name) return;

        VERSIONS.forEach(function(version) {
            var url = [STATIC_URL, pid2dir[pid], version, 'hubs', pid];
            if (ext == '.css') {
                url.push(pid);
            }
            url.push(basename);
            url = url.join('/');

            get(url, function(oldFile) {
                var fileName = version + '@' + name + '.json';
                var patch = generatePatch(oldFile, newFile);
                var newFileLen = newFile.length;
                var data = { k: key };
                var newFileLen = newFile.length;

                if (JSON.stringify(patch).length <= newFileLen * 0.3) {
                    data.p = patch;
                    data.s = calcFletcherSum(newFile.substring(0, 100)) + calcFletcherSum(newFile.substring(newFileLen - 100));
                }
                fs.writeFileSync(path.join(OPTIONS.diff, OPTIONS.project, fileName), JSON.stringify(data), 'utf-8');
            }, 3);
        });
    }
}

function getKey(basename, ext, pid, withLang) {
    var lang = basename.match(/\.(..)\./);
    var hub = basename.substr(1).replace(ext, '');
    var key;

    if (lang) {
        lang = lang[0].slice(0, 3);
        hub = hub.replace(lang, '');
    }

    if (ext == '.css' || pid == hub) {
        key = pid;
        if (pid != 'jane') {
            key += '.' + pid;
        }
    } else {
        key = pid + '.' + hub;
    }

    if (lang && withLang) {
        key += lang;
    }
    key += ext;

    return key;
}

module.exports = function(options) {
    if (!options.versions) {
        error('Give me path to old versions file!');
    }

    if (!options.project) {
        error('Give me project name!');
    }

    if (!options.resource) {
        error('Give me resourcePath!');
    }

    if (!options.diff) {
        error('Give me diffPath!');
    }

    OPTIONS = options;
    checkIfDirExists(options.diff);

    var projectDir = path.join(options.diff, options.project);
    if (!fs.existsSync(projectDir)) {
        fs.mkdirSync(projectDir);
    }
    makePatches();
};
