const queries = require('./queries');

module.exports = require('inherit')(
    {
        __constructor: function(lodId, dao, userVariables) {
            require('assert')(
                dao instanceof require('pdao/Http'),
                'Data access object should be an instance of Http DAO'
            );

            /**
             * @type DAO
             */
            this._userVariables = userVariables || {};
            this._dao = dao;
            this._logger = new (require('plog'))(lodId, 'api', 'afisha');
        },

        _getHandle: function(queryName) {
            return '/graphql?query_name=' + queryName;
        },

        _makeRequest: function(query, queryVariables, userVariables, queryName) {
            const options = Object.assign({}, this._userVariables, userVariables || {});
            const input = {
                query,
                variables: queryVariables || {},
                city: options.city,
                geoid: options.geoid,
                lat: options.lat,
                lon: options.lon,
                lang: options.lang,
                region: options.region,
                userIp: options.userIp,
                yandexuid: options.yandexuid,
                userAgent: options.userAgent,
                ticketsClientKey: options.ticketsClientKey
            };

            return this._dao.call('post', this._getHandle(queryName), input).then(function(response) {
                return response;
            });
        },

        getOrder: function(orderID, userVariables, queryName) {
            queryName = queryName || 'services_order';

            return this._makeRequest(queries.order, {id: orderID}, userVariables, queryName);
        },

        getOrders: function(pageLimit, pageOffset, userVariables, queryName) {
            queryName = queryName || 'services_orders';

            return this._makeRequest(queries.orders, {pageLimit, pageOffset}, userVariables, queryName);
        },

        getFavoriteEvents: function(pageLimit, pageOffset, userVariables, queryName) {
            queryName = queryName || 'services_favorites';

            return this._makeRequest(queries.favoriteEvents, {pageLimit, pageOffset}, userVariables, queryName);
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: require('inherit')(require('../error'), {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor: function(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains: function(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode: function(callback) {
                this._errors.forEach(callback);
            }
        })
    }
);
