const PLog = require('plog');
const assert = require('assert');
const inherit = require('inherit');
const pdaoHttp = require('../../pdao/Http');
const error = require('../error');
const showcase = require('./showcase');
const {fromIntervals, getType} = require('./utils');

module.exports = inherit(
    {
        __constructor(lodId, dao) {
            assert(dao instanceof pdaoHttp, 'Data access object should be an instance of Http DAO');

            this._dao = dao;
            this._logger = new PLog(lodId, 'api', 'billing');
        },

        /**
         * Вытягивание данных об интервалах (подписках) пользователя
         * @see {@link https://wiki.yandex-team.ru/mb/intervals/ wiki} инфо по ручке
         * @param {string} uid идентификатор пользователя
         * @returns {Promise<Interval[]>} список текущих (активные + отмененные до даты отключения)
         * интервалов пользователя
         */
        getIntervals(uid) {
            return this._dao
                .call('get', '/billing/intervals', {
                    __uid: uid,
                    // Параметр start нужен для филтрации устаревших интервалов
                    start: new Date().toISOString()
                })
                .then(function(response) {
                    const {result = {}, error} = response;
                    const {intervals} = result;

                    return {
                        status: error ? 'error' : 'success',
                        intervals: fromIntervals(intervals)
                    };
                });
        },

        stopSubscription({uid, productId}) {
            return this._dao
                .call('post', '/account/stop-subscription', {
                    __uid: uid,
                    productId
                })
                .then((response) => response);
        },

        restoreSubscription({uid, productId, ip}) {
            return this._dao
                .call('post', '/subscription/resume-subscription', {
                    uid,
                    productId,
                    ip
                })
                .then((response) => {
                    let result = response.result;

                    if (result.status == null) {
                        return Promise.reject(result);
                    }

                    return result;
                });
        },

        getShowcase(countryId, exp) {
            const showcaseId = exp ? `${countryId}:${exp}` : countryId;

            return showcase[showcaseId] || showcase[countryId];
        },

        getNativeProducts({target, uid, ip, language}) {
            return this._dao
                .call('get', '/account/native-products', {
                    ip,
                    target,
                    language,
                    __uid: uid
                })
                .then((response) => response);
        },

        submitNativeOrder({productId, ip, uid, paymentMethodId, source = 'passport_profile', from, templateTag}) {
            return this._dao
                .call('post', '/account/submit-native-order', {
                    ip,
                    productId,
                    __uid: uid,
                    paymentMethodId,
                    source,
                    from,
                    templateTag
                })
                .then((response) => {
                    const {result} = response;

                    if (result.status.toLowerCase() !== 'success') {
                        return Promise.reject(result.status);
                    }

                    return result;
                });
        },

        orderInfo({uid, orderId}) {
            return this._dao
                .call('get', '/account/billing/order-info', {
                    __uid: uid,
                    orderId
                })
                .then((response) => {
                    return response.result;
                });
        },

        productInfo({productId}) {
            return this._dao
                .call('get', '/billing/product-info', {
                    productId
                })
                .then((response) => {
                    const result = response.result; // totally not same productInfo as in /intervals

                    result.type = getType({bundle: result.defaultFeatureBundle, family: result.hasFamily});

                    return result;
                });
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: inherit(error, {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode(callback) {
                this._errors.forEach(callback);
            }
        })
    }
);
