const _ = require('lodash');

const BUNDLES = {
    'basic-music': 'basic-music',
    'basic-plus': 'basic-plus',
    'kinopoisk-plus': 'kinopoisk-plus',
    'new-plus': 'new-plus',
    'basic-kinopoisk': 'basic-kinopoisk',
    'kp-amediateka': 'kp-amediateka',
    'disk-basic': 'disk-basic',
    'station-lease-plus-kinopoisk': 'station-lease-plus-kinopoisk',
    'station-lease-plus': 'station-lease-plus',
    'beru-basic': 'beru-basic',
    'kinopoisk-go': 'kinopoisk-go',
    'kp-super-plus': 'kp-super-plus'
};

const TYPES = {
    PLUS: 'plus',
    PLUS_FAMILY: 'family_plus',
    KP_AMEDIATEKA: 'kinopoiskhd_amediateka',
    DISK: 'disk_plus',
    KP: 'kinopoiskhd',
    MUSIC: 'music',
    BERU: 'beru_plus',
    STATION_LEASE: 'station_lease',
    NEW_PLUS: 'new-plus',
    NEW_PLUS_MULTI: 'new-plus-multi',
    NEW_AMEDIATEKA_MULTI: 'new-amediateka-multi',
    KINOPOISK_GO: 'kinopoisk-go',
    NEW_PLUS_MORE: 'new-plus-more',
    NEW_PLUS_MULTI_MORE: 'new-plus-multi-more'
};

const BASE_PLUS_TYPES = [
    TYPES.PLUS,
    TYPES.PLUS_FAMILY,
    TYPES.NEW_PLUS,
    TYPES.NEW_PLUS_MULTI,
    TYPES.NEW_AMEDIATEKA_MULTI,
    TYPES.NEW_PLUS_MORE,
    TYPES.NEW_PLUS_MULTI_MORE
];

const getType = (productInfo) => {
    const {bundle, family, brand} = productInfo;

    switch (bundle) {
        case BUNDLES['basic-music']:
            return TYPES.MUSIC;
        case BUNDLES['basic-plus']:
        case BUNDLES['kinopoisk-plus']:
            return family ? TYPES.PLUS_FAMILY : TYPES.PLUS;
        case BUNDLES['new-plus']:
            return family ? TYPES.NEW_PLUS_MULTI : TYPES.NEW_PLUS;
        case BUNDLES['basic-kinopoisk']:
            return TYPES.KP;
        case BUNDLES['kp-amediateka']:
            return family ? TYPES.NEW_AMEDIATEKA_MULTI : TYPES.KP_AMEDIATEKA;
        case BUNDLES['disk-basic']:
            return TYPES.DISK;
        case BUNDLES['station-lease-plus']:
        case BUNDLES['station-lease-plus-kinopoisk']:
            return TYPES.STATION_LEASE;
        case BUNDLES['beru-basic']:
            return TYPES.BERU;
        case BUNDLES['kinopoisk-go']:
            return TYPES.KINOPOISK_GO;
        case BUNDLES['kp-super-plus']:
            return family ? TYPES.NEW_PLUS_MULTI_MORE : TYPES.NEW_PLUS_MORE;
        default:
            return brand;
    }
};

const getTypeFromInterval = (interval = {}) => {
    if (!interval.orderInfo || !interval.orderInfo.productInfo) {
        return null;
    }

    return getType(interval.orderInfo.productInfo);
};

/**
 * Статус/cостояние интервала
 * @see {@link https://wiki.yandex-team.ru/users/smanakhova/Podpiski-v-Pasporte-logika-otobrazhenija/#teksty  Статусы}
 * @param {object} interval - интервал с ручки биллинга /internal_api/intervals/?__uid=%userId%
 * @returns {string} статус подписки:
 * 'active' - подписка активна,
 * 'stopped' - подписка отменена,
 * 'in-grace' - не получается списать деньги за подписку,
 * 'inactive-promo' - неактивный подарочный код,
 * 'days-stock' - отложенные дни
 */
const getStatus = (interval = {}) => {
    if (new Date(interval.start) < new Date()) {
        if (interval.orderInfo.stopped) {
            return 'stopped';
        }
        const isInInterval = (date, {start, end}) => new Date(start) < new Date(date) && new Date(end) > new Date(date);

        if (interval.orderInfo.lastGrace && isInInterval(Date.now(), interval.orderInfo.lastGrace)) {
            return 'in-grace';
        }
        return 'active';
    }
    /* Если начало интервала в будущем, то это либо дни в запасе, либо неактивный промокод */
    return interval.orderInfo.orderType === 'promo-code' ? 'inactive-promo' : 'days-stock';
};

/**
 * Склеивание интервалов в один
 * @param intervals
 * @returns {Array}
 */
const collapseIntervals = (intervals = []) => {
    if (intervals.length <= 1) {
        return intervals;
    }

    const sortedIntervals = intervals.sort((a, b) => a.end >= b.end);
    const newInterval = sortedIntervals.pop();

    newInterval.start = sortedIntervals[0].start;

    return newInterval;
};

/** Декорация интервала
 * @param {Interval} interval - интервал
 * @returns {Subscription} Подписка (декоратор интервала)
 */
// eslint-disable-next-line max-params
const fromInterval = (interval, {promos, daysStocks, parentType} = {}) =>
    Object.assign(
        {},
        {
            start: interval.start,
            end: interval.end
        },
        interval.orderInfo,
        {
            status: getStatus(interval),
            type: getTypeFromInterval(interval),
            daysStocks,
            promos,
            parentType
        }
    );

const PROMO_TYPES = [
    'dummy',
    'gift-2-0',
    'promo-code',
    'mcdonalds-promo-code',
    'coca-cola-promo-code',
    'samsung-galaxy-s5-gift',
    'yandex-kit-gift',
    'branch-gift',
    'referral-program',
    'samsung-galaxy-gift',
    'subscription-remainder'
];

const cleanUpIntervals = (intervals = []) =>
    intervals
        // eslint-disable-next-line max-len
        .filter(
            (interval = {}) =>
                interval.orderInfo &&
                interval.orderInfo.productInfo &&
                interval.orderInfo.productInfo.brand !== 'unknown'
        )
        .map((interval = {}) => {
            const {orderType} = interval.orderInfo || {};

            return Object.assign({}, interval, {
                orderInfo: Object.assign({}, interval.orderInfo, {
                    orderType: PROMO_TYPES.includes(orderType) ? 'promo-code' : orderType
                })
            });
        });

/**
 * Формирование списка подписок по списку интервалов
 * @param {Interval[]} __intervals - список интервалов
 * @returns {Subscription[]} Список подписок
 */
const fromIntervals = (__intervals = []) => {
    const plus = [];
    const intervals = cleanUpIntervals(__intervals);
    const grouped = _.groupBy(intervals, getStatus);
    const subs = (grouped.active || []).concat(grouped.stopped || []).concat(grouped['in-grace'] || []);
    const isPlus = (interval) => BASE_PLUS_TYPES.includes(getTypeFromInterval(interval));
    const withPlus = subs
        .filter((sub) => !isPlus(sub))
        .filter((sub) => sub.orderInfo.productInfo.features.includes('basic-plus'));
    const plusPromos = (grouped['inactive-promo'] || []).filter(isPlus).map((sub) => fromInterval(sub, {}));
    const plusDaysStock = (grouped['days-stock'] || []).filter(isPlus).map((sub) => fromInterval(sub, {}));
    const plusSubs = subs.filter(isPlus);

    let pseudoPluses = [];

    if (withPlus.length > 0 && plusSubs.length === 0) {
        pseudoPluses = [
            {
                pseudo: true,
                partOf: _.uniq(withPlus.map((sub) => getTypeFromInterval(sub))),
                otherPlus: subs.filter(isPlus).map((sub) => fromInterval(sub)),
                type: TYPES.NEW_PLUS,
                status: withPlus.some((sub) => getStatus(sub) === 'active') ? 'active' : 'stopped',
                // eslint-disable-next-line max-len
                promos: collapseIntervals(
                    plusPromos.map((sub) => Object.assign({}, sub, {parentType: TYPES.NEW_PLUS}))
                ),
                // eslint-disable-next-line max-len
                daysStocks: collapseIntervals(
                    plusDaysStock.map((sub) => Object.assign({}, sub, {parentType: TYPES.NEW_PLUS}))
                )
            }
        ];
    } else {
        if (plusSubs.length >= 1) {
            plusSubs.forEach((sub) => {
                const lastToEndSub = plusSubs
                    .filter((otherSub) => getTypeFromInterval(sub) === getTypeFromInterval(otherSub))
                    .sort((a, b) => (a.end > b.end ? 1 : -1))[0];
                const isLastToEnd = lastToEndSub === sub;

                plus.push(
                    fromInterval(sub, {
                        promos: isLastToEnd
                            ? collapseIntervals(
                                  (grouped['inactive-promo'] || [])
                                      .filter((y) => getTypeFromInterval(sub) === getTypeFromInterval(y))
                                      .map((y) => fromInterval(y, {parentType: getTypeFromInterval(sub)}))
                              )
                            : [],
                        daysStocks: isLastToEnd
                            ? collapseIntervals(
                                  (grouped['days-stock'] || [])
                                      .filter((y) => getTypeFromInterval(sub) === getTypeFromInterval(y))
                                      .map((y) => fromInterval(y, {parentType: getTypeFromInterval(sub)}))
                              )
                            : [],
                        otherPlus: plusSubs.map((sub) => fromInterval(sub, {parentType: TYPES.PLUS}))
                    })
                );
            });
        }

        const plusPromosOrphan = (grouped['inactive-promo'] || [])
            .filter(isPlus)
            .filter(
                (promoSub) =>
                    !plusSubs.some((parentSub) => getTypeFromInterval(parentSub) === getTypeFromInterval(promoSub))
            )
            .map((sub) => fromInterval(sub, {}));
        const plusDayStockOrphan = (grouped['days-stock'] || [])
            .filter(isPlus)
            .filter(
                (promoSub) =>
                    !plusSubs.some((parentSub) => getTypeFromInterval(parentSub) === getTypeFromInterval(promoSub))
            )
            .map((sub) => fromInterval(sub, {}));

        if (plusPromosOrphan.length > 0 || plusDayStockOrphan.length > 0) {
            const PLUS_GROUPS = [
                [TYPES.NEW_PLUS, TYPES.PLUS],
                [TYPES.NEW_PLUS_MULTI, TYPES.PLUS_FAMILY]
            ];

            const otherOrphanTypes = _.uniq([
                ...plusPromosOrphan.map((sub) => sub.type),
                ...plusDayStockOrphan.map((sub) => sub.type)
            ]).filter((type) => !PLUS_GROUPS.some((group) => group.includes(type)));

            pseudoPluses = PLUS_GROUPS.map((group) => ({
                type: group[0],
                promos: plusPromosOrphan.filter((sub) => group.includes(sub.type)),
                daysStock: plusDayStockOrphan.filter((sub) => group.includes(sub.type))
            }))
                .concat(
                    otherOrphanTypes.map((type) => ({
                        type: type,
                        promos: plusPromosOrphan.filter((sub) => sub.type === type),
                        daysStock: plusDayStockOrphan.filter((sub) => sub.type === type)
                    }))
                )
                .filter((group) => group.promos.length > 0 || group.daysStock.length > 0)
                .map(({type, promos, daysStock}) => ({
                    pseudo: true,
                    partOf: _.uniq(
                        plusSubs.map((sub) => getTypeFromInterval(sub)).filter((partOfType) => type !== partOfType)
                    ),
                    otherPlus: [],
                    type: type,
                    status: plusSubs.some((sub) => getStatus(sub) === 'active') ? 'active' : 'stopped',
                    // eslint-disable-next-line max-len
                    promos: collapseIntervals(promos.map((sub) => Object.assign({}, sub, {parentType: type}))),
                    // eslint-disable-next-line max-len
                    daysStocks: collapseIntervals(daysStock.map((sub) => Object.assign({}, sub, {parentType: type})))
                }));
        }
    }

    pseudoPluses.forEach((_plus) => {
        if (_plus.status === 'stopped') {
            const stoppedSubs = withPlus.filter((sub) => getStatus(sub) === 'stopped').sort((a, b) => a.end <= b.end);

            if (stoppedSubs.length) {
                _plus.expires = stoppedSubs[0].orderInfo && stoppedSubs[0].orderInfo.expires;
            }
        }

        plus.push(_plus);
    });

    const _intervals = subs
        .filter((x) => !isPlus(x))
        .map((x) =>
            fromInterval(x, {
                promos: collapseIntervals(
                    (grouped['inactive-promo'] || [])
                        .filter((y) => getTypeFromInterval(x) === getTypeFromInterval(y))
                        .map((y) => fromInterval(y, {parentType: getTypeFromInterval(x)}))
                ),
                daysStocks: collapseIntervals(
                    (grouped['days-stock'] || [])
                        .filter((y) => getTypeFromInterval(x) === getTypeFromInterval(y))
                        .map((y) => fromInterval(y, {parentType: getTypeFromInterval(x)}))
                )
            })
        )
        .concat(plus.length ? plus : []);

    let warningVendors = [];
    const preparedIntervals = _.flatMap(_intervals, (sub = {}) => {
        const warnings = [];
        const {otherPlus = []} = sub;
        const otherSubs = otherPlus.filter((item) => {
            const {productInfo = {}} = item;
            const {vendor} = productInfo;

            return vendor !== 'yandex';
        });

        if (otherSubs.length >= 1) {
            otherSubs.forEach((otherSub) => {
                const {productInfo = {}} = otherSub;
                const {vendor} = productInfo;

                if (!warningVendors.length) {
                    warnings.push({
                        warning: true
                    });
                }

                warningVendors.push(vendor);
            });
        }

        return warnings
            .concat(sub)
            .concat(otherPlus)
            .concat(sub.promos)
            .concat(sub.daysStocks);
    });

    warningVendors = _.uniq(warningVendors);

    if (warningVendors.length) {
        const warningIndex = preparedIntervals.findIndex((item) => item.warning);

        if (warningVendors.length === 1) {
            preparedIntervals[warningIndex] = {
                vendor: warningVendors[0],
                warning: true
            };
        }
    }

    return preparedIntervals;
};

module.exports = {fromIntervals, getType};
