const PLog = require('plog');
const assert = require('assert');
const inherit = require('inherit');
const pdaoHttp = require('../../pdao/Http');
const error = require('../error');

module.exports = inherit(
    {
        __constructor(lodId, dao) {
            assert(dao instanceof pdaoHttp, 'Data access object should be an instance of Http DAO');

            this._dao = dao;
            this._logger = new PLog(lodId, 'api', 'billing');
        },

        getPassportInfo(uid, tld, language = 'ru') {
            return this._dao
                .call('get', '/passport-info/user', {
                    uid: uid,
                    tld,
                    language
                })
                .then(function(response) {
                    const {result = {}, error} = response;

                    return {
                        status: error ? 'error' : 'success',
                        result
                    };
                });
        },

        getTransitions({target, uid, ip, language, transitionTypes}) {
            return this._dao
                .call('get', 'billing/transitions', {
                    target,
                    ip,
                    uid,
                    language,
                    transitionTypes
                })
                .then(function(response) {
                    const {result = {}, error} = response;

                    return {
                        error,
                        ...result,
                        status: result.status === 'success' ? 'success' : 'error'
                    };
                });
        },

        getPromoCode(prototype) {
            return this._dao
                .call('post', `billing/promo-code/clone?prototype=${prototype}`, {
                    prototype
                })
                .then((response) => response);
        },

        downgradeSubscription({uid, ip, productIdFrom, productIdTo, source, origin, language, templateTag}) {
            return this._dao
                .call('post', 'subscription/downgrade-subscription', {
                    __uid: uid,
                    ip,
                    productIdFrom,
                    productIdTo,
                    // form params
                    source,
                    origin,
                    language,
                    templateTag
                })
                .then(function(response) {
                    const {result = {}, error} = response;

                    return {
                        error,
                        ...result,
                        status: result.status === 'success' ? 'success' : 'error'
                    };
                });
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: inherit(error, {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode(callback) {
                this._errors.forEach(callback);
            }
        })
    }
);
