module.exports = require('inherit')(
    {
        __constructor: function(lodId, dao) {
            require('assert')(
                dao instanceof require('pdao/Http'),
                'Data access object should be an instance of Http DAO'
            );

            /**
             * @type DAO
             */
            this._dao = dao;

            this._logger = new (require('plog'))(lodId, 'api', 'cloud');
        },

        getAddresses: function() {
            return this._dao.call('get', '/v2/personality/profile/addresses', {}).then(function(response) {
                return response;
            });
        },

        saveAddress: function(address) {
            var addressID = address.address_id;

            delete address.address_id;

            return this._dao
                .call('put', '/v2/personality/profile/addresses/' + addressID, address)
                .then(function(response) {
                    return response;
                });
        },

        deleteAddress: function(addressID) {
            return this._dao
                .call('delete', '/v2/personality/profile/addresses/' + addressID, {})
                .then(function(response) {
                    return response;
                });
        },

        getDeliveryAddresses: function() {
            return this._dao
                .call('get', '/v1/personality/profile/market/delivery_addresses', {})
                .then(function(response) {
                    return response;
                });
        },

        saveDeliveryAddress: function(address) {
            var addressID = address.id;

            delete address.id;

            return this._dao
                .call('put', '/v1/personality/profile/market/delivery_addresses/' + addressID, address)
                .then(function(response) {
                    return response;
                });
        },

        deleteDeliveryAddress: function(addressID) {
            return this._dao
                .call('delete', '/v1/personality/profile/market/delivery_addresses/' + addressID, {})
                .then(function(response) {
                    return response;
                });
        },

        getAliceSettings: function() {
            return this._dao
                .call('get', '/v1/personality/profile/alisa/settings/do_not_use_user_logs', {})
                .then(function(response) {
                    return response;
                });
        },

        saveAliceSettings: function(settings) {
            return this._dao
                .call('put', '/v1/personality/profile/alisa/settings/do_not_use_user_logs', settings)
                .then(function(response) {
                    return response;
                });
        },

        getAliceDeviceFlags: function() {
            return this._dao
                .call('get', '/v1/personality/profile/alisa_devices/personal_data', {})
                .then(function(response) {
                    return response;
                });
        },

        setAliceDeviceFlag: function(deviceId, allowed) {
            return this._dao
                .call('put', `/v1/personality/profile/alisa_devices/personal_data/${deviceId}`, {allowed})
                .then(function(response) {
                    return response;
                });
        },

        getUsersPassportSubscriptionsStatuses: function() {
            return this._dao
                .call('get', '/v1/psbilling/users/passport/subscriptions/statuses/', {})
                .then(function(response) {
                    return response;
                });
        },

        readDiskMeta: function() {
            return this._dao.call('get', '/v1/disk', {}).then(function(response) {
                return response;
            });
        },

        readDiskResourceMeta: function(path, options = {}) {
            return this._dao
                .call('get', `/v1/disk/resources?path=${path}`, {
                    preview_size: options.preview_size || 'L',
                    preview_allow_big_size: options.preview_allow_big_size || true,
                    offset: options.offset || 0,
                    limit: options.limit || 20
                })
                .then(function(response) {
                    return response;
                });
        },

        createDiskFolder: function(path) {
            return this._dao.call('put', `/v1/disk/resources?path=${path}`, {}).then(function(response) {
                return response;
            });
        },

        deleteDiskResource: function(path, permanently = false) {
            return this._dao
                .call('delete', `/v1/disk/resources?path=${path}&permanently=${permanently}`, {})
                .then(function(response) {
                    return response;
                });
        },

        searchDiskResource: function(query, options = {}) {
            return this._dao
                .call('get', `/v1/disk/resources/search?query=${query}`, {
                    preview_size: options.preview_size || 'L',
                    preview_allow_big_size: options.preview_allow_big_size || true,
                    offset: options.offset || 0,
                    limit: options.limit || 20
                })
                .then(function(response) {
                    return response;
                });
        },

        copyDiskResource: function(from, path) {
            return this._dao
                .call('post', `/v1/disk/resources/copy?from=${from}&path=${path}`, {})
                .then(function(response) {
                    return response;
                });
        },

        createDiskResource: function(path) {
            return this._dao.call('get', `/v1/disk/resources/upload?path=${path}`, {}).then(function(response) {
                return response;
            });
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: require('inherit')(require('../error'), {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor: function(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains: function(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode: function(callback) {
                this._errors.forEach(callback);
            }
        })
    }
);
