const fs = require('fs');

module.exports = require('inherit')(
    {
        __constructor: function(lodId, dao) {
            require('assert')(
                dao instanceof require('pdao/Http'),
                'Data access object should be an instance of Http DAO'
            );

            /**
             * @type DAO
             */
            this._dao = dao;

            this._logger = new (require('plog'))(lodId, 'api', 'documents');
        },

        getDocumentList: function(type, userId, options = {}) {
            return this._dao
                .call('get', `/document/list?doc_type=${type}&user_id=${userId}`, {
                    offset: options.offset || 0,
                    length: options.limit || 20
                })
                .then(function(response) {
                    return response;
                });
        },

        uploadDocument: function(type, userId, file) {
            return this._dao
                .call('post', `/document/image/upload?doc_type=${type}&user_id=${userId}`, {
                    file: file && file.size && file.path ? fs.createReadStream(file.path) : undefined
                })
                .then(function(response) {
                    return response;
                });
        },

        deleteDocument: function(userId, documentId) {
            return this._dao
                .call('get', `/document/delete?user_id=${userId}&id=${documentId}`, {})
                .then(function(response) {
                    return response;
                });
        },

        copyDocument: function(userId, path, type) {
            return this._dao
                .call('get', `/document/image/disk/copy?user_id=${userId}&path=${path}&doc_type=${type}`, {})
                .then(function(result) {
                    return result;
                });
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: require('inherit')(require('../error'), {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor: function(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains: function(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode: function(callback) {
                this._errors.forEach(callback);
            }
        })
    }
);
