const plog = require('plog');
const Error = require('../error');

class FamilyPayApi {
    constructor(lodId, dao) {
        this._dao = dao;
        this._logger = new plog(lodId, 'api', 'family-pay');
    }

    _getFamilyId(familyId) {
        return String(familyId).match(/\d+/)[0];
    }

    fetchFamilyByIdSingleLimit(familyId) {
        return this._dao.call('get', `/family/${this._getFamilyId(familyId)}/single_limit`, {});
    }

    updateFamilyCurrencySingleLimit(familyId, currency) {
        return this._dao.call('post', `/family/${this._getFamilyId(familyId)}/update/single_limit`, {
            cardInfo: {currency}
        });
    }

    bindCardSingleLimit(familyId, uid, cardId, maskedNumber, paySystem, currency) {
        return this._dao.call(
            'post',
            `/family/${this._getFamilyId(familyId)}/start/single_limit?uid=${uid}&origin=passport`,
            {cardInfo: {cardId, maskedNumber, currency, paySystem}}
        );
    }

    updateCardSingleLimit(familyId, cardInfo) {
        return this._dao.call('post', `/family/${this._getFamilyId(familyId)}/update/single_limit`, {cardInfo});
    }

    updateCardBound(familyId, bound) {
        return this._dao.call('post', `/family/${this._getFamilyId(familyId)}/update`, {cardInfo: {bound}});
    }

    unbindCard(familyId) {
        return this._dao.call('post', `/family/${this._getFamilyId(familyId)}/stop`, {});
    }

    updateUsersSingleLimit(familyId, newLimits) {
        return this._dao.call('post', `/family/${this._getFamilyId(familyId)}/update/users/single_limit`, newLimits);
    }

    resetUserExpenses(uid) {
        return this._dao.call('post', `/user/${uid}/reset_expenses`, {});
    }
}

FamilyPayApi.ApiError = class extends Error {
    /* eslint-disable no-this-before-super */
    constructor(errors, response) {
        this.name = 'ApiError';
        this.message = 'Api errors encountered: ' + JSON.stringify(errors);
        this._errors = errors;

        super(response);
    }

    contains(code) {
        return this._errors.indexOf(code) > -1;
    }

    forEveryCode(callback) {
        this._errors.forEach(callback);
    }
};

module.exports = FamilyPayApi;
