var _ = require('lodash');
var assert = require('assert');
var crypto = require('crypto');
var fs = require('fs');
var PLog = require('plog');
var HTTPDao = require('pdao/Http');
var Client = require('./models/Client');
var RetryError = require('../error/RetryError');
var ScopesCollection = require('./models/ScopesCollection');
var retryCondition = require('./retryCondition');

/**
 * Oauth frontend handles
 * @see https://wiki.yandex-team.ru/oauth/iface_api
 *
 * @constructor
 * @typedef Api
 */
module.exports = require('inherit')(
    {
        /**
         * @param {string}  logID       Log ID
         * @param {DAO}     dao         Api DAO
         * @param {string}  consumer    Consumer identifier
         * @param {object}  rawHeaders  Raw headers hash, forwarded to the API
         * @param {string}  lang        Interface language, a two-letter code, one of [en, ru, tr]
         * @param {string}  [uid]       User id
         * @constructor
         * @class Api
         */
        __constructor: function(logID, dao, consumer, rawHeaders, lang, uid) {
            assert(logID && typeof logID === 'string', 'Log ID should be defined');
            assert(dao instanceof HTTPDao, 'Data access object should be an instance of Http DAO');
            assert(consumer && typeof consumer === 'string', 'Consumer should be a string');
            assert(_.isObjectLike(rawHeaders), 'Headers should be a hash of raw headers');
            assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');
            assert(!uid || typeof uid === 'string', 'Uid should be a string if defined');

            /**
             * @type DAO
             */
            this._dao = dao;
            this._dao.mixHeaders(this.__self.transformHeaders(rawHeaders));

            /**
             * User uid
             * @type {?string}
             */
            this._uid = uid;

            /**
             * User language
             * One of en, ru, tr, uk
             * @type {string}
             */
            this._lang = lang;

            /**
             * Consumer identifier for the api
             * @type {string}
             * @private
             */
            this._consumer = consumer;

            this._logger = new PLog(logID, 'api', 'oauth');
        },

        _filterEmpty: function(values) {
            return _.omitBy(values, function(value) {
                var type = typeof value;

                if (type === 'undefined') {
                    return true;
                }

                if (type === 'string') {
                    return value.length === 0;
                }

                return false;
            });
        },

        _responseHandler: function(response) {
            if (response.status === 'error' && _.isArray(response.errors) && response.errors.length > 0) {
                throw new this.__self.ApiError(response.errors, response);
            }

            return response;
        },

        _checkUidIsSet: function(method) {
            assert(
                this._uid && typeof this._uid === 'string',
                'Uid should be set before using method "%s"'.replace('%s', method)
            );
        },

        _checkClientId: function(clientId) {
            if (!this.__self.ClientModel.isValidClientId(clientId)) {
                this._logger.error('Client ID "%s" does not looks like a valid client id', clientId);
                throw new this.__self.ApiError(['client.not_found']);
            }
        },

        _checkGlogout: function(method, glogout) {
            assert(
                glogout && typeof glogout === 'string',
                'Global logout time should be provided for method ' + method
            );
        },

        _mapScopes: function(scopes) {
            var ScopeModel = this.__self.ScopeModel;

            return new ScopesCollection(
                _.flatten(
                    _.map(scopes, function(scopes, sectionTitle) {
                        return _.map(scopes, function(scope, permissionId) {
                            return new ScopeModel(permissionId, scope).setSectionTitle(sectionTitle);
                        });
                    })
                )
            );
        },
        /**
         * Requested scopes и Already granted scopes имеют вид
         *
         *  "Test": {
         *       "test:foo": {
         *           "title": "Foo",
         *           "requires_approval": false,
         *           "ttl": null,
         *           "is_ttl_refreshable": false,
         *       },
         *       "test:bar": {
         *           "title": "Foo",
         *           "requires_approval": false,
         *           "ttl": 120,
         *           "is_ttl_refreshable": true,
         *       }
         *   }
         * @param {Object} requestedScopes
         * @param {Object} alreadyGrantedScopes
         * @param {Array<String>} optionalScopes - массив id опциональных доступов
         * @private
         */
        _mapRequestedScopes(requestedScopes, alreadyGrantedScopes, optionalScopes) {
            var ScopeModel = this.__self.ScopeModel;

            var requested = _.flatten(
                _.map(requestedScopes, function(scopes, sectionTitle) {
                    return _.map(scopes, function(scope, permissionId) {
                        var isOptional = optionalScopes.indexOf(permissionId) !== -1;

                        return new ScopeModel(permissionId, scope)
                            .setSectionTitle(sectionTitle)
                            .setOptional(isOptional)
                            .setAlreadyGranted(false);
                    });
                })
            );
            var alreadyGranted = _.flatten(
                _.map(alreadyGrantedScopes, function(scopes, sectionTitle) {
                    return _.map(scopes, function(scope, permissionId) {
                        return new ScopeModel(permissionId, scope)
                            .setSectionTitle(sectionTitle)
                            .setOptional(false)
                            .setAlreadyGranted(true);
                    });
                })
            );

            return new ScopesCollection(requested.concat(alreadyGranted));
        },

        /**
         * Get a list of clients created by user
         * Resolves with ClientModel[]
         * @see https://wiki.yandex-team.ru/oauth/iface_api#listcreatedclients
         *
         * @returns when.Promise
         */
        listCreatedClients: function() {
            this._checkUidIsSet('listCreatedClients');

            var that = this;
            var ClientModel = this.__self.ClientModel;

            return this._dao
                .call('get', '/2/clients/created', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then(function(createdClients) {
                    return _.map(createdClients.clients, function(client) {
                        client.scopes = that._mapScopes(client.scopes);
                        return new ClientModel(client);
                    });
                });
        },

        /**
         * Get a list of clients created by user
         * Resolves with ClientModel[]
         * @see https://wiki.yandex-team.ru/oauth/iface_api#listcreatedclients
         *
         * @returns when.Promise
         */
        listCreatedClientsVer2: function() {
            this._checkUidIsSet('listCreatedClients');

            return this._dao
                .call('get', '/1/clients/owned', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then((createdClients) => {
                    return _.map(createdClients.created_clients, (client) => {
                        client.scopes = this._mapScopes(client.scopes);
                        return new this.__self.ClientModel(client);
                    });
                });
        },

        listCreatedClientsVer2New: function() {
            this._checkUidIsSet('listCreatedClients');

            return this._dao
                .call('get', '/1/clients/owned', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then((response) => response.created_clients);
        },

        _parseToken: function(token) {
            token.client.scopes = this._mapScopes(token.client.scopes);
            token.client = new this.__self.ClientModel(token.client);
            token.scopes = this._mapScopes(token.scopes);
            return new this.__self.TokenModel(token);
        },

        /**
         * Get a list of tokens user has issued
         * Resolves with TokenModel[]
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#listtokens
         *
         * @param {string} glogouttime
         * @param {string} tokensRevokeTime
         * @param {string} appPasswordsRevokeTime
         * @param {string} webSessionsRevokeTime
         * @param {number} passwordVerificationAge
         *
         * @returns {Promise}
         */

        listTokens: function(glogouttime, tokensRevokeTime, appPasswordsRevokeTime, webSessionsRevokeTime) {
            //TODO: testme
            this._checkGlogout('authorizeSubmit', glogouttime);
            this._checkUidIsSet('listTokens');

            assert(typeof tokensRevokeTime === 'string', 'tokensRevokeTime should be a string');
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');
            assert(typeof webSessionsRevokeTime === 'string', 'webSessionsRevokeTime should be a string');

            var that = this;

            return this._dao
                .call('get', '/2/tokens/list', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang,
                    user_glogout_time: glogouttime,
                    user_tokens_revoke_time: tokensRevokeTime,
                    user_app_passwords_revoke_time: appPasswordsRevokeTime,
                    user_web_sessions_revoke_time: webSessionsRevokeTime
                })
                .then(this._responseHandler.bind(this))
                .then(function(tokens) {
                    return _.map(tokens.tokens, that._parseToken.bind(that));
                });
        },

        /**
         * Get a list of tokens user has issued, password protected
         * Resolves with TokenModel[]
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#listtokens
         *
         * @returns {Promise}
         */

        listTokensVer3: function() {
            this._checkUidIsSet('listTokensVer3');

            return this._dao
                .call('get', '/3/tokens/list', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then(({tokens}) => tokens.map(this._parseToken.bind(this)));
        },

        /**
         * Get a list of tokens grouped by device
         * @see https://wiki.yandex-team.ru/oauth/ifaceapi/#listtokengroups
         *
         * @returns {Promise}
         */

        listTokenGroups: function(isHideYandexTokens) {
            var options = {
                consumer: this._consumer,
                uid: this._uid,
                language: this._lang
            };

            if (isHideYandexTokens) {
                options.hide_yandex_device_tokens = true;
            }

            this._checkUidIsSet('listTokenGroups');

            return this._dao
                .call('get', '/1/token/list_groups', options)
                .then(this._responseHandler.bind(this))
                .then(function(tokens) {
                    return tokens;
                });
        },

        /**
         * Generate an application password
         *
         * @param {string} clientId
         * @param {string} deviceName
         * @param {string} userIP
         * @returns {Promise}
         */
        issueAppPassword: function(
            clientId,
            deviceName,
            userIP,
            glogouttime,
            tokensRevokeTime,
            appPasswordsRevokeTime,
            webSessionsRevokeTime,
            passwordVerificationAge
        ) {
            //TODO: test me :(
            this._checkUidIsSet('issueAppPassword');
            this._checkClientId(clientId);
            this._checkGlogout('issueAppPassword', glogouttime);
            assert(typeof deviceName === 'string', 'Device Name should be a string');
            assert(userIP && typeof userIP === 'string', 'User IP should be a string');
            assert(typeof tokensRevokeTime === 'string', 'tokensRevokeTime should be a string');
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');
            assert(typeof webSessionsRevokeTime === 'string', 'webSessionsRevokeTime should be a string');
            assert(typeof passwordVerificationAge !== 'undefined', 'passwordVerificationAge is required');

            var deviceId = crypto
                .createHash('md5')
                .update(this.__self.DEVICEID_SALT)
                .update(clientId)
                .update(deviceName)
                .update(userIP)
                .update(String(Math.random()))
                .digest('hex');

            var that = this;

            return this._dao
                .call('post', '/2/token/issue_app_password', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang,
                    client_id: clientId,
                    device_id: deviceId,
                    device_name: deviceName,
                    user_ip: userIP,
                    user_glogout_time: glogouttime,
                    user_tokens_revoke_time: tokensRevokeTime,
                    user_app_passwords_revoke_time: appPasswordsRevokeTime,
                    user_web_sessions_revoke_time: webSessionsRevokeTime,
                    password_verification_age: passwordVerificationAge
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    var token = that._parseToken(response.token);

                    token.setAlias(response.token_alias);
                    return token;
                });
        },

        /**
         * Generate an application password
         *
         * @param {string} clientId
         * @param {string} deviceName
         * @param {string} userIP
         * @returns {Promise}
         */
        issueAppPasswordVer2: function(clientId, deviceName, userIP) {
            this._checkUidIsSet('issueAppPassword');
            this._checkClientId(clientId);
            assert(typeof deviceName === 'string', 'Device Name should be a string');
            assert(userIP && typeof userIP === 'string', 'User IP should be a string');

            var deviceId = crypto
                .createHash('md5')
                .update(this.__self.DEVICEID_SALT)
                .update(clientId)
                .update(deviceName)
                .update(userIP)
                .update(String(Math.random()))
                .digest('hex');

            var that = this;

            // client_id - обязательный
            // device_id - обязательный уникальный идентификатор устройства.
            // От 6 до 50 любых ASCII-символы с кодами от 32 до 126
            // device_name - обязательный человекочитаемое имя устройства (от 1 до 100 символов)
            // user_ip - обязательный
            // language - обязательный

            return this._dao
                .call('post', '/2/token/app_passwords/issue', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang,
                    client_id: clientId,
                    device_id: deviceId,
                    device_name: deviceName,
                    user_ip: userIP
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    var token = that._parseToken(response.token);

                    token.setAlias(response.token_alias);
                    return token;
                });
        },

        /**
         * Get number of application passwords
         * @see https://wiki.yandex-team.ru/oauth/iface_api/#countapppasswords
         *
         * @param {string} user_glogout_time
         * @param {string} user_app_passwords_revoke_time
         * @returns {Promise}
         */
        countAppPasswords: function(glogouttime, appPasswordsRevokeTime) {
            this._checkUidIsSet('сountAppPasswords');
            this._checkGlogout('сountAppPasswords', glogouttime);
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');

            return this._dao
                .call('get', '/1/token/app_passwords_count', {
                    consumer: this._consumer,
                    uid: this._uid,
                    user_glogout_time: glogouttime,
                    user_app_passwords_revoke_time: appPasswordsRevokeTime
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        countAppPasswordsVer2: function() {
            this._checkUidIsSet('сountAppPasswords');

            return this._dao
                .call('get', '/2/token/app_passwords/count', {
                    consumer: this._consumer,
                    uid: this._uid
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Revoke tokens for a given client
         * @see https://wiki.yandex-team.ru/oauth/iface_api#revoketokens
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        revokeTokens: function(clientID) {
            this._checkUidIsSet('revokeTokens');
            this._checkClientId(clientID);

            return this._dao
                .call('post', '/1/tokens/revoke', {
                    consumer: this._consumer,
                    uid: this._uid,
                    client_id: clientID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Revokes single token
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#revoketoken
         *
         * @param {string} tokenID
         * @returns when.Promise
         */
        revokeSingleToken: function(tokenID, passwordVerificationAge) {
            this._checkUidIsSet('revokeTokens');
            assert(tokenID && typeof tokenID === 'number', 'Token id should be a number');
            assert(typeof passwordVerificationAge !== 'undefined', 'passwordVerificationAge is required');

            return this._dao
                .call('post', '/2/token/revoke', {
                    consumer: this._consumer,
                    uid: this._uid,
                    token_id: tokenID,
                    password_verification_age: passwordVerificationAge
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Revokes single token ver3
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#revoketoken /iface_api/3/token/revoke
         *
         * @param {number|Array<number>} tokenID
         * @returns when.Promise
         */
        revokeSingleTokenVer3: function(tokenID) {
            this._checkUidIsSet('revokeTokens');
            assert(
                tokenID && (typeof tokenID === 'number' || tokenID instanceof Array),
                'Token id should be a number or array of numbers'
            );

            return this._dao
                .call('post', '/3/token/revoke', {
                    consumer: this._consumer,
                    uid: this._uid,
                    token_id: tokenID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Revokes all tokens binded to device
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#revokedevicetokens /iface_api/1/token/revoke_device
         *
         * @param {number} deviceID
         * @returns when.Promise
         */
        revokeDeviceTokens: function(deviceID) {
            this._checkUidIsSet('revokeDeviceTokens');

            return this._dao
                .call('post', '/1/token/revoke_device', {
                    consumer: this._consumer,
                    uid: this._uid,
                    device_id: deviceID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Get the info about a given client
         * Resolves with ClientModel instance
         * @see https://wiki.yandex-team.ru/oauth/iface_api#clientinfo
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        clientInfo: function(clientID) {
            this._checkClientId(clientID);

            var that = this;
            var ClientModel = this.__self.ClientModel;

            return this._dao
                .call(
                    'get',
                    '/2/client/info',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        language: this._lang,
                        client_id: clientID
                    })
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    var rawResponse = _.merge({}, response);

                    response.client.scopes = that._mapScopes(response.client.scopes);
                    return new ClientModel(
                        response.client,
                        response.viewed_by_owner,
                        response.can_be_edited,
                        rawResponse
                    );
                });
        },

        /**
         * Get the info about a given client
         * Resolves with ClientModel instance
         * @see https://wiki.yandex-team.ru/oauth/iface_api#clientinfo
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        clientInfoVer3: function(clientID, withExtraScopes) {
            this._checkClientId(clientID);

            return this._dao
                .call(
                    'get',
                    '/3/client/info',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        language: this._lang,
                        client_id: clientID
                    })
                )
                .then(this._responseHandler.bind(this))
                .then((response) => {
                    const rawResponse = _.merge({}, response);

                    response.client.scopes = this._mapScopes(response.client.scopes);
                    if (withExtraScopes && response.extra_visible_scopes) {
                        response.client.extraScopes = this._mapScopes(response.extra_visible_scopes);
                    }

                    return new this.__self.ClientModel(
                        response.client,
                        response.viewed_by_owner,
                        response.can_be_edited,
                        rawResponse
                    );
                });
        },

        clientInfoVer3New: function(clientID) {
            this._checkClientId(clientID);

            return this._dao
                .call(
                    'get',
                    '/3/client/info',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        language: this._lang,
                        force_no_slugs: true,
                        client_id: clientID
                    })
                )
                .then(this._responseHandler.bind(this))
                .then((response) => _.merge({}, response));
        },

        /**
         * Validate user input before creating/editing app
         *
         * @see https://wiki.yandex-team.ru/oauth/iface_api#validateclientchanges
         *
         * @param {string}              clientId                    Идентификатор приложения
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {string}              [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {string}              [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         * (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */

        validateClientChanges({
            clientId,
            title,
            description,
            homepage,
            isYandex,
            iconId,
            iconFile,
            platforms,
            iosAppId,
            iosAppstoreUrl,
            androidPackageName,
            androidAppstoreUrl,
            androidCertFingerprints,
            turboappBaseUrl,
            redirectUris,
            scopes,
            ownerUids,
            ownerGroups
        }) {
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppId,
                    iosAppstoreUrl,
                    androidPackageName,
                    androidAppstoreUrl
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    redirectUris,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/1/client/validate',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        client_id: clientId,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Validate user input before creating/editing app
         *
         * @see https://wiki.yandex-team.ru/oauth/iface_api#validateclientchanges
         *
         * @param {string}              clientId                    Идентификатор приложения
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {Array<String>}       [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {Array<String>}       [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         *                                                              (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */

        validateClientChangesV3(
            {
                clientId,
                title,
                description,
                homepage,
                isYandex,
                iconId,
                iconFile,
                platforms,
                iosAppId,
                iosAppstoreUrl,
                androidPackageName,
                androidAppstoreUrl,
                androidCertFingerprints,
                turboappBaseUrl,
                redirectUris,
                scopes,
                contact,
                ownerUids,
                ownerGroups
            },
            {requirePlatform = false} = {}
        ) {
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppstoreUrl,
                    androidAppstoreUrl
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    iosAppId,
                    androidPackageName,
                    redirectUris,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/3/client/validate',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        client_id: clientId,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        require_platform: requirePlatform,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        contact_email: contact,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Create a new client
         * Resolves with clientId
         * @see https://wiki.yandex-team.ru/oauth/iface_api#createclient
         *
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {string}              [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {string}              [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         * (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */
        createClientV2({
            title,
            description,
            homepage,
            isYandex,
            iconId,
            iconFile,
            platforms,
            iosAppId,
            iosAppstoreUrl,
            androidPackageName,
            androidAppstoreUrl,
            androidCertFingerprints,
            turboappBaseUrl,
            redirectUris,
            scopes,
            ownerUids,
            ownerGroups
        }) {
            this._checkUidIsSet('createClient');
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppId,
                    iosAppstoreUrl,
                    androidAppstoreUrl,
                    androidPackageName
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    redirectUris,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/2/client/create',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Create a new client
         * Resolves with clientId
         * @see https://wiki.yandex-team.ru/oauth/iface_api#createclient
         *
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {Array<String>}       [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {Array<String>}       [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         *                                                              (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */
        createClientV3(
            {
                title,
                description,
                homepage,
                isYandex,
                iconId,
                iconFile,
                platforms,
                iosAppId,
                iosAppstoreUrl,
                androidPackageName,
                androidAppstoreUrl,
                androidCertFingerprints,
                turboappBaseUrl,
                redirectUris,
                scopes,
                contact,
                ownerUids,
                ownerGroups
            },
            {requirePlatform = false} = {}
        ) {
            this._checkUidIsSet('createClient');
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppstoreUrl,
                    androidAppstoreUrl
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    redirectUris,
                    iosAppId,
                    androidPackageName,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/3/client/create',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        require_platform: requirePlatform,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        contact_email: contact,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Edit a given client
         * Resolves with clientId
         * @see https://wiki.yandex-team.ru/oauth/iface_api#editclient
         *
         * @param {string}              clientId                    Идентификатор приложения
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {string}              [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {string}              [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         * (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */
        editClientV2({
            clientId,
            title,
            description,
            homepage,
            isYandex,
            iconId,
            iconFile,
            platforms,
            iosAppId,
            iosAppstoreUrl,
            androidPackageName,
            androidAppstoreUrl,
            androidCertFingerprints,
            turboappBaseUrl,
            redirectUris,
            scopes,
            ownerUids,
            ownerGroups
        }) {
            this._checkUidIsSet('editClient');
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    clientId,
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppId,
                    iosAppstoreUrl,
                    androidAppstoreUrl,
                    androidPackageName
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    redirectUris,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/2/client/edit',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        client_id: clientId,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Edit a given client
         * Resolves with clientId
         * @see https://wiki.yandex-team.ru/oauth/iface_api#editclient
         *
         * @param {string}              clientId                    Идентификатор приложения
         * @param {string}              title                       Имя приложения
         * @param {string}              [description]                 Описание приложения
         * @param {ScopesCollection}    scopes                      Список доступов приложения
         * @param {string}              [homepage]                  Урл сайта приложения
         * @param {Array<String>}       [redirectUris]              Урл, куда отдается токен при веб-регистрации
         *                                                              9 аттрибут ББ account.is_corporate
         * @param {boolean}             [isYandex]                  Флаг "корпоративности" приложения,
         *                                                              описывает ли оно яндексовский сервис
         * @param {string}              [iconId]                    Id иконки
         * @param {object}              [iconFile]                  Файл иконки
         * @param {Array<String>}       [platforms]                 Выбранные платформы
         * @param {Array<String>}       [iosAppId]                  id приложения
         * @param {string}              [iosAppstoreUrl]            ссылка на приложение в AppStore
         * @param {Array<String>}       [androidPackageName]        название пакета приложения
         * @param {string}              [androidAppstoreUrl]        ссылка на приложение в Google Play
         * @param {Array<string>}       [androidCertFingerprints]   список sha256-отпечатков Android приложения
         *                                                              (не более 5)
         * @param {Array<string>}       [ownerUids]                 список дополнительных уидов,
         *                                                              имеющих доступ к аккаунту (не более 5)
         * @param {Array<string>}       [ownerGroups]               список групп уидов, имеющих доступ к аккаунту
         *                                                              (не более 5)
         * @returns {Promise}                                       when.js promise
         */
        editClientV3(
            {
                clientId,
                title,
                description,
                homepage,
                isYandex,
                iconId,
                iconFile,
                platforms,
                iosAppId,
                iosAppstoreUrl,
                androidPackageName,
                androidAppstoreUrl,
                androidCertFingerprints,
                turboappBaseUrl,
                redirectUris,
                scopes,
                contact,
                ownerUids,
                ownerGroups
            },
            {requirePlatform = false} = {}
        ) {
            this._checkUidIsSet('editClient');
            const config = this._dao.cloneConfig();

            config.setMultipart(true);

            _.each(
                {
                    clientId,
                    title,
                    description,
                    homepage,
                    iconId,
                    iosAppstoreUrl,
                    androidAppstoreUrl
                },
                function(value, key) {
                    assert(
                        !value || typeof value === 'string',
                        'Client %s should be a string if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    scopes,
                    platforms,
                    redirectUris,
                    iosAppId,
                    androidPackageName,
                    androidCertFingerprints,
                    ownerUids,
                    ownerGroups
                },
                function(value, key) {
                    assert(
                        !value || value instanceof Array,
                        'Client %s should be an array if defined'.replace('%s', key)
                    );
                }
            );

            _.each(
                {
                    'Corporate client': isYandex
                },
                function(value, key) {
                    assert(
                        ['boolean', 'undefined'].indexOf(typeof value) > -1,
                        '%s flag should be a boolean if defined'.replace('%s', key)
                    );
                }
            );

            return this._dao
                .call(
                    'post',
                    '/3/client/edit',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        client_id: clientId,
                        title,
                        description,
                        homepage,
                        is_yandex: typeof isYandex === 'boolean' ? isYandex.toString() : undefined,
                        icon_id: iconId,
                        icon_file:
                            iconFile && iconFile.size && iconFile.path ? fs.createReadStream(iconFile.path) : undefined,
                        platforms,
                        require_platform: requirePlatform,
                        ios_app_id: iosAppId,
                        ios_appstore_url: iosAppstoreUrl,
                        android_package_name: androidPackageName,
                        android_appstore_url: androidAppstoreUrl,
                        android_cert_fingerprints: androidCertFingerprints,
                        turboapp_base_url: turboappBaseUrl,
                        redirect_uri: redirectUris,
                        scopes,
                        contact_email: contact,
                        owner_uids: ownerUids,
                        owner_groups: ownerGroups
                    }),
                    config
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response;
                });
        },

        /**
         * Delete a given client
         * @see https://wiki.yandex-team.ru/oauth/iface_api#deleteclient
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        deleteClientV2: function(clientID) {
            this._checkUidIsSet('deleteClientV2');
            this._checkClientId(clientID);

            return this._dao
                .call('post', '/2/client/delete', {
                    consumer: this._consumer,
                    uid: this._uid,
                    client_id: clientID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Revoke all tokens for the given client
         * @see https://wiki.yandex-team.ru/oauth/iface_api#glogoutclient
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        gLogoutClient: function(clientID) {
            this._checkUidIsSet('gLogoutClient');
            this._checkClientId(clientID);

            return this._dao
                .call('post', '/1/client/glogout', {
                    consumer: this._consumer,
                    uid: this._uid,
                    client_id: clientID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Generate a new secret for the given client
         * Resolves with a new secret value
         * @see https://wiki.yandex-team.ru/oauth/iface_api#newclientsecret
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        newClientSecretV2: function(clientID) {
            this._checkUidIsSet('newClientSecretV2');
            this._checkClientId(clientID);

            return this._dao
                .call('post', '/2/client/secret/new', {
                    consumer: this._consumer,
                    uid: this._uid,
                    client_id: clientID
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Revert the previous secret for the given client
         * Resolves with a reverted secret
         * @see https://wiki.yandex-team.ru/oauth/iface_api#undonewclientsecret
         *
         * @param {string} clientID
         * @returns when.Promise
         */
        undoNewClientSecretV2: function(clientID) {
            this._checkUidIsSet('undoNewClientSecretV2');
            this._checkClientId(clientID);

            return this._dao
                .call('post', '/2/client/secret/undo', {
                    consumer: this._consumer,
                    uid: this._uid,
                    client_id: clientID
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return response.secret;
                });
        },

        /**
         * Describes the current backend behaviour
         * Resolves with a hash of options
         * @see https://wiki.yandex-team.ru/oauth/iface_api#settings
         *
         * @returns when.Promise
         */
        settings: function() {
            return this._dao
                .call('get', '/1/global/settings', {
                    consumer: this._consumer
                })
                .then(this._responseHandler.bind(this));
        },

        /**
         * Get the list of all scopes
         * Resolves with Scope[]
         * @see https://wiki.yandex-team.ru/oauth/ifaceapi/#allscopes
         */
        allScopes: function() {
            var that = this;

            return this._dao
                .call('get', '/1/scopes/all', {
                    consumer: this._consumer,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return that._mapScopes(response.all_scopes);
                });
        },

        /**
         * Get the scopes accessible by the current user
         * Resolves with Scope[]
         * @see https://wiki.yandex-team.ru/oauth/iface_api#visiblescopes
         *
         * @deprecated  Use userInfo handle instead
         * @see userInfo
         *
         * @returns when.Promise
         */
        visibleScopes: function() {
            this._checkUidIsSet('visibleScopes');

            var that = this;

            return this._dao
                .call('get', '/1/scopes/visible', {
                    consumer: this._consumer,
                    uid: this._uid,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    return that._mapScopes(response.visible_scopes);
                });
        },

        /**
         * Get scopes visible to the user and other specific information
         * Resolves with {
         *  scopes: Scope[]
         *  ipInternal: Boolean
         *  allowRegisteringYandexClients: Boolean
         * }
         *
         * @see https://beta.wiki.yandex-team.ru/oauth/iface_api/#userinfo
         *
         * @returns when.Promise
         */
        userInfoV2: function({noSlugs = false} = {}) {
            this._checkUidIsSet('userInfo');

            return this._dao
                .call('get', '/2/user/info', {
                    consumer: this._consumer,
                    uid: this._uid,
                    force_no_slugs: noSlugs,
                    language: this._lang
                })
                .then(this._responseHandler.bind(this))
                .then((response) => ({
                    scopes: this._mapScopes(response.visible_scopes),
                    ipInternal: response.is_ip_internal,
                    allowRegisteringYandexClients: response.allow_register_yandex_clients
                }));
        },

        /**
         * Initiate the authorization process
         * @see https://wiki.yandex-team.ru/oauth/iface_api#authorizesubmit
         *
         * @param {string} glogouttime      Global logout time
         * @param {string} clientId
         * @param {string} userIp
         * @param {string} responseType     Either 'code' or 'token'
         * @param {string} [redirectUrl]
         * @param {string} [state]
         * @param {string} [device_id]
         * @param {string} [device_name]
         *
         * @returns when.Promise
         */
        authorizeSubmit: function(
            glogouttime,
            clientId,
            userIp,
            responseType,
            redirectUrl,
            state,
            device_id,
            device_name,
            tokensRevokeTime,
            appPasswordsRevokeTime,
            webSessionsRevokeTime
        ) {
            this._checkGlogout('authorizeSubmit', glogouttime);
            this._checkUidIsSet('authorizeSubmit');
            this._checkClientId(clientId);

            assert(this.__self.isResponseTypeValid(responseType), 'Response type should be either "code" or "token"');
            assert(userIp && typeof userIp === 'string', 'User IP should be a string');
            assert(typeof tokensRevokeTime === 'string', 'tokensRevokeTime should be a string');
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');
            assert(typeof webSessionsRevokeTime === 'string', 'webSessionsRevokeTime should be a string');

            _.each(
                {
                    'Redirect url': redirectUrl,
                    State: state,
                    'Device id': device_id,
                    'Device name': device_name
                },
                function(value, description) {
                    assert(
                        !value || typeof value === 'string',
                        '%s should be a string if defined'.replace('%s', description)
                    );
                }
            );

            var that = this;

            return this._dao
                .call(
                    'post',
                    '/2/authorize/submit',
                    this._filterEmpty({
                        consumer: this._consumer,
                        uid: this._uid,
                        language: this._lang,
                        user_glogout_time: glogouttime,
                        client_id: clientId,
                        user_ip: userIp,
                        response_type: responseType,
                        redirect_uri: redirectUrl,
                        state: state,
                        device_id: device_id,
                        device_name: device_name,
                        user_tokens_revoke_time: tokensRevokeTime,
                        user_app_passwords_revoke_time: appPasswordsRevokeTime,
                        user_web_sessions_revoke_time: webSessionsRevokeTime
                    })
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    response.requested_scopes = that._mapScopes(response.requested_scopes);
                    return response;
                });
        },

        /**
         * Initiate the authorization process
         * @see https://wiki.yandex-team.ru/oauth/iface_api#authorizesubmit
         *
         * @param {string}          clientId
         * @param {string}          responseType     Either 'code' or 'token'
         * @param {string}          [redirectUrl]
         * @param {string}          [state]
         * @param {string}          [deviceId]
         * @param {string}          [deviceName]
         * @param {Array<string>}   [requiredScopes]
         * @param {Array<string>}   [optionalScopes]
         * @param {string}          [codeChallenge]         секрет для дополнительной защиты выдаваемого
         * кода подтверждения
         * @param {string}          [codeChallengeMethod]   метод формирования code_challenge: plain или S256,
         *                                                  если не передано - то plain
         *
         * @returns when.Promise
         */
        authorizeSubmitV2: function(
            clientId,
            responseType,
            redirectUrl,
            state,
            deviceId,
            deviceName,
            requiredScopes,
            optionalScopes,
            codeChallenge,
            codeChallengeMethod,
            paymentAuthScheme
        ) {
            requiredScopes = requiredScopes || [];
            optionalScopes = optionalScopes || [];

            const self = this;
            const scopes = _.union(requiredScopes, optionalScopes);

            self._checkUidIsSet('authorizeSubmit');
            self._checkClientId(clientId);

            _.each(
                {
                    redirectUrl,
                    state,
                    deviceId,
                    deviceName,
                    codeChallenge,
                    codeChallengeMethod,
                    paymentAuthScheme
                },
                function(value, description) {
                    assert(
                        !value || typeof value === 'string',
                        '%s should be a string if defined'.replace('%s', description)
                    );
                }
            );

            assert(!requiredScopes || requiredScopes instanceof Array, 'requestedScopes should be an array if defined');
            assert(!optionalScopes || optionalScopes instanceof Array, 'optionalScopes should be an array if defined');

            return self._dao
                .call(
                    'post',
                    '/3/authorize/submit',
                    this._filterEmpty({
                        consumer: self._consumer,
                        uid: self._uid,
                        client_id: clientId,
                        language: self._lang,
                        redirect_uri: redirectUrl,
                        response_type: responseType,
                        state,
                        device_id: deviceId,
                        device_name: deviceName,
                        requested_scopes: scopes,
                        code_challenge: codeChallenge,
                        code_challenge_method: codeChallengeMethod,
                        payment_auth_scheme: paymentAuthScheme
                    }),
                    this._dao.cloneConfig().setMixContentTypeForceBody(true)
                )
                .then(self._responseHandler.bind(self))
                .then(function(response) {
                    const requestedScopes = response.requested_scopes;
                    const alreadyGrantedScopes = response.already_granted_scopes;

                    response.requested_scopes = self._mapRequestedScopes(
                        requestedScopes,
                        alreadyGrantedScopes,
                        optionalScopes
                    );

                    return response;
                });
        },

        /**
         * Grant the authorization
         * @see https://wiki.yandex-team.ru/oauth/iface_api#authorizecommit
         *
         * @param {string} glogouttime  Global logout time
         * @param {string} userIp
         * @param {string} requestId
         * @param {string} tokensRevokeTime
         * @param {string} appPasswordsRevokeTime
         * @param {string} webSessionsRevokeTime
         *
         * @returns when.Promise
         */
        authorizeCommit: function(
            glogouttime,
            userIp,
            requestId,
            tokensRevokeTime,
            appPasswordsRevokeTime,
            webSessionsRevokeTime
        ) {
            this._checkGlogout('authorizeCommit', glogouttime);
            this._checkUidIsSet('authorizeCommit');

            assert(typeof tokensRevokeTime === 'string', 'tokensRevokeTime should be a string');
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');
            assert(typeof webSessionsRevokeTime === 'string', 'webSessionsRevokeTime should be a string');

            _.each(
                {
                    'User IP': userIp,
                    'Request ID': requestId
                },
                function(value, description) {
                    assert(value && typeof value === 'string', '%s should be a string'.replace('%s', description));
                }
            );

            var callConfig = this._dao.cloneConfig();
            var currentFailCheck = callConfig.getApiFailCheck();
            var reqNotFoundCode = 'request.not_found';

            callConfig.setMaxRetries(1).setApiFailCheck(function() {
                return (
                    currentFailCheck.apply(null, arguments) || retryCondition([reqNotFoundCode]).apply(null, arguments)
                );
            });

            var that = this;

            return this._dao
                .call(
                    'post',
                    '/1/authorize/commit',
                    {
                        consumer: this._consumer,
                        uid: this._uid,
                        user_glogout_time: glogouttime,
                        user_ip: userIp,
                        request_id: requestId,
                        user_tokens_revoke_time: tokensRevokeTime,
                        user_app_passwords_revoke_time: appPasswordsRevokeTime,
                        user_web_sessions_revoke_time: webSessionsRevokeTime
                    },
                    callConfig
                )
                .then(this._responseHandler.bind(this))
                .catch(function(err) {
                    if (err instanceof RetryError && err.getErrCode() === reqNotFoundCode) {
                        //Transform the error
                        throw new that.__self.ApiError([reqNotFoundCode], err.getResponse());
                    }

                    //Unknown error, rethrow
                    throw err;
                });
        },

        /**
         * Grant the authorization
         * @see https://wiki.yandex-team.ru/oauth/iface_api#authorizecommit
         *
         * @param {string}          requestedId
         * @param {Array<string>}   grantedScopes
         *
         * @returns when.Promise
         */
        authorizeCommitV2: function(requestedId, grantedScopes, paymentAuthRetpath) {
            var self = this;

            self._checkUidIsSet('authorizeCommit');

            var callConfig = self._dao.cloneConfig();
            var currentFailCheck = callConfig.getApiFailCheck();
            var reqNotFoundCode = 'request.not_found';

            assert(typeof requestedId === 'string', 'Request id should be a string');
            assert(!grantedScopes || grantedScopes instanceof Array, 'Granted scoped should be array if defined');
            assert(
                !paymentAuthRetpath || typeof paymentAuthRetpath === 'string',
                'Payment auth retpath should be a string'
            );

            callConfig
                .setMaxRetries(1)
                .setMixContentTypeForceBody(true)
                .setApiFailCheck(
                    (...args) =>
                        currentFailCheck.apply(null, args) || retryCondition([reqNotFoundCode]).apply(null, args)
                );

            const params = {
                consumer: self._consumer,
                uid: self._uid,
                request_id: requestedId,
                granted_scopes: grantedScopes
            };

            if (paymentAuthRetpath) {
                params.payment_auth_retpath = paymentAuthRetpath;
            }

            return self._dao
                .call('post', '/2/authorize/commit', params, callConfig)
                .then(this._responseHandler.bind(this))
                .catch(function(err) {
                    if (err instanceof RetryError && err.getErrCode() === reqNotFoundCode) {
                        // Transform the error
                        throw new self.__self.ApiError([reqNotFoundCode], err.getResponse());
                    }

                    // Unknown error, rethrow
                    throw err;
                });
        },

        /**
         * Get info about the authorization process
         * @see https://wiki.yandex-team.ru/oauth/iface_api/#authorizegetstate
         *
         * @param {string} glogouttime  Global logout time
         * @param {string} userIp
         * @param {string} requestId
         * @param {string} tokensRevokeTime
         * @param {string} appPasswordsRevokeTime
         * @param {string} webSessionsRevokeTime
         *
         * @returns when.Promise
         */
        authorizeGetState: function(
            glogouttime,
            userIp,
            requestId,
            tokensRevokeTime,
            appPasswordsRevokeTime,
            webSessionsRevokeTime
        ) {
            this._checkGlogout('authorizeGetState', glogouttime);
            this._checkUidIsSet('authorizeGetState');

            assert(typeof tokensRevokeTime === 'string', 'tokensRevokeTime should be a string');
            assert(typeof appPasswordsRevokeTime === 'string', 'appPasswordsRevokeTime should be a string');
            assert(typeof webSessionsRevokeTime === 'string', 'webSessionsRevokeTime should be a string');

            _.each(
                {
                    'User IP': userIp,
                    'Request ID': requestId
                },
                function(value, description) {
                    assert(value && typeof value === 'string', '%s should be a string'.replace('%s', description));
                }
            );

            var callConfig = this._dao.cloneConfig();
            var currentFailCheck = callConfig.getApiFailCheck();

            var reqNotFoundCode = 'request.not_found';

            callConfig.setMaxRetries(1).setApiFailCheck(function() {
                return (
                    currentFailCheck.apply(null, arguments) || retryCondition([reqNotFoundCode]).apply(null, arguments)
                );
            });

            var that = this;

            return this._dao
                .call(
                    'post',
                    '/1/authorize/get_state',
                    {
                        consumer: this._consumer,
                        uid: this._uid,
                        user_glogout_time: glogouttime,
                        user_ip: userIp,
                        request_id: requestId,
                        user_tokens_revoke_time: tokensRevokeTime,
                        user_app_passwords_revoke_time: appPasswordsRevokeTime,
                        user_web_sessions_revoke_time: webSessionsRevokeTime
                    },
                    callConfig
                )
                .then(this._responseHandler.bind(this))
                .catch(function(err) {
                    if (err instanceof RetryError && err.getErrCode() === reqNotFoundCode) {
                        //Transform the error
                        throw new that.__self.ApiError([reqNotFoundCode], err.getResponse());
                    }

                    //Unknown error, rethrow
                    throw err;
                });
        },
        /**
         * Get info about the authorization process
         * @see https://wiki.yandex-team.ru/oauth/iface_api/#authorizegetstate
         *
         * @param {string} requestId
         *
         * @returns when.Promise
         */
        authorizeGetStateV2: function(requestId) {
            const self = this;

            this._checkUidIsSet('authorizeGetState');

            assert(requestId && typeof requestId === 'string', 'Request id should be a string');

            const callConfig = self._dao.cloneConfig();
            const currentFailCheck = callConfig.getApiFailCheck();
            const reqNotFoundCode = 'request.not_found';

            callConfig.setMaxRetries(1).setApiFailCheck(function() {
                return (
                    currentFailCheck.apply(null, arguments) || retryCondition([reqNotFoundCode]).apply(null, arguments)
                );
            });

            return self._dao
                .call(
                    'post',
                    '/2/authorize/get_state',
                    {
                        consumer: self._consumer,
                        uid: self._uid,
                        request_id: requestId
                    },
                    callConfig
                )
                .then(self._responseHandler.bind(self))
                .catch(function(err) {
                    if (err instanceof RetryError && err.getErrCode() === reqNotFoundCode) {
                        // Transform the error
                        throw new self.__self.ApiError([reqNotFoundCode], err.getResponse());
                    }

                    // Unknown error, rethrow
                    throw err;
                });
        },

        /**
         * Initiate the device authorization process
         * @see https://wiki.yandex-team.ru/oauth/iface_api/#deviceauthorizesubmit
         *
         * @param {string} code
         * @param {string} clientId
         *
         * @returns when.Promise
         */
        deviceAuthorizeSubmit: function(code, clientId) {
            const self = this;

            self._checkUidIsSet('deviceAuthorizeSubmit');

            const params = {
                consumer: self._consumer,
                uid: self._uid,
                language: self._lang,
                code
            };

            if (clientId) {
                params.client_id = clientId;
            }

            return self._dao.call('post', '/1/device/authorize/submit', params).then(self._responseHandler.bind(self));
        },

        /**
         * Grant the device authorization
         * @see https://wiki.yandex-team.ru/oauth/iface_api/#deviceauthorizecommit
         *
         * @param {string} code
         * @param {string} clientId
         *
         * @returns when.Promise
         */
        deviceAuthorizeCommit: function(code, clientId) {
            const self = this;

            self._checkUidIsSet('deviceAuthorizeCommit');

            const callConfig = self._dao.cloneConfig();
            const currentFailCheck = callConfig.getApiFailCheck();
            const reqNotFoundCode = 'request.not_found';

            callConfig.setMaxRetries(1).setApiFailCheck(function() {
                return (
                    currentFailCheck.apply(null, arguments) || retryCondition([reqNotFoundCode]).apply(null, arguments)
                );
            });

            const params = {
                consumer: self._consumer,
                uid: self._uid,
                language: self._lang,
                code
            };

            if (clientId) {
                params.client_id = clientId;
            }

            return self._dao
                .call('post', '/1/device/authorize/commit', params, callConfig)
                .then(this._responseHandler.bind(this))
                .catch(function(err) {
                    if (err instanceof RetryError && err.getErrCode() === reqNotFoundCode) {
                        // Transform the error
                        throw new self.__self.ApiError([reqNotFoundCode], err.getResponse());
                    }

                    // Unknown error, rethrow
                    throw err;
                });
        }
    },
    (function() {
        var appPasswordsClientIdMapping = null;

        return {
            setAppPasswordsClientIdMapping: function(mapping) {
                assert(_.isObjectLike(mapping), 'Mapping should be a plain object');
                assert(
                    _.each(mapping, function(slug, clientId) {
                        return Client.isValidClientId(clientId);
                    }),
                    'Each mapping key should be a valid client id string'
                );
                assert(
                    _.each(mapping, function(slug) {
                        return typeof slug === 'string';
                    }),
                    'Each mapping value should be a string'
                );

                appPasswordsClientIdMapping = mapping;
            },
            getAppPasswordsSlugByClientId: function(clientId) {
                assert(
                    appPasswordsClientIdMapping !== null,
                    'App passwords slugs to client id mapping should be defined ' +
                        'with require("papi/OAuth").setAppPasswordsClientIdMapping()'
                );

                var slug = appPasswordsClientIdMapping[clientId];

                assert(slug && typeof slug === 'string', 'No known slug for client id ' + clientId);
                return slug;
            },
            _resetAppPasswordsClientIdMapping: function() {
                appPasswordsClientIdMapping = null;
            },

            /**
             * An error object to represent errors returned by api
             *
             * @class ApiError
             * @extends Error
             */
            ApiError: require('inherit')(require('../error'), {
                /**
                 * @param {string[]} errors
                 * @param {object} response
                 * @constructor
                 */
                __constructor: function(errors, response) {
                    this.name = 'ApiError';
                    this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                    this._errors = errors;

                    this.__base(response);
                },

                /**
                 * Whether the given code was among the errors
                 * @param {string} code
                 * @returns {boolean}
                 */
                contains: function(code) {
                    return this._errors.indexOf(code) > -1;
                },

                forEveryCode: function(callback) {
                    this._errors.forEach(callback);
                }
            }),

            isResponseTypeValid: function(responseType) {
                return ['code', 'token'].indexOf(responseType) > -1;
            },

            DEVICEID_SALT: 'Quoovoje9eidighuupeivufo5pereutoo0aenoh6epui5Ib1KaiL8Ah1oTh6kaes',

            /*
        Method to transform raw headres into a hash suitable for the API.
        Uses Passport implementation.
         */
            transformHeaders: require('../Passport').transformHeaders,

            TokenModel: require('./models/Token'),
            ClientModel: require('./models/Client'),
            ScopeModel: require('./models/Scope')
        };
    })()
);
