/**
 * A client model
 *
 * @constructor
 * @class ClientModel
 */
module.exports = require('inherit')(
    {
        __constructor: function(client, viewedByOwner, canBeEdited, response) {
            this._logger = new (require('plog'))();
            this._logger.type('oauth', 'model', 'client');

            this._logger.verbose('Creating model from %j', client);

            var assert = require('assert');

            assert(require('lodash').isPlainObject(client), 'Client should be a hash of values describing an client');

            assert(
                client.scopes instanceof require('./ScopesCollection'),
                'Client scopes should be an instance of ScopesCollection'
            );
            ['id', 'title'].forEach(function(requiredField) {
                assert(
                    client[requiredField] && typeof client[requiredField] === 'string',
                    requiredField + ' should be a string and should not be empty'
                );
            });
            ['description', 'callback', 'homepage', 'icon'].forEach(function(field) {
                assert(typeof client[field] === 'string', field + ' should be a string');
            });

            require('lodash').each(
                {
                    approval_status: 'number',
                    blocked: 'boolean',
                    can_be_edited: 'boolean'
                },
                function(type, field) {
                    assert(
                        ['undefined', type].indexOf(typeof client[field]) > -1,
                        require('util').format('Client %s should be a %s if defined', field, type)
                    );
                }
            );

            if (typeof client.create_time !== 'number') {
                client.create_time = 0; //Treat as a start of an epoch
            }

            if (typeof client.secret !== 'string') {
                client.secret = ''; //Treat as an empty secret
            }

            this._client = client;
            this._viewedByOwner = Boolean(viewedByOwner);
            this._canBeEdited = Boolean(canBeEdited);
            this._scopes = client.scopes;
            this._extraScopes = client.extraScopes;
            this._response = response;
        },

        /**
         * Get client id
         * @returns {string}
         */
        getId: function() {
            return this._client.id;
        },

        /**
         * Get client title
         * @returns {string}
         */
        getTitle: function() {
            return this._client.title;
        },

        /**
         * @returns {string}
         */
        getDescription: function() {
            return this._client.description;
        },

        /**
         * @returns {string}
         */
        getIcon: function() {
            return this._client.icon_url;
        },

        /**
         * @returns {string}
         */
        getIconId: function() {
            return this._client.icon_id;
        },

        getPlatforms: function() {
            return this._client.platforms;
        },

        /**
         * @returns {string}
         */
        getHomepage: function() {
            return this._client.homepage;
        },

        /**
         * @returns {string}
         */
        getCallbackUrl: function() {
            return this._client.callback;
        },

        /**
         * @returns {?string}
         */
        getSecret: function() {
            return this._client.secret;
        },

        /**
         * @returns {?boolean}
         */
        isBlocked: function() {
            return this._client.blocked;
        },

        /**
         * @returns {?boolean}
         */
        isEditable: function() {
            return this._canBeEdited;
        },

        /**
         * @returns {?boolean}
         */
        isYandex: function() {
            return this._client.is_yandex;
        },

        /**
         * @returns {?string}
         */
        getApprovalStatus: function() {
            var approval_status = this._client.approval_status;

            if (typeof approval_status !== 'number') {
                return null;
            }

            var status = null;

            require('lodash').some(this.__self.APPROVAL, function(value, statusName) {
                if (value === approval_status) {
                    status = statusName;
                    return true;
                }

                return false;
            });

            if (status === null) {
                throw new Error('Unknown approval status');
            }

            return status;
        },

        /**
         * @returns {ScopesCollection}
         */
        getScopes: function() {
            return this._scopes;
        },

        /**
         * @returns {ScopesCollection}
         */
        getExtraScopes: function() {
            return this._extraScopes;
        },

        /**
         * @returns {boolean}
         */
        isViewedByOwner: function() {
            return this._viewedByOwner;
        },

        /**
         * @returns {Date}
         */
        getCreationTime: function() {
            return new Date(this._client.create_time * 1000);
        },

        getJSONResponse: function() {
            return this._response;
        }
    },
    {
        isValidClientId: (function() {
            var clientIdRe = /^[0-9a-f]{32}$/;

            return function(clientId) {
                return clientIdRe.test(clientId);
            };
        })(),

        APPROVAL: {
            NOT_REQUIRED: 0,
            PENDING: 1,
            RESOLVED: 2,
            REJECTED: 3
        }
    }
);
