var assert = require('assert');
var _ = require('lodash');
/**
 * A single scope
 *
 * @constructor
 * @class ScopeModel
 */

module.exports = require('inherit')(
    {
        /**
         * @param {string}   id      Scope id
         * @param {object}   [scope] Descriptive hash for a scope
         * @private
         */
        __constructor: function(id, scope) {
            assert(id && typeof id === 'string', 'Scope id should be a string');
            assert(!scope || _.isObjectLike(scope), 'Scope should be a plain object if given');

            this._id = id;
            this._section = null;
            this._optional = false;
            this._alreadyGranted = false;

            if (scope) {
                assert(scope.title && typeof scope.title === 'string', 'Scope title should be a string');
                assert(typeof scope.requires_approval === 'boolean', 'Approval flag should be a boolean');
                assert(typeof scope.is_ttl_refreshable === 'boolean', 'Refreshable flag should be a boolean');
                assert(scope.ttl === null || typeof scope.ttl === 'number', 'TTL should be a number if given');

                this._title = scope.title;
                this._requiresApproval = scope.requires_approval;
                this._ttlRefreshable = scope.is_ttl_refreshable;
                this._ttl = scope.ttl;
                this._tags = scope.tags;
            }
        },

        /**
         * @returns {string}
         */
        getId: function() {
            return this._id;
        },

        /**
         * Explicitly define a human-readable section title for the scope
         *
         * @param {string} title
         * @returns {ScopeModel}
         */
        setSectionTitle: function(title) {
            assert(title && typeof title === 'string', 'Scope section title should be a string');
            this._section = title;
            return this;
        },

        /**
         * @param   {boolean} optional - Флаг, указывающий, что доступ опционален, и может быть отменен пользователем
         * @returns {ScopeModel}
         */
        setOptional: function(optional) {
            assert(typeof optional === 'boolean', 'Scope optional flag should be a boolean');
            this._optional = optional;
            return this;
        },

        /**
         * @param   {boolean} optional - Флаг, указывающий, что доступ был предоставлен
         * @returns {ScopeModel}
         */
        setAlreadyGranted: function(granted) {
            assert(typeof granted === 'boolean', 'Scope optional flag should be a boolean');
            this._alreadyGranted = granted;
            return this;
        },

        /**
         * @returns {string}
         */
        getSectionTitle: function() {
            return this._section;
        },

        /**
         * @returns {boolean}
         */
        getOptional: function() {
            return this._optional;
        },

        /**
         * @returns {boolean}
         */
        getAlreadyGranted: function() {
            return this._alreadyGranted;
        },

        /**
         * @returns {number|string}
         */
        getTtl: function() {
            if (this._ttl === null) {
                return this.__self.TTL_INFINITE;
            }

            return Number(this._ttl);
        },

        /**
         * @returns {boolean}
         */
        isTtlInfinite: function() {
            return this.getTtl() === this.__self.TTL_INFINITE;
        },

        /**
         * @returns {boolean}
         */
        isTtlRefreshable: function() {
            return Boolean(this._ttlRefreshable);
        },

        /**
         * @returns {boolean}
         */
        requiresApproval: function() {
            return Boolean(this._requiresApproval);
        },

        /**
         * @returns {string}
         */
        getTitle: function() {
            return this._title;
        },

        /**
         * Whether this scope is the same scope as another one
         * @param {ScopeModel} scope
         * @return {boolean}
         */
        isSame: function(scope) {
            assert(scope instanceof this.constructor, 'Scope should be a Scope Model');
            return this.getId() === scope.getId();
        }
    },
    {
        TTL_INFINITE: 'INFINITE',
        TTL_NOT_DEFINED: 'NOT_DEFINED'
    }
);
