var assert = require('assert');
var _ = require('lodash');

/**
 * @constructor
 * @class ScopesCollection
 */
var ScopesCollection = require('inherit')({
    /**
     * @param {ScopeModel...|ScopeModel[]} scope
     * @constructor
     */
    __constructor: function(scope) {
        var scopes = Array.prototype.slice.call(arguments);

        if (_.isArray(scope)) {
            scopes = scope;
        }
        scopes.forEach(this._assertScope);

        this._scopes = scopes;
    },

    _assertScope: function(scope) {
        assert(scope instanceof require('./Scope'), 'Scope should be a Scope Model');
        return true;
    },

    /**
     * Whether the collection contains a specific scope
     * @param {Scope} scope
     * @returns {boolean}
     */
    contains: function(scope) {
        return _.some(this._scopes, function(otherScope) {
            return otherScope.isSame(scope);
        });
    },

    /**
     * Calculate the ttl
     * @returns {string|number}
     */
    getTtl: function() {
        var Scope = require('./Scope');

        if (this._scopes.length === 0) {
            return Scope.TTL_NOT_DEFINED;
        }

        if (
            this.every(function(scope) {
                return scope.isTtlInfinite();
            })
        ) {
            return Scope.TTL_INFINITE;
        }

        var minScope = this.min(function(scope) {
            var ttl = scope.getTtl();

            return ttl === Scope.TTL_INFINITE ? Infinity : Number(ttl);
        });

        if (minScope === Infinity) {
            //Lodash .min edge case
            return Scope.TTL_INFINITE;
        }

        return minScope.getTtl();
    },

    /**
     * @returns {boolean}
     */
    isTtlRefreshable: function() {
        //Either scope is not defined or is infinite
        if (typeof this.getTtl() !== 'number') {
            return false;
        }

        //Or the ttl is refreshable if all the scopes have refreshable ttl
        return this.every(function(scope) {
            return scope.isTtlRefreshable();
        });
    }
});

Object.defineProperty(ScopesCollection.prototype, 'length', {
    get: function() {
        return this._scopes.length;
    }
});

// Proxying lodash methods for the collection
['map', 'every', 'any', 'min', 'groupBy'].forEach(function(method) {
    ScopesCollection.prototype[method] = function(callback) {
        return _[method].call(null, this._scopes, callback);
    };
});

module.exports = ScopesCollection;
