const assert = require('assert');

module.exports = require('inherit')(
    {
        __constructor: function(lodId, dao, rawHeaders, consumer, lang) {
            require('assert')(
                dao instanceof require('pdao/Http'),
                'Data access object should be an instance of Http DAO'
            );
            require('assert')(require('lodash').isPlainObject(rawHeaders), 'Headers should be passed as a dictionary');
            require('assert')(consumer && typeof consumer === 'string', 'Consumer should be a string');
            assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');

            /**
             * @type DAO
             */
            this._dao = dao.mixHeaders(this.__self.transformHeaders(rawHeaders));

            /**
             * Consumer identifier
             * @type {string}
             * @private
             */
            this._consumer = consumer;

            /**
             * User language
             * One of en, ru, tr, uk
             * @type {string?}
             */
            this._lang = lang;

            this._logger = new (require('plog'))(lodId, 'api', 'passport');
        },

        _responseHandler: function(response) {
            if (
                response.status === 'error' &&
                require('lodash').isArray(response.errors) &&
                response.errors.length > 0
            ) {
                throw new this.__self.ApiError(response.errors, response);
            }

            return response;
        },

        getLastAuth: function(track_2fa_was_enabled_with) {
            //TODO: test
            require('assert')(
                !track_2fa_was_enabled_with || typeof track_2fa_was_enabled_with === 'string',
                'Track id should be a string if any given'
            );

            var that = this;

            //No uid there, since headers already contain session_id cookie
            return this._dao
                .call(
                    'get',
                    '/1/bundle/account/lastauth/',
                    require('lodash').omit(
                        {
                            consumer: this._consumer,
                            //TODO: remove me around september 2015 (should not be necessary by now)
                            enabled_2fa_track_id: track_2fa_was_enabled_with
                        },
                        function(value) {
                            return !value;
                        }
                    )
                )
                .then(this._responseHandler.bind(this))
                .then(function(response) {
                    var _ = require('lodash');
                    var factory = require('../OAuth/models/AuthLogEntries/factory');

                    return {
                        webPass: response.lastauth.password.web.map(factory),
                        webApp: response.lastauth.password.apps.map(factory),
                        tokens: _(response.lastauth.tokens)
                            .toArray()
                            .flatten()
                            .map(factory)
                            .valueOf(),
                        appPass: _(response.lastauth.application_passwords)
                            .toArray()
                            .flatten()
                            .map(factory)
                            .valueOf()
                    };
                })
                .catch(function(error) {
                    that._logger
                        .warn()
                        .type('getLastAuth')
                        .write('Proceeding with empty history due to', error);

                    return {
                        webPass: [],
                        webApp: [],
                        tokens: [],
                        appPass: []
                    };
                });
        },

        writeStatbox: function(dict) {
            //TODO: test
            require('assert')(require('lodash').isPlainObject(dict), 'Statbox data should be a plain object');

            return this._dao
                .call('post', '/1/statbox/?consumer=' + this._consumer, dict)
                .then(this._responseHandler.bind(this));
        },

        /**
         * @param {any} flag possible values: yes|no|y|n|1|0|true|false
         * @returns {string} yes|no boolean equivalent
         */
        _parseFlag(flag) {
            if (flag === 'yes' || flag === 'y' || flag === true || flag === '1' || flag === 1) {
                return 'yes';
            }

            return 'no';
        },

        /**
         * Send SMS with confirmation code
         * @see https://wiki.yandex-team.ru/passport/api/bundle/phone
         *
         * @param {object} params number, displayLanguage, trackId, country
         *
         * @returns {Promise} handled promise
         */
        phoneConfirmAndBindSecureSubmit({number, displayLanguage, trackId, country, code_format}) {
            assert(number && typeof number === 'string', 'number should be a string');
            assert(!displayLanguage || typeof displayLanguage === 'string', 'displayLanguage should be a string');
            assert(!trackId || typeof trackId === 'string', 'trackId should be a string');
            assert(!country || typeof country === 'string', 'country should be a string');

            const params = {
                consumer: this._consumer,
                number,
                track_id: trackId,
                display_language: displayLanguage ? displayLanguage : this._lang
            };

            if (country) {
                params.country = country;
            }

            if (code_format) {
                params.code_format = code_format;
            }

            return this._dao
                .call('post', `/2/bundle/phone/confirm_and_bind_secure/submit/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Send SMS with confirmation code
         * @see https://wiki.yandex-team.ru/passport/api/bundle/phone/#submit
         *
         * @param {object} params number, displayLanguage, trackId
         *
         * @returns {Promise} handled promise
         */
        bindSimpleOrConfirmBoundPhoneSubmit({number, displayLanguage, trackId, country, code_format}) {
            assert(number && typeof number === 'string', 'number should be a string');
            assert(!displayLanguage || typeof displayLanguage === 'string', 'displayLanguage should be a string');

            const params = {
                consumer: this._consumer,
                number,
                track_id: trackId,
                display_language: displayLanguage ? displayLanguage : this._lang,
                country
            };

            if (code_format) {
                params.code_format = code_format;
            }

            return this._dao
                .call('post', `/2/bundle/phone/bind_simple_or_confirm_bound/submit/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Skip additional data request until next time
         * @see https://wiki.yandex-team.ru/passport/api/bundle/auth/
         *
         * @param {string} isUserDeclined user declined action flag (yes|no)
         * @param {string} trackId request track_id
         * @param {string} uid unique user id
         *
         * @returns {Promise} handled promise
         */
        additionalDataFreeze(isUserDeclined, trackId, uid) {
            assert(!trackId || typeof trackId === 'string', 'trackId should be a string');
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = {
                consumer: this._consumer,
                user_declined: this._parseFlag(isUserDeclined),
                track_id: trackId
            };

            if (uid) {
                params.uid = uid;
            }

            return this._dao
                .call('post', `/1/bundle/auth/additional_data/freeze/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Get additional data request
         * @see https://wiki.yandex-team.ru/passport/api/bundle/auth/
         *
         * @param {string} uid unique user id
         *
         * @returns {Promise} handled promise
         */
        additionalDataAsk(uid) {
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = uid ? {uid, consumer: this._consumer} : {consumer: this._consumer};

            return this._dao
                .call('get', `/1/bundle/auth/additional_data/ask/`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Check confirmation code and bind phone as secure
         * @see https://wiki.yandex-team.ru/passport/api/bundle/phone/#/commit3
         *
         * @param {string} code confirmation code from SMS
         * @param {string} password user password
         * @param {string} trackId request track_id
         *
         * @returns {Promise} handled promise
         */
        phoneConfirmAndBindSecureCommit(code, password, trackId) {
            assert(code && typeof code === 'string', 'code should be a string');
            assert(!password || typeof password === 'string', 'password should be a string');
            assert(!trackId || typeof trackId === 'string', 'trackId should be a string');

            const params = {
                code,
                track_id: trackId,
                consumer: this._consumer
            };

            if (password) {
                params.password = password;
            }

            return this._dao
                .call('post', `/2/bundle/phone/confirm_and_bind_secure/commit/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Check confirmation code and bind phone as secure
         * @see https://wiki.yandex-team.ru/passport/api/bundle/phone/#commit
         *
         * @param {string} code confirmation code from SMS
         * @param {string} password user password
         * @param {string} trackId request track_id
         *
         * @returns {Promise} handled promise
         */
        bindSimpleOrConfirmBoundPhoneCommit(code, trackId) {
            assert(code && typeof code === 'string', 'code should be a string');

            const params = {
                code,
                track_id: trackId,
                consumer: this._consumer
            };

            return this._dao
                .call('post', `/2/bundle/phone/bind_simple_or_confirm_bound/commit/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Phone number confirmation without SMS
         * @see https://wiki.yandex-team.ru/passport/api/bundle/phone/manage/
         *
         * @param {string} phoneId unique phone id
         * @param {string} displayLanguage SMS text language
         * @param {string} uid unique user id
         *
         * @returns {Promise} handled promise
         */
        phoneProlongValid(phoneId, displayLanguage, uid) {
            assert(phoneId && typeof phoneId === 'string', 'phoneId should be a string');
            assert(!displayLanguage || typeof displayLanguage === 'string', 'displayLanguage should be a string');
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = {
                consumer: this._consumer,
                phone_id: phoneId,
                display_language: displayLanguage ? displayLanguage : this._lang
            };

            if (uid) {
                params.uid = uid;
            }

            return this._dao
                .call('post', `/1/bundle/phone/manage/prolong_valid/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Send mail for email confirmation
         * @see https://wiki.yandex-team.ru/passport/api/bundle/email/#otpravkapismadljapodtverzhdenija
         *
         * @param {string} email email
         * @param {string} retpath url for redirect after confirmation
         * @param {boolean} isCodeOnly send message only with confirmation code (without link)
         * @param {string} trackId request track_id
         *
         * @returns {Promise} handled promise
         */
        sendConfirmationEmail({email, retpath, uid, codeOnly, validatorUIUrl, trackId}) {
            assert(email && typeof email === 'string', 'email should be a string');
            assert(retpath && typeof retpath === 'string', 'retpath should be a string');
            assert(!trackId || typeof trackId === 'string', 'trackId should be a string');
            assert(!validatorUIUrl || typeof validatorUIUrl === 'string', 'validatorUIUrl should be a string');
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = {
                consumer: this._consumer,
                email,
                retpath,
                validator_ui_url: validatorUIUrl,
                track_id: trackId,
                code_only: this._parseFlag(codeOnly),
                language: this._lang
            };

            if (uid) {
                params.uid = uid;
            }

            return this._dao
                .call('post', `/1/bundle/email/send_confirmation_email/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Check confirmation code from mail
         * @see https://wiki.yandex-team.ru/passport/api/bundle/email/#podtverzhdenieemejjlapokodu
         *
         * @param {string} key confirmation code from mail
         * @param {string} trackId request track_id
         * @param {string} uid unique user id
         *
         * @returns {Promise} handled promise
         */
        confirmEmailByCode(key, trackId, uid) {
            assert(key && typeof key === 'string', 'key should be a string');
            assert(!trackId || typeof trackId === 'string', 'trackId should be a string');
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = {
                key,
                consumer: this._consumer,
                track_id: trackId
            };

            if (uid) {
                params.uid = uid;
            }

            return this._dao
                .call('post', `/1/bundle/email/confirm/by_code/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Setup email for recovery
         * @see https://wiki.yandex-team.ru/passport/api/bundle/email/#redaktirovanieprivjazannogoemejjla
         *
         * @param {string} email recovery email
         * @param {string} isSafe recovery flag ('yes'|'no')
         * @param {string} uid unique user id
         *
         * @returns {Promise} handled promise
         */
        setupConfirmedEmail(email, isSafe, uid) {
            assert(email && typeof email === 'string', 'email should be a string');
            assert(!uid || typeof uid === 'string', 'uid should be a string');

            const params = {
                email,
                consumer: this._consumer,
                is_safe: this._parseFlag(isSafe)
            };

            if (uid) {
                params.uid = uid;
            }

            return this._dao
                .call('post', `/1/bundle/email/setup_confirmed/?consumer=${this._consumer}`, params)
                .then(this._responseHandler.bind(this));
        },

        /**
         * Setup social profile for authorization
         * @see https://wiki.yandex-team.ru/passport/api/bundle/socialprofiles/
         *
         * @param {string} profileId unique social profile id
         * @param {string} isAuthEnabled enable flag fro auth via social profile (yes|no)
         * @param {string} trackId request track_id
         *
         * @returns {Promise} handled promise
         */
        setSocialProfileAuth(profileId, isAuthEnabled, trackId) {
            assert(profileId && typeof profileId === 'string', 'profileId should be a string');
            assert(trackId && typeof trackId === 'string', 'trackId should be a string');

            return this._dao
                .call('post', `/2/bundle/change_social/profile/?consumer=${this._consumer}`, {
                    consumer: this._consumer,
                    profile_id: profileId,
                    track_id: trackId,
                    set_auth: isAuthEnabled
                })
                .then(this._responseHandler.bind(this));
        }
    },
    {
        /**
         * An error object to represent errors returned by api
         *
         * @class ApiError
         * @extends Error
         */
        ApiError: require('inherit')(require('../error'), {
            /**
             * @param {string[]} errors
             * @param {object} response
             * @constructor
             */
            __constructor: function(errors, response) {
                this.name = 'ApiError';
                this.message = 'Api errors encountered: ' + JSON.stringify(errors);
                this._errors = errors;

                this.__base(response);
            },

            /**
             * Whether the given code was among the errors
             * @param {string} code
             * @returns {boolean}
             */
            contains: function(code) {
                return this._errors.indexOf(code) > -1;
            },

            forEveryCode: function(callback) {
                this._errors.forEach(callback);
            }
        }),

        transformHeaders: (function() {
            var specialTransformations = {
                'x-real-ip': 'ya-consumer-client-ip',
                authorization: 'ya-consumer-authorization',
                'x-yproxy-header-ip': 'x-yproxy-header-ip',
                'x-real-scheme': 'ya-consumer-client-scheme',
                'X-Ya-Service-Ticket': 'X-Ya-Service-Ticket'
            };

            var explicitlyEmpty = ['cookie', 'user-agent'];

            var prefix = 'ya-client-';

            return function(headers) {
                var _ = require('lodash');

                require('assert')(_.isObjectLike(headers), 'Headers should be a dictionary');

                headers = _.transform(
                    headers,
                    function(headers, value, header) {
                        if (header in specialTransformations) {
                            headers[specialTransformations[header]] = value;
                        } else if (header.indexOf(prefix) !== 0) {
                            headers[prefix + header] = value;
                        }
                    },
                    {}
                );

                explicitlyEmpty.forEach(function(explicitlyEmpty) {
                    var header = prefix + explicitlyEmpty;

                    if (!headers[header]) {
                        headers[header] = '';
                    }
                });

                return headers;
            };
        })()
    }
);
