(function(global, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory(require('./passport'));
    } else {
        factory(global.passport);
    }
})(this, function(passport) {
    'use strict';

    if (typeof window !== 'object') {
        return {};
    }

    var DOM_EVENTS = {
        blur: 1,
        change: 1,
        click: 1,
        dblclick: 1,
        dragstart: 1,
        dragenter: 1,
        dragover: 1,
        dragleave: 1,
        drag: 1,
        drop: 1,
        dragend: 1,
        focus: 1,
        focusin: 1,
        focusout: 1,
        keydown: 1,
        keypress: 1,
        keyup: 1,
        mousedown: 1,
        mouseenter: 1,
        mouseleave: 1,
        mousemove: 1,
        mouseout: 1,
        mouseover: 1,
        mouseup: 1,
        resize: 1,
        scroll: 1,
        submit: 1
    };

    // eslint-disable-next-line no-useless-escape
    var RE_EVENT = /^(([^\.\s]+)(?:\.[^\s]+)*)\s?(.+)?$/;

    /**
     * Blocks store
     * @type Object
     */
    passport.blocks = {};

    /**
     * Defines block
     * @param {String} id
     * @param {String|Object} [parent]
     * @param {Object} [extra]
     *
     * @returns Object
     */
    passport.block = function(id, parent, extra) {
        var block;
        var blocks = passport.blocks;

        if (arguments.length === 1) {
            return blocks[id];
        }

        if (arguments.length === 2) {
            extra = parent;
            parent = 'block';
        }

        parent = blocks[parent] || {};

        var Block = function() {};

        Block.prototype = parent;
        block = new Block();

        $.extend(block, extra);

        block.id = id;
        block.parent = parent;

        /**
         * Deferred to keep track if the block is inited
         * @type jQuery.Deferred | null
         *
         * TODO: switch to blocks instantiation, remove "inited" from here
         */
        block.inited = new $.Deferred();

        blocks[id] = block;

        return block;
    };

    //noinspection JSValidateJSDoc,JSValidateJSDoc
    /**
     * Base block definition
     */
    passport.block('block', {
        /**
         * @constructor
         * @param {string} selector      Selector, determining the element on the page the block is bound to
         * @param {object} [options]     Initialization options
         */
        construct: function(selector, options) {
            this.$el = $(selector);
            this.initEvents();
            this.init(options);
            this.options = options || {};
        },

        /**
         * Stub. Use to init whatever you want in your block
         */
        init: function() {},

        /**
         * Parses `events` object and bind events
         */
        initEvents: function() {
            var rule;
            var match;
            var $target;
            var events = this.events || {};

            for (rule in events) {
                if (typeof this[events[rule]] !== 'function') {
                    throw new Error(
                        'Method ' +
                            events[rule] +
                            ' is undefined in ' +
                            this.id +
                            ' block, but is referenced in events hash.'
                    );
                }

                /**
                 * match -> ['click.ns span', 'click.ns', 'click', 'span']
                 * match[1]: full event name
                 * match[2]: event
                 * match[3]: target element
                 */
                match = rule.match(RE_EVENT);

                if (!match) {
                    throw new Error('Wrong rule "' + rule + '"');
                }

                var callback = this[events[rule]].bind(this);

                if (['blur', 'focusout'].indexOf(match[2]) > -1) {
                    /*
                     Suppose you have a field with focusout handle
                     You put focus in that field and click a button
                     Focus event happens sooner, than click event
                     If focusout reflows the page, then click event doesn't gets triggered
                     Setting timeout fixes that for most cases
                     */

                    /*jshint loopfunc:true*/
                    callback = (function(callback) {
                        return function() {
                            setTimeout(callback, 50);
                        };
                    })(callback);
                }

                if (!match[3]) {
                    // Можно услышать Custom Events из других блоков, DOM events только внутри блока
                    $target = match[2] in DOM_EVENTS ? this.$el : $(document);
                    $target.on(match[1], callback);
                } else {
                    this.$el.on(match[1], match[3], callback);
                }
            }
        },

        /**
         * Get element within block
         *
         * @param {string} selector
         */
        $: function(selector) {
            return this.$el.find(selector);
        },

        /**
         * Emit an event
         *
         * @param eventName
         * @param [parameters] all the arguments after the eventName get passed to handlers
         */
        // eslint-disable-next-line no-unused-vars
        emit: function(eventName, parameters) {
            $(document).emit(eventName + '.' + this.id, Array.prototype.slice.call(arguments, 1));
        }
    });

    return passport;
});
