(function() {
    var CTRL_SELECTOR = '*:input';

    var VALIDATION_EVENT = 'validation';
    var VALIDATION_FAILED = false;
    var VALIDATION_PASSED = true;

    passport.block('control', {
        /**
         * Indicates that block is control
         */
        isControl: true,

        /**
         * Indicates whether this control may not be empty
         */
        isRequired: true,

        /**
         * Indicates whether the block needs server validation
         */
        needsServerValidation: true,

        /**
         * Checking for update timeout
         */
        updateTimeout: 50,

        /**
         * Helpers to call validation only when the input stops for debounce.period
         */
        debounce: {
            /**
             * Timestamp when the control was changed for the last time
             */
            lastChangedTS: null,

            /**
             * Debounce preiod
             *
             * Validation would be called that much milliseconds since last change
             * If no more changes would happen during this period
             */
            period: 150,

            /**
             * Timer
             *
             * Holds the timeout to validation
             */
            timer: null
        },

        /**
         * Used to determine if value had changed
         */
        oldValue: null,

        /**
         * Keep old request in case it should be aborted
         */
        request: null,

        /**
         * Selector for the control's data input
         * Overrides default CTRL_SELECTOR
         * @type {string}
         * @see CTRL_SELECTOR
         */
        controlSelector: null,

        construct: function() {
            passport.block('block').construct.apply(this, arguments);

            this.$ctrl = this.$(this.controlSelector || CTRL_SELECTOR).eq(0);

            this.debounce.lastChangedTS = new Date('1970-01-01');
            this.oldValue = this.val();

            this.watchUpdates(this.watchSilent);

            this.inited.resolve(this);
            this.inited.already = true; // if (passport.block('myBlock').inited.already) {...}
        },

        /**
         * Gets and sets value of control if control isn't multiply, like jquery val.
         * Gets value of control if control is multiply
         *
         * @param {String} [value]
         * @return $|String|Array
         */

        // eslint-disable-next-line no-unused-vars
        val: function(value) {
            if (this.options && this.options.multiplyFields) {
                var vals = [];

                this.$('input[name="' + this.id + '"]').each(function() {
                    var _val = $(this).val();

                    if (_val && String(_val).trim() !== '') {
                        vals.push(_val);
                    }
                });

                return vals;
            }

            return this.$ctrl.val.apply(this.$ctrl, arguments);
        },

        /**
         * Checks whether the control is empty,
         * unsuitable for checkboxes
         *
         * @returns {Boolean}
         */
        isEmpty: function() {
            var currentValue = this.val();

            return !currentValue || !currentValue.length || String(currentValue).trim() === '';
        },

        /**
         * Gets an id required by backend for validation
         * @returns {String}
         */
        getHandleID: function() {
            return this.$ctrl.attr('name') || this.id;
        },

        /**
         * Gathers parameters to send along with validation request
         * @returns {}
         */
        getValidationParams: function() {
            var params = {};

            params[this.getHandleID()] = this.val();

            return params;
        },

        /**
         * Checks the validation response from server
         * and determines what happens next
         *
         * @param {Object} data
         * @param {Boolean} [suppressError]
         */
        validationCallback: function(data, suppressError) {
            if (data.validation_errors) {
                this.validationResult(
                    VALIDATION_FAILED,
                    passport.validator.getErrorCode(data.validation_errors),
                    suppressError
                );
            } else if (data.errors && data.errors.length) {
                this.validationResult(
                    VALIDATION_FAILED,
                    passport.validator.getErrorCodeFromEntity(data.errors),
                    suppressError
                );
            } else {
                this.validationResult(VALIDATION_PASSED, null, suppressError);
            }

            return !data.validation_errors;
        },

        /**
         * Emits validation event and shows error
         *
         * @param {Boolean} result
         * @param {String} errorCode
         * @param {Boolean} [suppressError]
         */
        validationResult: function(result, errorCode, suppressError) {
            if (!suppressError) {
                this.error(errorCode);
            }

            this.emit(VALIDATION_EVENT, result, errorCode);
        },

        /**
         * Run synchronous validations
         * @returns {string|null}  an error code, or nothing if is valid
         */
        validateSync: function() {
            return null;
        },

        /**
         * Validates the control
         *
         * @param {Boolean} [suppressError]
         * @return {jQuery.Deferred}
         * @fires validation
         */
        validate: function(suppressError) {
            var validated = new $.Deferred();

            if (!this.inited.already) {
                validated.reject();
                return validated;
            }

            var isEmpty = this.isEmpty();

            if (this.isRequired && isEmpty) {
                this.validationResult(VALIDATION_FAILED, 'missingvalue', suppressError);
                validated.resolve(VALIDATION_FAILED);
                return validated;
            }

            if (!isEmpty) {
                var syncCode = this.validateSync();

                if (syncCode) {
                    this.validationResult(VALIDATION_FAILED, syncCode, suppressError);
                    validated.resolve(VALIDATION_FAILED);
                    return validated;
                }
            }

            if (!this.needsServerValidation || isEmpty) {
                this.validationResult(VALIDATION_PASSED, null, suppressError);
                validated.resolve(VALIDATION_PASSED);
                return validated;
            }

            this.request = passport.api.request(
                this.validationPath || this.id,
                $.extend(this.getValidationParams(), {track_id: passport.track_id}),
                {
                    abortPrevious: true,
                    cache: true
                }
            );

            this.request
                .done(
                    function(data) {
                        var result = this.validationCallback(data, suppressError);

                        validated.resolve(result);
                    }.bind(this)
                )
                .fail(function() {
                    validated.reject();
                });

            return validated.promise();
        },

        /**
         * Check whether input had updated
         *
         * @fires update
         */
        checkUpdate: function(silent) {
            var value = this.val();
            var oldValue = this.oldValue;
            var changes = [value, oldValue];
            var hasChanges = false;
            var arrayToCheck = [];
            var arrayToCompare = [];

            if (Array.isArray(value)) {
                arrayToCheck = value.length >= oldValue.length ? value : oldValue;
                arrayToCompare = value.length >= oldValue.length ? oldValue : value;

                arrayToCheck.forEach(function(val, index) {
                    if (val !== arrayToCompare[index]) {
                        hasChanges = true;
                    }
                });
            } else {
                hasChanges = value !== oldValue;
            }

            if (hasChanges) {
                this.oldValue = value;

                //Debounce
                var onChange = function() {
                    this.validate(silent);

                    if (!silent) {
                        // emit before/after values on control
                        this.emit('update', changes);
                    }
                }.bind(this);

                //Clear previous timer
                clearTimeout(this.debounce.timer);

                var now = new Date().getTime();

                if (now - this.debounce.lastChangedTS > this.debounce.period) {
                    //Call instantly, if the wait was long enough
                    onChange();
                } else {
                    //Wait for more input
                    this.debounce.timer = setTimeout(onChange, this.debounce.period);
                }

                this.debounce.lastChangedTS = now;
            }
        },

        /**
         * Watch for input updates
         */
        watchUpdates: function(silent) {
            if (this.updateTimeout) {
                setTimeout(
                    function() {
                        this.checkUpdate(silent);

                        // infinite loop
                        this.watchUpdates(silent);
                    }.bind(this),
                    this.updateTimeout
                );
            }
        },

        /**
         * Hide all errors
         */
        hideErrors: function() {
            this.$('.p-control__error').addClass('g-hidden');
        },

        /**
         * Show a specific error message by code
         * @param {string} code
         */
        showError: function(code) {
            if (typeof code !== 'string') {
                throw new Error('Error code should be a string');
            }

            var error = this.$(
                '.p-control__error__%blockID%_%codeID%'.replace('%blockID%', this.id).replace('%codeID%', code)
            );

            if (!error.length) {
                error = this.$(
                    '.p-control__error__%blockID%_%codeID%'
                        .replace('%blockID%', this.id)
                        .replace('%codeID%', 'globalinternal')
                );
            }

            error.removeClass('g-hidden');
        },

        /**
         * Shows errors by codes,
         * or hides all errors if no code passed
         * @param {String|Array} [code]
         */
        error: function(code) {
            this.hideErrors();

            if (code) {
                var that = this;

                [].concat(code).forEach(function(code) {
                    that.showError(code);
                });
            }
        }
    });
})();
