(function(global, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory(require('putils/libs/xor_cipher'));
    } else {
        global.passport = factory(global.simpleCipher);
    }
})(this, function(simpleCipher) {
    'use strict';

    var passport = {};

    /**
     * Initializes all @data-block on the pages
     * @param {Object} params
     */
    passport.init = function(params) {
        var $body = $('body');
        var $html = $('html');

        passport.basePath = '/registration-validations/';
        passport.language = $('#language').val();
        passport.staticPath = $body.attr('data-static-url');
        passport.soundManagerPath = passport.staticPath + 'js/soundmanager';
        passport.soundManager = window.soundManager;
        passport.track_id = $('#track_id').val() || (params && params.trackId);
        passport.country = $('#country').val();
        passport.metricsId = $body.attr('data-metrics-id');
        passport.isTouch = $html.hasClass('is-touch');
        passport.isTablet = $html.hasClass('is-tablet');
        passport.sk = $('#sk').val();
        passport.csrf_token = $body.data('csrf') || (params && params.csrf);
        passport.lastUsedTrackId = passport.track_id;

        $('*[data-block]').each(function() {
            var block = passport.ensureBlock($(this).attr('data-block'));

            if (block) {
                block.construct(this, $(this).data('options'));
                if (block.isControl) {
                    passport.validator.register(block);
                }
            }
        });

        passport.setupDebugAPI();
    };

    passport.setupDebugAPI = function() {
        var Passport = (window.Passport = window.Passport || {});

        if (Passport.isYandex) {
            Passport.debug = {
                getTrackID: function() {
                    return passport.lastUsedTrackId;
                }
            };
        }
    };

    /**
     * Ensures block by id or by complex name (parent_name)
     * So it returns or creates block
     *
     * @param {String} id
     * @returns block
     */
    passport.ensureBlock = function(id) {
        var split = id.split('_');
        var target = split.pop();
        var parent = split.pop();

        var block = passport.block(target);

        if (!block) {
            block = parent ? passport.block(target, parent, {}) : passport.block(target, {});
        }

        return block;
    };

    var spacesRegexp = /\s+/g;
    // eslint-disable-next-line no-useless-escape
    var emailRegexp = /^([a-zа-я0-9\+\.-]+)@([a-zа-я0-9-\.]+)(\.)([a-zа-я])+$/i;
    var passRegexp = /[^ -~]/;

    passport.util = {
        entityMap: {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#39;',
            '/': '&#x2F;'
        },

        escape: function(str) {
            // eslint-disable-next-line no-useless-escape
            return String(str).replace(/[&<>"'\/]/g, function(s) {
                return passport.util.entityMap[s];
            });
        },

        keymap: {
            backspace: 8,
            enter: 13,
            esc: 27,
            tab: 9,
            cmd: 224,
            'webkit-cmd': 91,
            up: 38,
            down: 40
        },

        getUrlInfo: function(url) {
            var a = document.createElement('a');

            a.href = url;
            return {
                hash: a.hash,
                host: a.host,
                hostname: a.hostname,
                href: a.href,
                origin: a.origin,
                pathname: a.pathname,
                port: a.port,
                protocol: a.protocol,
                search: a.search
            };
        },

        getUrlParam: function(param) {
            var params = document.location.search.substr(1).split('&');

            for (var i = 0, len = params.length; i < len; i++) {
                var singleParam = params[i].split('=');

                if (singleParam[0] === param) {
                    return decodeURIComponent(singleParam[1]);
                }
            }

            return '';
        },

        getUrlParams: function(url) {
            var _query = {};
            var urlSearchString = (typeof url === 'string' && url.split('?')[1]) || document.location.search.substr(1);
            var params = urlSearchString.split('&');

            if (!params.length) {
                return;
            }

            params.forEach(function(param) {
                var split = param.split('=');

                _query[split.shift().trim()] = split.join('=');
            });

            return _query;
        },

        getUrlParamsV2: function(url) {
            url = url || document.location.href;
            var query = url.split('?')[1];
            var params = query ? query.split('&') : [];
            var _query = {};

            if (!params.length) {
                return;
            }

            params.forEach(function(param) {
                var split = param.split('=');

                _query[split.shift().trim()] = split.join('=');
            });

            return _query;
        },

        getKeyCode: function(event) {
            if (event) {
                return event.keyCode || event.which;
            }

            return null;
        },

        isKeyboardEvent: function(event) {
            return (
                passport.util.getKeyCode(event) !== null && ['keydown', 'keyup', 'keypress'].indexOf(event.type) > -1
            );
        },

        isKeyboardSubmit: function(event) {
            return [13, 32].indexOf(passport.util.getKeyCode(event)) > -1;
        },

        isEmail: function(str) {
            return emailRegexp.test(passport.util.normalize(str));
        },

        /**
         * Underscores _.once
         *
         * Calls function only once
         *
         * @param {Function} func   Function to be augmented
         * @returns {Function}
         */
        once: function(func) {
            var ran = false;
            var memo;

            return function() {
                if (ran) {
                    return memo;
                }

                ran = true;
                memo = func.apply(this, arguments);
                func = null;
                return memo;
            };
        },

        /**
         * Abort request, if possible
         * @param {jqXHR} request
         * @private
         */
        abortRequest: function(request) {
            if (request && request.readyState !== 4 && 'abort' in request) {
                request.abort();
            }
        },

        /**
         * Capitalizes the first symbol of the string
         * @param {string} string
         * @returns {string}
         */
        capitalize: function(string) {
            return string.charAt(0).toUpperCase() + string.slice(1);
        },

        /**
         * Trim leading and ending spaces, replace all the double spaces with single ones
         * @param {*} str   A value to normalize
         * @returns {string}
         */
        normalize: function(str) {
            if (typeof str !== 'string') {
                if (typeof str === 'number') {
                    return String(str);
                }

                return '';
            }

            return str.replace(spacesRegexp, ' ').trim();
        },

        /**
         * Hack for ie showing an element with background-image even though it has display: none
         * @param {jQuery} $el
         */
        reattach: function($el) {
            var parent = $el.parent();

            $el.detach();
            parent.append($el);
        },

        /**
         * проверка корретности указанной даты
         * @param {string} date yyyy-mm-dd
         * @returns {boolean}
         */
        isCorrectDate: function(date) {
            var splitedDate = date.split('-');
            var _date = new Date(date + 'T08:00:00+00:00'); // hack for PST timezone

            return _date.getMonth() + 1 === parseInt(splitedDate[1], 10);
        },

        /**
         * возвращает значение cookie по имени
         * @param {string} code
         * @returns {string} cookie value or empty string
         */
        getCookie: function(code) {
            if (!document.cookie) {
                return;
            }

            var _cookies = {};
            var cookiesArr = document.cookie.split(';');

            cookiesArr.forEach(function(item) {
                var split = item.split('=');

                _cookies[split.shift().trim()] = split.join('=');
            });

            return _cookies[code] || '';
        },

        /**
         * проверка валидности пароля согласно требованиям паспорта
         * @param {string} pass
         * @returns {boolean} содердит ли пароль запрещенные символы
         */
        checkPassword: function(pass) {
            return passRegexp.test(pass);
        },

        omitEmpty: function(obj) {
            Object.keys(obj).forEach(function(key) {
                if (!obj[key]) {
                    delete obj[key];
                }
            });

            return obj;
        }
    };

    passport.api = (function() {
        /**
         * Storage for the previous requests aggregated by the methods and then by serialized params.
         * Used for caching.
         *
         * @type {object.<string, object.<string, promise>>}
         * @example {
         *      "phone": {
         *          "{phone_number896535,track_id6fbc316eed0bb97220c26651301dac9b50}":
         *          <$.Deferred.promise for the response>
         *      }
         * }
         *
         * @private
         */
        var _cachedPromises = {};

        /**
         * Storage for the last request by method to easily abort previous requests
         * @type {object.<string, jqXhr>}
         * @example {
         *      "phone": <jqXhr for the last request>
         * }
         *
         * @private
         */
        var _lastRequests = {};

        /**
         * Serialize an object into a suitable key
         *
         * @param {Object} obj
         * @returns {string}
         * @private
         */
        function _serialize(obj) {
            var serialized = [];

            if ($.isPlainObject(obj)) {
                for (var field in obj) {
                    if (obj.hasOwnProperty(field)) {
                        serialized.push(field + _serialize(obj[field]));
                    }
                }

                return '{' + serialized.join(',') + '}';
            }

            if ($.isArray(obj)) {
                obj.forEach(function(item) {
                    serialized.push(_serialize(item));
                });

                return '[' + serialized.join(',') + ']';
            }

            if (!obj) {
                return '';
            }

            return obj.toString();
        }

        return {
            /**
             * Schedule a log a message to be sent to the server
             *
             * @param {string|Object}   message                     Message to log
             * @param {Object}          [options]                   Log options
             * @param {string}          [options.loglevel=info]     Log level for the message
             * @param {boolean}         [options.encrypt=false]     Encrypt message using simple xor cipher
             */
            log: function(message, options) {
                /*
                 Logging should:
                 * Save timestamp, message, track, token and url
                 * Keep the messages queue to send messages in bulk, if there's a lot
                 * Wait for the request to api to send the messages with
                 * When timed out, message queue should be sent with a separate request
                 * Dropping the queue should unfreeze the timer

                 Request should:
                 * get the logs from getQueue
                 * drop the queue if request was actually sent
                 * Freeze the timer so logs aren't sent by timeout while there request is pending.
                 * Unfreeze the timer once the request failed

                 */
                var json;
                var defaultOptions = {
                    loglevel: 'info',
                    logUrl: 'logger',
                    encrypt: false
                };

                var trackId = message.track_id || passport.track_id;

                options = $.extend({}, defaultOptions, options);
                var commonInfo = {
                    ev_tsf: new Date().getTime(),
                    loglevel: options.loglevel,
                    uid: window.uid,
                    track_id: trackId,
                    url: window.location.href
                };

                if ($.isPlainObject(message)) {
                    json = JSON.stringify($.extend({}, commonInfo, message));
                } else {
                    json = JSON.stringify($.extend({}, commonInfo, {action: message}));
                }
                if (options.encrypt) {
                    json = simpleCipher.encode(trackId, json);
                }
                passport.api.request(options.logUrl, {log: json, track_id: trackId});
            },

            /**
             * Make a request to the Passport API
             *
             * @param {string|object}   method                                 Api url or url and method object
             * @param {object}          params                              Parameters passed to the api
             * @param {object}          [options]                           Request options
             * @param {number}          [options.timeout=30000]             **not implemented** Time to wait for the
             * request to api to send the log with. If no api requests are made during that time,
             * log would be sent separately.
             * @param {boolean}         [options.cache=false]               should request be resolved with
             * previous api response for the same parameters?
             * @param {boolean}         [options.abortPrevious=false]       should previous request to the same
             * api method be aborted?
             */
            request: function(method, params, options) {
                var apiHandleParams = {
                    type: 'POST'
                };
                var cacheId;

                if (!method || !(typeof method === 'string' || method.url)) {
                    throw new Error('Method should be string or object with url');
                }

                if (!params || (!(params instanceof FormData) && !$.isPlainObject(params))) {
                    throw new Error('Params should be a plain object or FormData');
                }

                if (typeof method === 'string') {
                    cacheId = method;
                    apiHandleParams.url = method;
                } else {
                    cacheId = method.url;
                    apiHandleParams.url = method.url;

                    if (method.type) {
                        apiHandleParams.type = method.type;
                    }
                }

                options = options || {};
                var deferred = new $.Deferred();

                if (options.abortPrevious) {
                    passport.util.abortRequest(_lastRequests[cacheId]);
                }

                var cached;

                if (options.cache) {
                    //Establish the structure, if it is not present yet.
                    _cachedPromises[cacheId] = _cachedPromises[cacheId] || {};
                    cached = _cachedPromises[cacheId][_serialize(params)];
                }

                var settings = {
                    type: apiHandleParams.type,
                    url: method.match(/^(\/|https:\/\/)/) ? method : passport.basePath + method,
                    data:
                        params instanceof FormData
                            ? params
                            : $.extend({}, {track_id: passport.track_id, csrf_token: passport.csrf_token}, params),
                    dataType: 'json'
                };

                passport.lastUsedTrackId = settings.data.track_id;

                if (options.processData !== undefined) {
                    settings.processData = options.processData;
                }

                if (options.contentType !== undefined) {
                    settings.contentType = options.contentType;
                }

                var promise = cached || $.ajax(settings);

                promise
                    .done(function(data) {
                        var result;
                        //Cached requests should be independent.
                        //Cloning the data provides isolation in case the response is modified

                        if ($.isArray(data)) {
                            result = data.slice();
                        } else {
                            result = $.extend({}, data);
                        }

                        if (result.track_id) {
                            passport.lastUsedTrackId = result.track_id;
                        }

                        deferred.resolve(result);
                    })
                    .fail(function(err) {
                        deferred.reject(err);
                    });

                if (options.cache) {
                    //Store the promise
                    _cachedPromises[cacheId][_serialize(params)] = deferred.promise();
                    return promise;
                }

                if (options.abortPrevious) {
                    //Store this request. In that case "promise" contains jqXHR;
                    _lastRequests[cacheId] = promise;
                }

                return deferred.promise();
            },

            /**
             * Drops existing caches and previous requests
             */
            dropCaches: function() {
                _lastRequests = {};
                _cachedPromises = {};
            },

            getTrackId: function(type, isInit) {
                var def = new $.Deferred();
                var that = this;

                this.request(
                    'track',
                    passport.util.omitEmpty({
                        type: type,
                        passErrors: true,
                        sk: passport.sk,
                        isInit: isInit
                    }),
                    {
                        abortPrevious: false,
                        cache: false
                    }
                ).then(
                    function(res) {
                        if (res && (res.id || res.track_id)) {
                            that.setTrackId(res.id || res.track_id);
                        }

                        def.resolve(res);
                    },
                    function(err) {
                        def.reject(err);
                    }
                );

                return def.promise();
            },

            setTrackId: function(trackId) {
                if (!trackId) {
                    return;
                }

                passport.track_id = trackId;
                $('#track_id').val(trackId);
            },

            setCsrfToken: function(csrfToken) {
                if (!csrfToken) {
                    return;
                }

                passport.csrf_token = csrfToken;
            }
        };
    })();

    return passport;
});
