// @see https://lego.yandex-team.ru/components/lego/latest/?storybook=%2Fdocs%2Fsurface-popup-desktop--playground
import './Popup.styl';
import React from 'react';
import PropTypes from 'prop-types';
import {compose} from '@bem-react/core';
import {Popup as PopupDesktop, withViewDefault, withTargetAnchor} from '@yandex-lego/components/Popup/desktop';
import {withViewBlack} from './view/black';
import {withModeBlind} from './mode/blind';

// Композиция из различных модификаторов
const PopupCompose = compose(withViewDefault, withViewBlack, withModeBlind)(PopupDesktop);
const PopupComposeWithTargetAnchor = compose(
    withViewDefault,
    withViewBlack,
    withModeBlind,
    withTargetAnchor
)(PopupDesktop);

export class Popup extends React.PureComponent {
    maybePreventScroll = () => {
        const {mode, visible} = this.props;

        // не шторка, нет смысла превентить скролл
        if (mode !== 'blind' || typeof window === 'undefined') {
            return;
        }

        visible && !this.isPreventBinded ? this.bindPreventScroll() : this.unbindPreventScroll();
    };

    bindPreventScroll = () => {
        if (this.isPreventBinded) {
            return;
        }

        if (this.wheelOpt === undefined) {
            let supportsPassive = false;

            try {
                window.addEventListener(
                    'test',
                    null,
                    Object.defineProperty({}, 'passive', {
                        get: () => (supportsPassive = true)
                    })
                );
            } catch (error) {
                throw error;
            }

            this.wheelOpt = supportsPassive ? {passive: false} : false;
            this.wheelEvent = 'onwheel' in document.createElement('div') ? 'wheel' : 'mousewheel';
        }

        window.addEventListener('touchmove', this.preventDefault, this.wheelOpt);
        window.addEventListener(this.wheelEvent, this.preventDefault, this.wheelOpt);
        this.isPreventBinded = true;
    };

    unbindPreventScroll = () => {
        window.removeEventListener('touchmove', this.preventDefault, this.wheelOpt);
        window.removeEventListener(this.wheelEvent, this.preventDefault, this.wheelOpt);
        this.isPreventBinded = false;
    };

    preventDefault = (event) => {
        event.preventDefault();
    };

    componentDidMount() {
        this.maybePreventScroll();
    }

    componentDidUpdate() {
        this.maybePreventScroll();
    }

    componentWillUnmount() {
        this.unbindPreventScroll();
    }

    render() {
        const {dataT, view = 'default', directions, mode, visible, anchor, ...rest} = this.props;
        const popupDirections = mode === 'blind' ? ['bottom-center'] : directions;
        const PopupComponent = anchor ? PopupComposeWithTargetAnchor : PopupCompose;

        return (
            <PopupComponent
                view={view}
                data-t={dataT ? `popup:${dataT}` : 'popup'}
                directions={popupDirections}
                visible={visible}
                mode={mode}
                anchor={anchor}
                {...rest}
            />
        );
    }
}

Popup.propTypes = {
    view: PropTypes.oneOf(['default', 'black', 'mobile']),
    dataT: PropTypes.string,
    onClose: PropTypes.func,
    mode: PropTypes.oneOf(['blind', '']),
    visible: PropTypes.bool,
    directions: PropTypes.array,
    anchor: PropTypes.object
};
