var url = require('url');
var _ = require('lodash');
var util = require('util');
var assert = require('assert');
var inherit = require('inherit');

var ExpressController = require('./ExpressController');
var RetryCondition = require('pdao').RetryCondition;
var portNumberRE = /:\d+/;
var AuthError = require('./AuthError');
var PdaoHttp = require('pdao/Http');

var BlackboxErrorTypes = {
    INVALID_PARAMS: 2,
    DB_FETCHFAILED: 9,
    DB_EXCEPTION: 10,
    ACCESS_DENIED: 21,
    UNKNOWN: 1
};

var BB_ATTRIBUTES = {
    GLOBAL_LOGOUT_TIME: '4',
    ACCOUNT_IS_CORPORATE: '9',
    ACCOUNT_ENABLE_APP_PASSWORD: '107',
    ACCOUNT_2FA_ON: '1003',
    REVOKER_TOKENS: '134',
    REVOKER_APP_PASSWORDS: '135',
    REVOKER_WEB_SESSIONS: '136'
};

var BB_ATTRIBUTE_DEFAULTS = {
    GLOBAL_LOGOUT_TIME: 0,
    ACCOUNT_IS_CORPORATE: false,
    ACCOUNT_ENABLE_APP_PASSWORD: false,
    ACCOUNT_2FA_ON: false,
    REVOKER_TOKENS: 0,
    REVOKER_APP_PASSWORDS: 0,
    REVOKER_WEB_SESSIONS: 0
};

var getBlackboxErrorType = function(bbResponse) {
    if (!bbResponse || !bbResponse.exception) {
        return null;
    }

    var id = bbResponse.exception.id;

    if (_.values(BlackboxErrorTypes).indexOf(id) > -1) {
        return id;
    }

    return BlackboxErrorTypes.UNKNOWN;
};

class AuthController extends ExpressController {
    constructor(req, res, logId, tvmServiceAliases, serviceName) {
        super(req, res, logId, tvmServiceAliases, serviceName);

        this._blackboxPath = null;
        this._lastRequestResult = null;
        this._lastRequestPromise = null;
        this._lastRequestAttributes = [];

        this._logger.type('auth');

        this._pinningOn = false;
        this._tvm_service_aliases = tvmServiceAliases;
    }

    _getPinnedUid() {
        var param = this.getRequestParam('uid');
        var uid = null;

        if (param && this._pinningOn) {
            if (Array.isArray(param)) {
                param = param[0];
            }

            uid = param.match(/^\d+/);

            if (Array.isArray(uid)) {
                uid = uid[0];
            }
        }

        return uid;
    }

    _pinUrl(url) {
        var pinnedUid = this._getPinnedUid();

        if (pinnedUid) {
            var separator = url.indexOf('?') > -1 ? '&' : '?';

            url += separator + 'uid=' + encodeURIComponent(pinnedUid);
        }

        return url;
    }

    /**
     * Get a blackbox url
     */
    getBlackbox() {
        if (this._blackboxPath === null) {
            throw new Error('Set the blackbox path with auth.setBlackbox(path)');
        }

        return this._blackboxPath;
    }

    getBlackboxAO() {
        if (this._blackboxAO) {
            return this._blackboxAO;
        }

        this.updateBlackboxAO();
        return this._blackboxAO;
    }

    updateBlackboxAO() {
        this._blackboxAO = this.constructor.getBlackboxAO(this.getLogId(), 'http://' + this.getBlackbox());
    }

    setBlackbox(path) {
        if (this._blackboxPath !== path) {
            this._blackboxPath = path;
            this.updateBlackboxAO();
        }

        return this;
    }

    togglePinning(onOff) {
        this._pinningOn = onOff;
    }

    getDefaultQueryParams() {
        return {
            authid: 'yes',
            regname: true,
            format: 'json'
        };
    }

    getDefaultHeaders() {
        const ticket =
            this._request &&
            this._request.serviceTickets &&
            this._request.serviceTickets[
                this._tvm_service_aliases[`BLACKBOX${this.getBlackbox().includes('mimino') ? '_MIMINO' : ''}`]
            ];

        if (!ticket) {
            return {};
        }

        return {
            'X-Ya-Service-Ticket': ticket
        };
    }

    getLastRequestPromise() {
        return this._lastRequestPromise;
    }

    simpleSessionID(queryparams, headers) {
        queryparams = Object.assign(this.getDefaultQueryParams(), queryparams);
        headers = Object.assign(this.getDefaultHeaders(), headers);

        var that = this;
        var ip = that.getIp();
        var httpCookie = this.getCookie('Session_id');
        var secureCookie = this.getCookie('sessionid2');
        var sessguardCookie = this.getCookie('sessguard');
        var params = Object.assign({}, queryparams, {
            sessionid: httpCookie,
            sslsessionid: secureCookie,
            sessguard: sessguardCookie,
            userip: ip,
            format: 'json',
            host: this.getHeader('host')
        });

        var blackboxAO = this.getBlackboxAO();

        var config = blackboxAO.cloneConfig();

        if (headers) {
            config.setExplicitHeaders(headers);
        }

        return that
            .getBlackboxAO()
            .call('get', 'sessionid', params, config)
            .then(function(response) {
                if (response.exception) {
                    return Promise.reject(response.exception);
                }

                return response;
            })
            .catch(function(err) {
                return Promise.reject(err);
            });
    }

    userInfo(queryparams, headers) {
        var that = this;
        var ip = that.getIp();
        var params = Object.assign({}, queryparams, {
            userip: ip,
            format: 'json'
        });

        delete params.returnAllUsers;

        headers = Object.assign(this.getDefaultHeaders(), headers);

        var blackboxAO = this.getBlackboxAO();

        var config = blackboxAO.cloneConfig();

        if (headers) {
            config.setExplicitHeaders(headers);
        }

        return that
            .getBlackboxAO()
            .call('get', 'userinfo', params, config)
            .then(function(response) {
                if (response.exception) {
                    return Promise.reject(response.exception);
                }

                return response.users && (queryparams.returnAllUsers ? response.users : response.users[0]);
            })
            .catch(function(err) {
                return Promise.reject(err);
            });
    }

    sessionID(queryparams, headers) {
        queryparams = Object.assign(this.getDefaultQueryParams(), queryparams);
        headers = Object.assign(this.getDefaultHeaders(), headers);

        var logger = this._logger;
        var comparingParams = Object.assign({}, queryparams, headers);

        if (this._lastRequestPromise && _.isEqual(this._lastRequestedParams, comparingParams)) {
            //Already made the request, no need to do that again
            logger.debug('Blackbox request already made with query params %j, returning cached promise', queryparams);
            return this._lastRequestPromise;
        }

        this._lastRequestedParams = comparingParams;

        var that = this;

        var httpCookie = this.getCookie('Session_id');
        var secureCookie = this.getCookie('sessionid2');
        var sessguardCookie = this.getCookie('sessguard');

        if (!httpCookie) {
            logger.info('No Session_id cookie');
            this._lastRequestPromise = Promise.resolve(null); //User is not logged in if the cookie is empty

            return this._lastRequestPromise;
        }

        var ip = this.getIp();
        var params = Object.assign({}, queryparams, {
            sessionid: httpCookie,
            sslsessionid: secureCookie,
            sessguard: sessguardCookie,
            userip: ip,

            // blackbox doesn't handle port number in the host param
            host: this.getHeader('host').replace(portNumberRE, '')
        });

        var pinnedUid = this._getPinnedUid();
        // if (uid) {
        //     params['default_uid'] = uid;
        // }

        var blackboxAO = this.getBlackboxAO();

        var config = blackboxAO.cloneConfig();

        if (headers) {
            config.setExplicitHeaders(headers);
        }

        // logger.info('Making a request to blackbox for ip %s, query params %j',
        // ip, queryparams, secureCookie ? 'and secure cookie' : 'without secure cookie');
        this._lastRequestPromise = blackboxAO.call('get', 'sessionid', params, config).then(function(response) {
            var users = response && response.users;
            var user;
            var userUid;
            var i;

            if (!response) {
                logger.info('Blackbox responded with an empty message');
                return null;
            }

            if (that.constructor.getBlackboxErrorType(response) !== null) {
                logger.error('Blackbox responded with an error', response);
                throw new that.constructor.BlackboxException(response);
            } else {
                var status = that.constructor.getBlackboxStatus(response);

                if (status === that.constructor.BlackboxStatuses.VALID) {
                    logger.info('Blackbox status was VALID — user is logged in');
                    response.initial_default_uid = response.default_uid;

                    if (pinnedUid && users && users.length) {
                        i = users.length;

                        while (i--) {
                            user = users[i];
                            userUid = user.uid && user.uid.value;

                            if (pinnedUid === userUid) {
                                response.default_uid = pinnedUid;
                                break;
                            }
                        }
                    }

                    that._lastRequestResult = response;

                    //Attributes are requested as a comma-delimeted string
                    //So the default value is a string
                    that._lastRequestAttributes =
                        typeof queryparams.attributes === 'string' ? queryparams.attributes.split(',') : [];

                    return response;
                }

                if (status === AuthController.BlackboxStatuses.WRONG_GUARD) {
                    logger.info(
                        'Blackbox status is WRONG_GUARD in a root mda domain — redirect to passport /auth/guard'
                    );
                    return Promise.reject(new AuthError('Wrong Guard', 'wrong_guard'));
                }

                if (status === AuthController.BlackboxStatuses.NEED_RESET) {
                    logger.info('Blackbox status is NEED_RESET in a root mda domain — proceeding to resign');
                    return Promise.reject(new AuthError('Need resign', 'need_resign'));
                } else {
                    logger.info('Blackbox status was %s in a root mda domain — user not logged in', status);
                    return null;
                }
            }
        });

        return this._lastRequestPromise;
    }

    /**
     * Determine if current user is logged in based on cookies of the request
     * @returns {when.promise}
     */
    loggedIn() {
        const {serviceTickets = {}} = this._request;
        const headers = {'X-Ya-Service-Ticket': serviceTickets[this._tvm_service_aliases.BLACKBOX]};

        return this.sessionID({}, headers)
            .then((blackbox) => Boolean(blackbox))
            .catch((err) => {
                if (err.code === 'need_resign') {
                    this.resign();
                }

                if (err.code === 'wrong_guard') {
                    this.wrongGuard();
                }

                return Promise.reject(err);
            });
    }

    /**
     * Whether it is known that user is logged in.
     * Use after the loggedIn method was resolved.
     * @returns {boolean}
     */
    isLoggedIn() {
        return Boolean(this.getUid());
    }

    _getLastLoginInfo() {
        return this._lastRequestResult || {};
    }

    attributeWasRequested(attribute) {
        attribute = String(attribute);
        assert(_.includes(AuthController.BB_ATTRIBUTES, attribute), 'Unknown attribute');

        return this._lastRequestAttributes.indexOf(attribute) > -1;
    }

    getAttribute(attribute) {
        attribute = String(attribute);
        assert(this.attributeWasRequested(attribute), 'Attribute was not requested');

        var attributes = this._getLastLoginInfo().attributes || {};

        if (attribute in attributes) {
            return attributes[attribute];
        }

        var key = _.findKey(AuthController.BB_ATTRIBUTES, function(val) {
            return String(val) === attribute;
        });

        return AuthController.BB_ATTRIBUTE_DEFAULTS[key];
    }

    getUid() {
        return this._getLastLoginInfo().default_uid || (this._getLastLoginInfo().uid || {}).value;
    }

    isHosted() {
        return Boolean((this._getLastLoginInfo().uid || {}).hosted);
    }

    isLite() {
        return Boolean((this._getLastLoginInfo().uid || {}).lite);
    }

    hasPassword() {
        return Boolean(this._getLastLoginInfo().have_password);
    }

    getPasswordEntryDate() {
        var auth = this._getLastLoginInfo().auth;

        if (
            !_.isObjectLike(auth) ||
            !Number.isFinite(auth.password_verification_age) ||
            auth.password_verification_age === -1
        ) {
            return null;
        }

        var authid = this._getLastLoginInfo().authid;

        assert(authid, 'This method should be used with `authid=yes` in SessionID request');

        return new Date(authid.time - auth.password_verification_age * 1000);
    }

    isAutologged() {
        return this.isLite() && this.getPasswordEntryDate() === null;
    }

    getLogin() {
        var lastLoginInfo = this._getLastLoginInfo();
        var login = lastLoginInfo.login;
        var users = lastLoginInfo.users;
        var i;
        var uid;
        var user;

        if (login) {
            return login;
        }

        if (users && users.length) {
            i = users.length;

            while (i--) {
                user = users[i];
                uid = user.uid && user.uid.value;

                if (lastLoginInfo.default_uid === uid) {
                    return user.login;
                }
            }
        }

        return null;
    }

    getDisplayName() {
        return (this._getLastLoginInfo().display_name || {}).name;
    }

    isSecure() {
        return Boolean((this._getLastLoginInfo().auth || {}).secure);
    }

    /**
     * Returns the 4th attribute as is
     * IMPORTANT: ask for 4th attribute via session id, otherwise no value
     */
    getGLogoutTime() {
        return this.getAttribute(this.BB_ATTRIBUTES.GLOBAL_LOGOUT_TIME);
    }

    getTokensRevokeTime() {
        return this.getAttribute(this.BB_ATTRIBUTES.REVOKER_TOKENS);
    }

    getAppPasswordsRevokeTime() {
        return this.getAttribute(this.BB_ATTRIBUTES.REVOKER_APP_PASSWORDS);
    }

    getWebSessionsRevokeTime() {
        return this.getAttribute(this.BB_ATTRIBUTES.REVOKER_WEB_SESSIONS);
    }

    getPasswordVerificationAge() {
        var auth = this._getLastLoginInfo().auth;

        if (
            !_.isObjectLike(auth) ||
            !Number.isFinite(auth.password_verification_age) ||
            auth.password_verification_age === -1
        ) {
            return null;
        }

        return auth.password_verification_age;
    }

    canChangePassword() {
        var type = 'canChangePassword';

        if (!this.isHosted()) {
            var hasPassword = this.hasPassword();

            this._logger
                .info()
                .type(type)
                .write(
                    'Not a hosted user %s',
                    hasPassword ? 'with a password, no restrictions' : 'without a password, cannot change it'
                );
            return Promise.resolve(hasPassword);
        }

        var uid = this._getLastLoginInfo().uid;

        if (!uid) {
            return Promise.reject(new Error('No uid info, has the sessionid request been made?'));
        }

        var blackboxAO = this.getBlackboxAO();
        var headers = Object.assign({}, this.getDefaultHeaders());
        var config = blackboxAO.cloneConfig();

        config.setExplicitHeaders(headers);

        this._logger
            .debug()
            .type(type)
            .write('Requesting blackbox');

        return this.getBlackboxAO()
            .call(
                'post',
                'hosted_domains',
                {
                    domain_id: uid.domid,
                    domain: uid.domain,
                    format: 'json'
                },
                config
            )
            .then(function(response) {
                var hostOptions =
                    response.hosted_domains && response.hosted_domains[0] && response.hosted_domains[0].options;

                if (!hostOptions) {
                    return true; //True by default
                }

                return Boolean(hostOptions.can_users_change_password);
            });
    }

    resign() {
        var resignUrlObj;
        var resignUrl;
        var passportHost;

        const host = this.getHeader('host');
        const domain = host.replace(/^(.*)\.([^.]+\.[^.]+)$/g, '$2');

        passportHost = 'passport.' + domain;

        if (process.env.NODE_ENV === 'testing') {
            this._logger.info('Testing, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = 'passport-test.' + domain;
        }

        if (process.env.NODE_ENV === 'development') {
            this._logger.info('Development, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = '0.passportdev.' + domain;
        }

        if (process.env.NODE_ENV === 'rc') {
            this._logger.info('RC, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = 'passport-rc.' + domain;
        }

        if (process.env.NODE_ENV === 'stress') {
            this._logger.info('stress, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = 'passport-load.' + domain;
        }

        if (this.getRequestParam('nocookiesupport') === 'yes') {
            this._logger.info('No cookie support, can not resign via MDA');

            const query = {mode: 'error', error: 'nocki'};

            if (this._serviceName === 'oauth') {
                query.origin = this.getRequestParam('origin') || 'oauth';
            }

            return this.redirect(
                url.format({
                    protocol: 'https',
                    hostname: passportHost,
                    pathname: 'passport',
                    query
                })
            );
        }

        this._logger.info('Settings Cookie_check for domain %s', domain);
        this.setCookie('Cookie_check', 'CheckCookieCheckCookie', {
            domain: domain,
            path: '/'
        });

        resignUrlObj = {
            protocol: 'https',
            host: passportHost,
            pathname: 'auth/update',
            query: {
                retpath: this.getUrl().href
            }
        };

        resignUrl = url.format(resignUrlObj);
        resignUrl = this._pinUrl(resignUrl);

        this._logger.info('Resigning at %s', resignUrl);
        return this.redirect(resignUrl);
    }

    wrongGuard() {
        let domain = '.yandex.';

        if (process.env.INTRANET === 'intranet') {
            //TODO: hosts should be configurable properly
            this._logger.info('Intranet, process.env.INTRANET=%s', process.env.INTRANET);
            domain = '.yandex-team.';
        }
        domain += this.getTld();

        let passportHost = 'passport' + domain;

        if (['development', 'testing'].includes(process.env.NODE_ENV)) {
            const envType = process.env.NODE_ENV === 'development' ? 'Development' : 'Testing';

            this._logger.info(`${envType}, process.env.NODE_ENV=${process.env.NODE_ENV}`);
            passportHost = 'passport-test' + domain;
        }

        if (process.env.NODE_ENV === 'stress') {
            this._logger.info('Stress, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = 'passport-load' + domain;
        }

        if (process.env.NODE_ENV === 'rc') {
            this._logger.info('RC, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            passportHost = 'passport-rc' + domain;
        }

        if (this.getRequestParam('nocookiesupport') === 'yes') {
            this._logger.info('No cookie support, can not redirect to auth/guard via MDA');

            const query = {mode: 'error', error: 'nocki'};

            if (this._serviceName === 'oauth') {
                query.origin = this.getRequestParam('origin') || 'oauth';
            }

            return this.redirect(
                url.format({
                    protocol: 'https',
                    hostname: passportHost,
                    pathname: 'passport',
                    query
                })
            );
        }

        this._logger.info('Settings Cookie_check for domain %s', domain);
        this.setCookie('Cookie_check', 'CheckCookieCheckCookie', {
            domain,
            path: '/'
        });

        const wrongGuardUrlObj = {
            protocol: 'https',
            host: passportHost,
            pathname: 'auth/guard',
            query: {
                retpath: this.getUrl().href
            }
        };

        const wrongGuardUrl = this._pinUrl(url.format(wrongGuardUrlObj));

        this._logger.info('Redirecting with WRONG_GUARD to %s', wrongGuardUrl);
        return this.redirect(wrongGuardUrl);
    }

    _getPassportHost() {
        var host = 'passport.yandex.';

        if (process.env.INTRANET === 'intranet') {
            //TODO: hosts should be configurable properly
            if (process.env.NODE_ENV === 'testing') {
                this._logger.info(
                    'Intranet-testing, process.env.INTRANET=%s, process.env.NODE_ENV=%s',
                    process.env.INTRANET,
                    process.env.NODE_ENV
                );
                host = 'passport-test.yandex-team.';
            } else {
                this._logger.info('Intranet, process.env.INTRANET=%s', process.env.INTRANET);
                host = 'passport.yandex-team.';
            }
        } else if (process.env.NODE_ENV === 'development') {
            this._logger.info('Development, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            host = '0.passportdev.yandex.';
        } else if (process.env.NODE_ENV === 'testing') {
            this._logger.info('Testing, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            host = 'passport-test.yandex.';
        } else if (process.env.NODE_ENV === 'rc') {
            this._logger.info('RC, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            host = 'passport-rc.yandex.';
        } else if (process.env.NODE_ENV === 'stress') {
            this._logger.info('stress, process.env.NODE_ENV=%s', process.env.NODE_ENV);
            host = 'passport-load.yandex.';
        }
        host += this.getTld();
        return host;
    }

    /**
     * Redirect regular user with a light cookie to password entry page with retpath to current url
     */
    lightauth2full(noreturn) {
        const url = `https://${this._getPassportHost()}/?mode=lightauth2full${
            this._serviceName === 'oauth' ? `&origin=${this.getRequestParam('origin') || 'oauth'}` : ''
        }&retpath=${encodeURIComponent(this.getUrl().href)}${noreturn ? '&noreturn=1' : ''}`;

        this._logger.info('LightAuth2Full at %s', url);
        this.redirect(url);
    }

    /**
     * Redirect the user to authorization page with retpath to current url
     */
    authorize(queryparams) {
        assert(!queryparams || _.isObjectLike(queryparams), 'Query params should be a plain object if defined');

        let retpath = this.getUrl().href;

        if (this._serviceName === 'oauth') {
            const retpathObj = url.parse(retpath, true);

            if (retpathObj && retpathObj.query && retpathObj.query.login_hint) {
                delete retpathObj.query.login_hint;

                retpath = url.format({
                    protocol: 'https',
                    host: retpathObj.host,
                    pathname: retpathObj.pathname,
                    query: retpathObj.query
                });
            }
        }

        const query = {retpath, ...queryparams};

        if (this._serviceName === 'oauth') {
            query.origin = this.getRequestParam('origin') || 'oauth';
        }

        const authUrl = this._pinUrl(
            url.format({
                protocol: 'https',
                hostname: this._getPassportHost(),
                pathname: 'auth',
                query
            })
        );

        this._logger.info('Authorizing at %s', authUrl);
        this.redirect(authUrl);
    }

    /**
     *
     * @param {string|string[]} cookies Cookies to set
     * @param {string} track            Track id session was initiated from. Used to check the session.
     * @param {string} [retpath]        Optional retpath to return the user to (defaults to passport?mode=passport)
     */
    setSession(cookies, track, retpath) {
        this._response.header('Set-Cookie', cookies);

        const query = {track_id: track, retpath};

        if (this._serviceName === 'oauth') {
            query.origin = this.getRequestParam('origin') || 'oauth';
        }

        const finishUrl = url.format({
            protocol: 'https',
            hostname: this._getPassportHost(),
            pathname: '/auth/finish',
            query: _.omitBy(query, (arg) => !arg)
        });

        this._logger.info('Finishing', retpath ? 'with retpath to ' + retpath : '');

        // Redirect to /auth/finish on a passport domain.
        // It checks that session was initiated and redirects to retpath, see PASSP-10431
        this.redirect(finishUrl);
    }
}

AuthController.BlackboxException = inherit(Error, {
    __constructor: function(response) {
        this.name = 'BlackboxException';
        this.response = response;
    },

    toString: function() {
        return util.format('%s: %j', this.name, this.response);
    }
});

AuthController.getBlackboxStatus = function(blackbox) {
    if (!blackbox.status) {
        return this.BlackboxStatuses.UNKNOWN;
    }

    var status = 'UNKNOWN';

    Object.keys(this.BlackboxStatuses).some(function(bbstatus) {
        if (bbstatus === blackbox.status.value) {
            status = bbstatus;
            return true;
        }

        return false;
    });

    return this.BlackboxStatuses[status];
};

AuthController.BlackboxErrorTypes = BlackboxErrorTypes;
AuthController.getBlackboxErrorType = getBlackboxErrorType;
AuthController.BlackboxStatuses = {
    UNKNOWN: 'UNKNOWN',
    NEED_RESET: 'NEED_RESET',
    NOAUTH: 'NOAUTH',
    VALID: 'VALID',
    EXPIRED: 'EXPIRED',
    DISABLED: 'DISABLED',
    INVALID: 'INVALID',
    WRONG_GUARD: 'WRONG_GUARD'
};

AuthController.getBlackboxAO = function(logId, blackboxPath) {
    return new this.BlackboxAO(logId, blackboxPath);
};

AuthController.BB_ATTRIBUTES = BB_ATTRIBUTES;

AuthController.BB_ATTRIBUTE_DEFAULTS = (function() {
    var defaults = Object.keys(BB_ATTRIBUTE_DEFAULTS);

    assert(
        Object.keys(BB_ATTRIBUTES).every(function(attribute) {
            return defaults.indexOf(attribute) > -1;
        }),
        'Defaults should be present for every known blackbox attribute'
    );

    return BB_ATTRIBUTE_DEFAULTS;
})();

class BlackboxAO extends PdaoHttp {
    constructor(logId, blackboxPath) {
        super(
            logId,
            blackboxPath,
            BlackboxAO.maxRetries,
            BlackboxAO.retryAfter,
            BlackboxAO.maxConnections,
            BlackboxAO.timeout,

            // apiFailCheck uses a reference to this
            // and is called without context, hence .bind
            BlackboxAO.apiFailCheck.bind(BlackboxAO)
        );
    }

    call(method, handle, input, config) {
        return super.call(method, '/blackbox?method=' + handle, input, config);
    }
}

BlackboxAO.maxRetries = 2;
BlackboxAO.retryAfter = 100;
BlackboxAO.maxConnections = 1000;
BlackboxAO.timeout = 1000;
BlackboxAO.retryCodes = [BlackboxErrorTypes.DB_FETCHFAILED, BlackboxErrorTypes.DB_EXCEPTION];
BlackboxAO.apiFailCheck = function(logger, response, body) {
    var logType = 'retryCondition';

    if (response.statusCode !== 200) {
        logger
            .info()
            .type(logType)
            .write('Expected response status 200, got %s', response.statusCode);
        return new RetryCondition('Server response status was not 200 OK', body);
    }

    var errType = getBlackboxErrorType(body);

    if (this.retryCodes.indexOf(errType) > -1) {
        logger
            .info()
            .type(logType)
            .write('Error code %s was found in response %j', errType, body);
        return new RetryCondition('Api returned an error code from retryCodes list', body);
    }
};

AuthController.BlackboxAO = BlackboxAO;

module.exports = AuthController;
