const url = require('url');
const {Csrf} = require('@yandex-int/csrf');
const PLog = require('plog');
const querystring = require('querystring');

var AuthController; // = require('./Authentication.js');

class ExpressController {
    /**
     * Create a controller from express request and response
     *
     * @param {express.request} req
     * @param {express.response} res
     * @param {string} [logId]  Log identifier
     * @param {object} [tvmServiceAliases]  tvm service aliases
     * @constructs Controller
     * @throws
     */
    constructor(req, res, logId, tvmServiceAliases, serviceName) {
        this._request = req;
        this._response = res;
        this._logger = new PLog(logId, 'controller');
        this._cookie = {};
        this._logId = logId || null;
        this._authController = null;
        this._url = null; //Stub for caching url
        this._tvm_service_aliases = tvmServiceAliases;
        this._serviceName = serviceName;
    }

    /**
     * Get an instance of AuthController
     * @returns {AuthenticationController}
     */
    getAuth() {
        if (this._authController) {
            return this._authController;
        }

        AuthController = AuthController || require('./Authentication.js');
        this._authController = new AuthController(
            this._request,
            this._response,
            this.getLogId(),
            this._tvm_service_aliases,
            this._serviceName
        );

        return this.getAuth();
    }

    /**
     * Get a parsed url the page was opened at
     * @returns {object}
     */
    getUrl() {
        if (this._url) {
            return Object.assign(this._url);
        }

        var originalUrl = this.getHeader('x-real-scheme') + '://' + this.getHeader('host') + this._request.originalUrl;
        var parsedUrl = url.parse(originalUrl);

        this._logger.verbose('Parsed url', JSON.stringify(parsedUrl), 'from', originalUrl);

        this._url = parsedUrl;
        return Object.assign({}, parsedUrl);
    }

    /**
     * Return the top level domain the page was opened at
     * @returns {string}
     */
    getTld() {
        const domain = /^(?:.*\.)?(yandex|practicum|yango|toloka)\.(.*)$/.exec(this.getUrl().hostname);

        let tld = (domain !== null && domain[2]) || 'ru';

        if (tld === 'delivery' || tld === 'ai') {
            tld = 'com';
        }

        return tld;
    }

    /**
     * Whether the page was requested with retpath
     *
     * Does not checks if retpath is valid or if retpath exists in track.
     * Only checks if there is a retpath in request params.
     *
     * @returns {boolean}
     */
    requestedWithRetpath() {
        return Boolean(this.getRequestParam('retpath'));
    }

    /**
     * Get a cookie by name
     * @param {string} cookie   Cookie name
     * @returns {string|null}
     * @throws
     */
    getCookie(cookie) {
        var request = this._request;
        //TODO: check for cookieParser middleware

        if (typeof this._cookie[cookie] !== 'undefined') {
            return this._cookie[cookie];
        }

        return request.cookies && request.cookies[cookie];
    }

    /**
     * Set a cookie
     *
     * @param {string} cookieName
     * @param {string|Object} cookieContent     String or object that would be json.stringified
     * @param {Object} [options]                Expires, httponly, domain, path, secure etc
     * @throws
     */
    // eslint-disable-next-line no-unused-vars
    setCookie(cookieName, cookieContent, options) {
        this._response.cookie.apply(this._response, arguments);
        this._cookie[cookieName] = cookieContent;
    }

    /**
     * Get a request header by name
     *
     * @param {string} header   Header name
     * @returns {*}
     * @throws
     */
    getHeader(header) {
        return this._request.headers[header];
    }

    /**
     * Get the user ip
     *
     * Nginx should set `proxy_set_header X-Real-IP $remote_addr;` for this to work
     *
     * @returns {*}
     */
    getIp() {
        return this.getHeader('x-real-ip');
    }

    /**
     * Get a request param by its name
     *
     * Looks at nquery first (node.querystring parsed url query),
     * then looks at express' request.param method, which in turn looks into url query, body and path params
     *
     * @param {string} paramName
     * @returns {*}
     * @throws
     */
    getRequestParam(paramName) {
        const req = this._request;

        if (!req.nquery) {
            req.nquery = querystring.parse(this.getUrl().query);
        }

        return Object.assign({}, req.params, req.body, req.nquery)[paramName];
    }

    /**
     * Get the http method the request was made with
     */
    getMethod() {
        if (this._httpMethod) {
            return this._httpMethod;
        }

        this._httpMethod = new this.constructor.HTTPMethod(this._request.method);
        return this.getMethod();
    }

    /**
     * Get the request body
     * @returns {object}
     */
    getFormData() {
        return this._request.body || {};
    }

    /**
     * Get the request files
     * @returns {object}
     */
    getFormFiles() {
        return this._request.files || {};
    }

    /**
     * Render a page
     * Uses express.response.render
     *
     * @param {string} skinName         Template file name
     * @param {object} templateData     Plain object for templating
     * @returns undefined
     * @throws
     */
    render(skinName, templateData) {
        this._response.render(skinName, templateData);
    }

    /**
     * Send the string
     * @param {string} html         String to send
     * @param {number} [code=200]   Code to send the page with
     *
     * @returns When.Promise        resolves when the page is sent
     */
    sendPage(html, code) {
        var res = this._response;
        var originalEnd = res.end;

        code = code || res.statusCode || 200;

        return new Promise((resolve) => {
            res.end = function(chunk, encoding) {
                res.end = originalEnd;
                res.end(chunk, encoding);
                resolve(); //Resolve the promise when the request is sent
            };

            res.status(code).send(html);
        });
    }

    /**
     * Get the log id
     * @returns {string|null}
     */
    getLogId() {
        return this._logId;
    }

    /**
     * Redirect to the given url
     *
     * Beware of open redirects when using this method directly,
     * check the retpath is predefined or is validated by the backend
     *
     * @see followRetpath
     * @see redirectToFrontpage
     * @see redirectToFinish
     *
     * @param {string} url  An url to redirect to
     */
    redirect(url) {
        this._response.redirect(url);
    }

    getCachedSessionDataOrRequestNew() {
        var auth = this.getAuth();

        return auth.getLastRequestPromise() || auth.sessionID();
    }

    getMultiAuthUsers() {
        const blackboxMutliAccParams =
            this._serviceName === 'passport'
                ? {
                      multisession: 'yes',
                      full_info: 'yes',
                      get_family_info: 'yes',
                      get_public_id: 'yes',
                      get_public_name: 'yes',
                      get_user_ticket: 'yes',
                      allow_child: 'yes',
                      emails: 'getdefault',
                      // @see https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
                      attributes: '27,28,32,34,1003,1005,1011',
                      aliases: '1,5,6,13,21'
                  }
                : {
                      multisession: 'yes',
                      full_info: 'yes',
                      attributes: '27,28,34,1011',
                      emails: 'getdefault'
                  };

        return this.getAuth()
            .sessionID(blackboxMutliAccParams)
            .then(function(response) {
                return Promise.resolve((response && response.users) || null);
            })
            .catch((err) => {
                if (err.code !== 'need_resign') {
                    PLog.warn()
                        .logId(this._logId)
                        .type('controller', 'getMultiAuthUsers')
                        .write(err);
                }

                return Promise.reject(err);
            });
    }

    getCsrfToken() {
        var self = this;
        var usersString = 0;
        var uniqCookie = self.getCookie('uniqueuid') || self.getCookie('yandexuid');

        return self
            .getMultiAuthUsers()
            .then(function(users) {
                if (users) {
                    usersString = users
                        .map(function(acc) {
                            return acc.id;
                        })
                        .sort()
                        .join(',');
                }

                PLog.info()
                    .logId(self._logId)
                    .type('controller', 'getCsrfToken')
                    .write('Get csrf-token from uids ', usersString);
                return Promise.resolve(self.constructor.getUtils().csrf(usersString, uniqCookie));
            })
            .catch(function(err) {
                if (err.code === 'need_resign') {
                    return Promise.resolve(self.constructor.getUtils().csrf(usersString, uniqCookie));
                }
                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'getCsrfToken')
                    .write(err);
                return Promise.reject(err);
            });
    }

    getScopedCsrfToken({key}) {
        const csrf = new Csrf({key});
        var self = this;
        var usersString = 0;
        var uniqCookie = self.getCookie('uniqueuid') || self.getCookie('yandexuid');

        return self
            .getMultiAuthUsers()
            .then(function(users) {
                if (users) {
                    usersString = users
                        .map(function(acc) {
                            return acc.id;
                        })
                        .sort()
                        .join(',');
                }

                PLog.info()
                    .logId(self._logId)
                    .type('controller', 'getScopedCsrfToken')
                    .write('Get csrf-token from uids ', usersString);
                return Promise.resolve(
                    csrf.generateToken({
                        uid: usersString,
                        yandexuid: uniqCookie
                    })
                );
            })
            .catch(function(err) {
                if (err.code === 'need_resign') {
                    PLog.info()
                        .logId(self._logId)
                        .type('controller', 'getScopedCsrfToken')
                        .write('Get csrf-token from uids ', usersString);

                    return Promise.resolve(
                        csrf.generateToken({
                            uid: usersString,
                            yandexuid: uniqCookie
                        })
                    );
                }
                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'getCsrfToken')
                    .write(err);
                return Promise.reject(err);
            });
    }

    checkCsrfToken(token) {
        var self = this;
        var usersString = 0;
        var uniqCookie = self.getCookie('uniqueuid') || self.getCookie('yandexuid');

        return self
            .getMultiAuthUsers()
            .then(function(users) {
                if (users) {
                    usersString = users
                        .map(function(acc) {
                            return acc.id;
                        })
                        .sort()
                        .join(',');
                }

                PLog.info()
                    .logId(self._logId)
                    .type('controller', 'checkCsrfToken')
                    .write('Check csrf-token from uids ', usersString);
                return Promise.resolve(self.constructor.getUtils().csrf.isValid(usersString, uniqCookie, token));
            })
            .catch(function(err) {
                if (err.code === 'need_resign') {
                    return Promise.resolve(self.constructor.getUtils().csrf(usersString, uniqCookie));
                }

                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'checkCsrfToken')
                    .write(err);
                return Promise.reject(err);
            });
    }

    /**
     * Checks csrf token is valid
     *
     * If no token is passed, checks csrf field from the request
     *
     * @param {string} [token]  csrf token to check
     * @returns {boolean}
     */
    isCsrfTokenValidV2(token) {
        var self = this;

        token = token || this.getRequestParam('csrf') || this.getRequestParam('csrf_token');

        if (!token) {
            PLog.info()
                .logId(self._logId)
                .type('controller', 'isCsrfTokenValidV2')
                .write('CSRF-token is required');
            return Promise.reject(new Error('CSRF-token is required'));
        }

        return self
            .checkCsrfToken(token)
            .then(function(isValid) {
                if (!isValid) {
                    PLog.info()
                        .logId(self._logId)
                        .type('controller', 'isCsrfTokenValidV2')
                        .write('Invalid CSRF token');
                    return false;
                } else {
                    return Promise.resolve(true);
                }
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'isCsrfTokenValidV2')
                    .write(err);
                return Promise.reject(err);
            });
    }

    isScopedCsrfTokenValid({key, token}) {
        const csrf = new Csrf({key});
        const self = this;
        const uniqCookie = self.getCookie('uniqueuid') || self.getCookie('yandexuid');

        if (!token) {
            PLog.warn()
                .logId(self._logId)
                .type('controller', 'isScopedCsrfTokenValid')
                .write('CSRF-token is required');
            return Promise.reject(new Error('CSRF-token is required'));
        }

        return self
            .getMultiAuthUsers()
            .catch(function(err) {
                if (err.code === 'need_resign') {
                    return Promise.resolve(null);
                }

                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'checkCsrfToken')
                    .write(err);
                return Promise.reject(err);
            })
            .then(function(users) {
                let usersString = 0;

                if (users) {
                    usersString = users
                        .map(function(acc) {
                            return acc.id;
                        })
                        .sort()
                        .join(',');
                }

                PLog.info()
                    .logId(self._logId)
                    .type('controller', 'checkCsrfToken')
                    .write('Check csrf-token from uids ', usersString);

                return Promise.resolve(
                    csrf.isTokenValid(token, {
                        uid: usersString,
                        yandexuid: uniqCookie
                    })
                ).then(function(isValid) {
                    if (!isValid) {
                        PLog.info()
                            .logId(self._logId)
                            .type('controller', 'isScopedCsrfTokenValid')
                            .write('Invalid CSRF token');
                        return false;
                    }
                    return Promise.resolve(true);
                });
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'isCsrfTokenValidV2')
                    .write(err);
                return Promise.reject(err);
            });
    }

    /**
     * Checks csrf token is valid
     *
     * If no token is passed, checks csrf field from the request
     *
     * @param {string} [token]  csrf token to check
     * @returns {boolean}
     */
    isCsrfTokenValid(token) {
        var self = this;

        token = token || this.getRequestParam('csrf');
        // eslint-disable-next-line no-console
        //
        return self
            .getCsrfToken()
            .then(function(expectedtoken) {
                if (token !== expectedtoken) {
                    PLog.info()
                        .logId(self._logId)
                        .type('controller', 'isCsrfTokenValid')
                        .write('Invalid CSRF token');
                    PLog.debug()
                        .logId(self._logId)
                        .type('controller', 'isCsrfTokenValid')
                        .write('Expected csrf token %s, but got %s', expectedtoken, token);
                    return false;
                } else {
                    return Promise.resolve(true);
                }
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(self._logId)
                    .type('controller', 'isCsrfTokenValid')
                    .write('err');
                return Promise.reject(err);
            });
    }

    generateYandexuid() {
        return Math.floor(Math.random() * 1e8 + 1).toString() + Math.floor(Date.now() / 1000).toString();
    }
}

(function() {
    var utilsLib = null;

    ExpressController.setUtils = function(utils) {
        utilsLib = utils;
    };

    ExpressController.getUtils = function() {
        return utilsLib;
    };

    ExpressController._resetUtils = function() {
        utilsLib = null;
    };
})();

class HTTPMethod {
    constructor(method) {
        this._method = method.toLowerCase();
    }

    toString() {
        return this._method;
    }

    isPost() {
        return this._method === 'post';
    }

    isGet() {
        return this._method === 'get';
    }
}

ExpressController.HTTPMethod = HTTPMethod;

module.exports = ExpressController;
