const _ = require('lodash');
const PLog = require('plog');
const xml2js = require('xml2js');
const assert = require('assert');
const querystring = require('querystring');
const Abstract = require('./Abstract.js');
const prequest = require('prequest');
const tools = require('prequest/tools');
const noop = function() {};

/**
 * @constructor
 * @class DAO
 */
class HTTPDao extends Abstract {
    /**
     * @param {string}      logId               Request token for logging
     * @param {string}      baseURL             An url to prepend all handles with
     * @param {number}      maxRetries          Maximum number of retries
     * @param {number}      retryAfter          ms to wait before retrying
     * @param {number}      maxConnections      A maximum size of connections pool
     * @param {number}      timeout             Request timeout in ms
     * @param {function}    [apiFailCheck]      A function to determine if retry should be done after the response
     * is received. It is given the logger, response body and response object.
     * It should return an Error to trigger a retry.
     * @constructor
     * @class DAO
     */
    constructor(
        logId,
        baseURL,
        maxRetries,
        retryAfter,
        maxConnections,
        timeout,
        apiFailCheck,
        multipart,
        mixContentType,
        agentOptions,
        explicitHeaders,
        logSuffix
    ) {
        assert(!logSuffix || typeof logSuffix === 'string', 'logSuffix should be a string');
        assert(typeof logId === 'string' && logId.trim(), 'LogID should be a string');
        assert(baseURL && typeof baseURL === 'string', 'Base Url should be a string');
        assert(typeof maxRetries === 'number', 'Max Retries should be a number');
        assert(typeof retryAfter === 'number', 'Retry timeout should be a number in ms');
        assert(typeof maxConnections === 'number', 'Max Connections should be a number');
        assert(typeof timeout === 'number', 'Timeout should be a number');
        assert(
            typeof apiFailCheck === 'undefined' || typeof apiFailCheck === 'function',
            'Api Fail Check should be a function if given'
        );
        assert(
            typeof multipart === 'undefined' || typeof multipart === 'boolean',
            'multipart should be a boolean if given'
        );
        assert(
            typeof mixContentType === 'undefined' || typeof mixContentType === 'boolean',
            'mixContentType should be a boolean if given'
        );
        assert(
            typeof agentOptions === 'undefined' || typeof agentOptions === 'object',
            'agentOptions should be an object if given'
        );
        assert(
            typeof explicitHeaders === 'undefined' || typeof explicitHeaders === 'object',
            'explicitHeaders should be an object if given'
        );

        super();

        /**
         * @type {string}
         * @private
         */
        this._logId = logId;

        /**
         * @type {string}
         * @private
         */
        this._logSuffix = logSuffix;

        /**
         * Default dao config
         * @type {Config}
         * @private
         */
        this._defaultConfig = new HTTPDao.Config()
            .setBaseUrl(baseURL)
            .setMaxRetries(maxRetries)
            .setRetryAfter(retryAfter)
            .setMaxConnections(maxConnections)
            .setTimeout(timeout)
            .setApiFailCheck(apiFailCheck || noop)
            .setMultipart(multipart === undefined ? false : multipart)
            .setMixContentType(mixContentType === undefined ? true : mixContentType)
            .setMixContentTypeForceBody(false)
            .setAgentOptions(agentOptions === undefined ? {} : agentOptions)
            .setExplicitHeaders(explicitHeaders === undefined ? {} : explicitHeaders);

        assert(this._defaultConfig.isValid(), 'Default config turned out to be invalid');

        /**
         * Headers dictionary to mix into the request
         * @type {{}}
         * @private
         */
        this._headers = {};

        /**
         * @type {Logger}
         */
        this._logger = new PLog(logId);
        this._logger.type('dao', 'http');

        this._encoding = undefined;
        this._responseType = undefined;
    }

    cloneConfig() {
        return this._defaultConfig.clone();
    }

    mixHeaders(headers) {
        assert(_.isObjectLike(headers), 'Headers should be a dict');
        this._headers = headers;
        return this;
    }

    setEncoding(encoding) {
        this._encoding = encoding;
        return this;
    }

    setResponseType(responseType) {
        this._responseType = responseType;
        return this;
    }

    /**
     * Make a call to API
     *
     * @param {Config} config
     * @param {string} method
     * @param {string} handle
     * @param {object} input
     * @returns Promise
     * @private
     */
    _makeCall(config, method, handle, input) {
        method = method.toLowerCase();

        const logger = this._logger;

        return new Promise((resolve, reject) => {
            const params = {
                //Request info
                method: method,

                //Dao config
                timeout: config.getTimeout(),
                https: config.getAgentOptions(),
                retry: {
                    limit: config.getMaxRetries()
                },

                headers: Object.assign({}, this._headers, config.getExplicitHeaders())
            };

            let url = config.getBaseUrl() + handle;

            if (typeof this._encoding !== 'undefined') {
                params.encoding = this._encoding;
            }

            if (typeof this._responseType !== 'undefined') {
                params.responseType = this._responseType;
            }

            const inputQS = input && querystring.stringify(input); //Using native quetystring

            if (method === 'get') {
                //Adding the querystring to the url
                const separator = url.indexOf('?') > -1 ? '&' : '?';

                url = url + separator + inputQS;
            } else {
                if (config.getMultipart()) {
                    const form = {};
                    const search = {};

                    Object.entries(input).forEach(([key, value]) => {
                        if (typeof value === 'string' || (typeof value === 'object' && value && value.path)) {
                            form[key] = value;
                        } else {
                            search[key] = value;
                        }
                    });

                    params.body = tools.getFormData(form);
                    params.searchParams = querystring.stringify(search);
                    params.headers = {...params.headers, ...tools.getFormDataHeaders(params.body)};
                } else if (config.getMixContentType()) {
                    if (config.getMixContentTypeForceBody()) {
                        params.body = inputQS;
                    } else {
                        params.form = input;
                    }
                    params.headers = {
                        ...params.headers,
                        'Content-type': 'application/x-www-form-urlencoded; charset=utf-8'
                    };
                } else {
                    if (typeof input === 'object' && Object.keys(input).length) {
                        params.body = input;
                    }
                }
            }

            const lift = new Promise((resolve) => {
                prequest({url, params, logID: this._logId, logSuffix: this._logSuffix})
                    .then((response) => {
                        resolve({
                            response,
                            body: response.body
                        });
                    })
                    .catch((error) => {
                        resolve({
                            error
                        });
                    });
            });

            lift.then(function(data) {
                if (typeof data.body === 'string' && data.body.indexOf('<?xml ') === 0) {
                    //If it looks like xml, it is xml
                    logger.verbose('Looks like xml:', data.body);

                    return new Promise((resolve, reject) => {
                        xml2js.parseString(data.body, {explicitArray: false}, (err, response) => {
                            if (err) {
                                reject(err);
                            } else {
                                resolve(response);
                            }
                        });
                    }).then((result) => {
                        logger.verbose('Parsed %s into %j', data.body, result);
                        data.body = result;
                        return data;
                    });
                }

                return data;
            })
                .then((data = {}) => {
                    data.error = data.error || config.getApiFailCheck()(logger, data.response, data.body);

                    if (data.error) {
                        logger.info('Response failed %s', data.error.toString());
                        reject(data.error);
                    } else {
                        resolve(data.body);
                    }
                })
                .catch(reject);
        });
    }

    /**
     * Call an api handle
     *
     * @param {string} method       HTTP Method name (either get, post, put or delete)
     * @param {string} handle       Api handle
     * @param {object} input        Plain object with input to send to api
     * @param {Config} [config]     A specific dao configuration for this call
     * @returns Promise
     *
     * @class DAO
     */
    call(method, handle, input, config) {
        assert(
            typeof method === 'string' && ['get', 'post', 'put', 'delete'].indexOf(method.toLowerCase()) > -1,
            'Method should be an http method name'
        );
        assert(handle && typeof handle === 'string', 'Handle should be a string');
        assert(_.isObjectLike(input), 'Input should be a plain object');
        assert(!config || config instanceof HTTPDao.Config, 'Config should be a Dao Config object if defined');
        assert(!config || config.isValid(), 'Config should have all values set');

        config = config || this._defaultConfig;
        return this._makeCall(config, method, handle, input);
    }
}

class Config {
    constructor() {
        _.each(Config.settings, (type, setting) => {
            this['_' + setting] = null;
        });
    }

    clone() {
        const config = new Config();

        _.each(Config.settings, (type, setting) => {
            const capitalized = _.upperFirst(setting);

            config['set' + capitalized](this['get' + capitalized]());
        });

        return config;
    }

    isValid() {
        return _.every(Config.settings, (type, setting) => {
            return this['_' + setting] !== null;
        });
    }
}

Config.settings = {
    baseUrl: 'string',
    timeout: 'number',
    maxRetries: 'number',
    retryAfter: 'number',
    maxConnections: 'number',

    /**
     * Method to determine if response should be treated as an error.
     * It should return an error object if there is a problem with the request.
     *
     * It is passed response object and response body
     *
     * @type {Function}
     * @returns {?Error}
     * @private
     */
    apiFailCheck: 'function',
    multipart: 'boolean',
    mixContentType: 'boolean',
    mixContentTypeForceBody: 'boolean',
    agentOptions: 'object',
    explicitHeaders: 'object'
};

_.each(Config.settings, function(type, setting) {
    const capitalized = _.upperFirst(setting);

    //Getter
    Config.prototype['get' + capitalized] = function() {
        const value = this['_' + setting];

        assert(value !== null, capitalized + ' is not defined yet, define it with set' + capitalized);
        return value;
    };

    //Setter
    Config.prototype['set' + capitalized] = function(value) {
        assert(typeof value === type, capitalized + ' should be a ' + type);
        this['_' + setting] = value;
        return this;
    };
});

HTTPDao.Config = Config;

module.exports = HTTPDao;
