var assert = require('assert');
var _ = require('lodash');
var PUtils = require('putils');
var spacesRegexp = /\s+/g;

var createGetter = function(field) {
    return function() {
        return this._values[field];
    };
};

var createSetter = function(field) {
    return function(value) {
        this._values[field] = String(value);
        return this;
    };
};

/**
 * Passport-form Field constructor
 *
 * @typedef Field
 * @class Field
 *
 * @constructor
 */
var Field = require('inherit')(
    {
        /**
         * @param {string} id   Field id
         *
         * @class Field
         * @constructor
         */
        __constructor: function(id) {
            this._init.call(this, id);
        },

        /**
         * Constructor logic for easier extension
         * @param {string} id
         * @private
         */
        _init: function(id) {
            assert(id && typeof id === 'string', 'ID String Required');

            /**
             * Storage for Field values
             *
             * @private
             */
            this._values = {
                /**
                 * Field id
                 * @type {string}
                 */
                id: id,

                /**
                 * Name
                 * @type {string}
                 */
                name: '',

                /**
                 * Label
                 * @type {string}
                 */
                label: '',

                /**
                 * Hint message
                 * @type {string}
                 */
                hint: '',

                /**
                 * Field contents
                 * @type {string}
                 */
                value: '',

                /**
                 * Field contents
                 * @type {string}
                 */
                placeholder: '',

                /**
                 * Flag whether this field is required
                 * @type {boolean}
                 */
                required: false,

                /**
                 * Flag whether this field is a hidden field
                 * @type {boolean}
                 */
                hidden: false,

                /**
                 * Field options
                 * @type {Object}
                 */
                options: {},

                /**
                 * Field errors
                 * @type {Array}
                 */
                errors: []
            };

            this.setName(id);
            this.addError(new Field.Error('globalinternal', '%errors.globalinternal'));

            return this;
        },

        getID: createGetter('id'),

        /**
         * @returns {string}
         */
        getName: createGetter('name'),

        /**
         * @returns {Field}
         */
        setName: createSetter('name'),

        /**
         * @returns {string}
         */
        getLabel: createGetter('label'),

        /**
         * @param {string} label Label localization string
         * @returns {Field}
         */
        setLabel: createSetter('label'),

        /**
         * @returns {string}
         */
        getHint: createGetter('hint'),

        /**
         * @param {string} hint Hint localization string
         * @returns {Field}
         */
        setHint: createSetter('hint'),

        /**
         * @returns {string}
         */
        getPlaceholder: createGetter('placeholder'),

        /**
         * @param {string} placeholder Placeholder localization string
         * @returns {Field}
         */
        setPlaceholder: createSetter('placeholder'),

        /**
         * Marks the field as required
         *
         * Fields are optional by default
         *
         * @returns {Field}
         */
        setRequired: function() {
            this._values.required = true;
            return this;
        },

        /**
         * Marks the field as optional
         *
         * Fields are optional by default
         *
         * @returns {Field}
         */
        setOptional: function() {
            this._values.required = false;
            return this;
        },

        /**
         * @returns {boolean}
         */
        isRequired: createGetter('required'),

        setHidden: function() {
            this._values.hidden = true;
            return this;
        },

        setVisible: function() {
            this._values.hidden = false;
            return this;
        },

        isHidden: createGetter('hidden'),

        /**
         * @returns {Field}
         */
        setValue: function(value) {
            assert(
                require('lodash').isPlainObject(value) || typeof value === 'string',
                'Value should be a plain object or a string'
            );
            this._values.value = value;
            return this;
        },

        /**
         * @returns {string}
         */
        getValue: createGetter('value'),

        /**
         * @param {object} options
         * @returns {Field}
         */
        setOptions: function(options) {
            assert(_.isObjectLike(options), 'Options should be a plain object');
            this._values.options = _.extend(this._values.options, options);
            return this;
        },

        /**
         * @returns {object}
         */
        getOptions: createGetter('options'),

        /**
         * @param {string} option
         * @param {*} [value]   Option value, true if undefined
         * @returns {Field}
         */
        setOption: function(option, value) {
            assert(option && typeof option === 'string', 'Option should be a string');

            if (typeof value === 'undefined') {
                value = true;
            }

            this._values.options[option] = value;
            return this;
        },

        /**
         * Get a value of an option
         * @param {string} option
         * @returns {*}
         */
        getOption: function(option) {
            assert(option && typeof option === 'string', 'Option should be a string');
            return this._values.options[option];
        },

        /**
         * @returns {Field}
         */
        clearOptions: function() {
            this._values.options = {};
            return this;
        },

        /**
         * Adds an Error to the field
         * @param {Field.Error} error
         * @returns {Field}
         */
        addError: function(error) {
            this._values.errors.push(error);
            return this;
        },

        /**
         * @returns {Field.Error[]}
         */
        getErrors: createGetter('errors'),

        /**
         * Gets an Error by it's code
         *
         * Returns null if no Error found with such code
         *
         * @param {string} code
         * @returns {Field.Error|Null}
         */
        getErrorByCode: function(code) {
            assert(code && typeof code === 'string', 'Code should be a string');
            return (
                _.find(this.getErrors(), function(error) {
                    return error.getCode() === code;
                }) || null
            );
        },

        /**
         * Get a compiled Field suitable for yate templating
         *
         * Returns the compiled object or a promise, that resolves with the compiled object
         * @param {string}        lang    Language code {ru, en, tr, uk}
         * @returns {Object|When.Promise}
         */
        compile: function(lang) {
            assert(typeof lang === 'string' && lang.length === 2, 'Language code should be defined');

            var compiled = _(this._values)
                .omit('errors')
                .omitBy(function(value) {
                    return !value;
                })
                .mapValues(function(value, key) {
                    if (['label', 'hint', 'placeholder'].indexOf(key) >= 0) {
                        return PUtils.i18n(lang, value);
                    }

                    return value;
                })
                .value();

            var errors = this.getErrors();

            if (errors.length > 0) {
                compiled.error = errors.map(function(error) {
                    return error.compile(lang);
                });
            }

            return _.cloneDeep(compiled);
        },

        /**
         * Stub. Whether the field is empty based on POSTed form.
         *
         * @returns {boolean|When.Promise}
         */
        isEmpty: function() {
            return false;
        },

        /**
         * Stub. Change internal state when the field is empty
         * @param {Object}          formData    Whole POSTed user input
         */
        onEmpty: function(formData) {
            this.setValue(this._parseValue(formData));
        },

        /**
         * Stub. Validate the field against the user input.
         * A result of validation is an array of errors.
         * Empty array means validation was successful.
         *
         * @returns {Array|When.Promise}
         */
        validate: function() {
            return [];
        },

        /**
         * Stub. Change internal state when the field valid
         * @param {Object}          formData    Whole POSTed user input
         */
        onValid: function(formData) {
            this.setValue(this._parseValue(formData));
        },

        /**
         * Stub. Change internal state when the field valid
         * @param {Array}           errors      Array of errors validation function returned
         * @param {Object}          formData    Whole POSTed user input
         */
        onInvalid: function(errors, formData) {
            /* jshint unused:false */
            this.setValue(this._parseValue(formData));
        },

        /**
         * Removes double-spaces from and trims the string
         *
         * @param {string} str
         * @returns {string}
         */
        normalizeValue: function(str) {
            if (typeof str !== 'string') {
                if (typeof str === 'number') {
                    return String(str);
                }

                return '';
            }

            return str.replace(spacesRegexp, ' ').trim();
        },

        /**
         * Normalizes those contents of the form, relevant to the field
         *
         * @param {Object} formData     Posted formData
         * @returns {Object}            Object with key for field name (or names,
         * if multiple required like captcha) and normalized value
         */
        normalize: function(formData) {
            var normalized = {};

            normalized[this.getName()] = this._parseValue(formData);
            return normalized;
        },

        /**
         * Common method to get a value of the field from the form
         * @param {Object}      formData    POSTed form data
         * @returns {string|array}
         * @protected
         */
        _parseValue: function(formData) {
            var field = formData[this.getName()];

            if (!this._values.options || !this._values.options.multiplyFields) {
                return this.normalizeValue(field);
            }

            var data = {};
            var self = this;
            var fieldsVals = [];

            [].concat(field).forEach(function(item) {
                if (item) {
                    fieldsVals.push(self.normalizeValue(item));
                }
            });

            data.values = fieldsVals;
            return data;
        },

        /**
         * Find errors by their codes and set them active
         * @param {string[]}    errors  An array of error codes
         */
        setErrorsActive: function(errors) {
            if (!_.isArray(errors)) {
                throw new Error('Errors are expected to be an array of string error codes');
            }

            this.getErrors().forEach((error) => {
                if (errors.includes(error.getCode())) {
                    error.setActive();
                }
            });
        },

        /**
         * Checks whether this field is in the posted form data
         * @param {object} formData
         * @returns {boolean}
         */
        isPresent: function(formData) {
            assert(_.isObjectLike(formData), 'Argument should be a posted form data');
            return this.getName() in formData;
        }
    },
    {
        /**
         * Field Error constructor
         * @typedef Field.Error
         * @class Field.Error
         * @constructor
         */
        Error: require('inherit')({
            /**
             * @param {string} code             Error code
             * @param {string} messageLocKey    Description message localization key
             * @param {boolean} [active]        Whether the error is active
             * @class Field.Error
             * @constructor
             */
            __constructor: function(code, messageLocKey, active, replace) {
                assert(code && typeof code === 'string', 'Error Code Required');
                assert(
                    messageLocKey && typeof messageLocKey === 'string',
                    'Description message localization key required'
                );
                assert(
                    ['undefined', 'boolean'].indexOf(typeof active) > -1,
                    'Activeness of the error should be a boolean if defined'
                );

                /**
                 * Storage for Error values
                 *
                 * @private
                 */
                this._values = {
                    /**
                     * Error code
                     * @type {string}
                     */
                    code: code,

                    /**
                     * String for replacing mark in description message localization key
                     * @type {string}
                     */
                    replace: replace,

                    /**
                     * Description message localization key
                     * @type {string}
                     */
                    message: messageLocKey,

                    /**
                     * Error active flag
                     * @type {boolean}
                     */
                    active: Boolean(active)
                };
            },

            /**
             * @returns string
             */
            getCode: createGetter('code'),

            /**
             * @returns string
             */
            getMessage: createGetter('message'),

            /**
             * @returns boolean
             */
            isActive: createGetter('active'),

            /**
             * @returns {Field.Error}
             */
            setActive: function() {
                this._values.active = true;
                return this;
            },

            /**
             * @returns {Field.Error}
             */
            setInactive: function() {
                this._values.active = false;
                return this;
            },

            /**
             * Get a compiled Field Error suitable for yate templating
             * @param {string} lang Language code
             * @returns {Object}
             */
            compile: function(lang) {
                assert(typeof lang === 'string' && lang.length === 2, 'Language code is not defined');
                var replace = '';

                return _(this._values)
                    .omit(function(propValue) {
                        return !propValue;
                    })
                    .mapValues(function(value, key) {
                        if (key === 'replace') {
                            replace = value;
                        }

                        if (key === 'message') {
                            return replace ? PUtils.i18n(lang, value, replace) : PUtils.i18n(lang, value);
                        }

                        return value;
                    })
                    .value();
            }
        })
    }
);

module.exports = Field;
