var assert = require('assert');
var when = require('when');
var _ = require('lodash');

/**
 * Passport form
 * @typedef {Form}
 */
var Form = require('inherit')({
    /**
     * @param {Field...} fields
     * @constructs Form
     */
    __constructor: function(fields) {
        /* jshint unused:false */

        this._fields = Array.isArray(fields) ? fields : Array.prototype.slice.call(arguments, 0);

        this._fields.forEach(function(field) {
            assert(field instanceof require('./Field'), 'All the arguments should be Fields');
        });
        assert(
            this._fields.some(function(field) {
                return field.isRequired();
            }),
            'At least one required field should be present'
        );

        this.api = null;
        this._isValid = false;

        this._logger = new (require('plog'))();
        this._logger.type('form');
    },

    /**
     * Returns fields the form contains
     * @returns {Array}
     */
    getFields: function() {
        return this._fields;
    },

    /**
     * @param {string} name
     * Returns field
     * @returns {Object}
     */
    getField: function(name) {
        assert(typeof name === 'string', 'Name should be a string');

        return _.find(this.getFields(), function(field) {
            return field.getName() === name;
        });
    },

    /**
     * Returns whether the form is currently valid
     *
     * Form is valid once it was successfully validated
     * @returns {boolean}
     */
    isValid: function() {
        return this._isValid;
    },

    /**
     * Compiles the form into an array for templating
     *
     * Calls .compile() on all fields
     *
     * @param {string}  lang    Language code {ru, en, tr, uk}
     * @returns {Object[]}  An array of compiled fields
     */
    compile: function(lang) {
        var api = this.api;

        assert(Boolean(api), 'Compilation requires API, set it using form.setApi(yourApi)');
        assert(typeof lang === 'string' && lang.length === 2, 'Language code should be defined');

        return when.all(
            this._fields.map(function(field) {
                return field.compile(lang, api);
            })
        );
    },

    /**
     * Validates the form against the user input
     *
     * Fields are matched using their .getName(): if formData has a field with Field's name, then Field has matched
     * Matching fields receive whole formData to validate against,
     * since they may need more than single field to validate.
     *
     * @param {Object} formData     POSTed client input
     * @returns {When.Promise}      Resolves with true if all the matched fields are
     * valid and all the required fields had matched
     */
    validate: function(formData) {
        var api = this.api;

        assert(Boolean(api), 'Validation requires API, set it using form.setApi(yourApi)');
        assert(_.isObjectLike(formData), 'formData should be a plain object');

        var logger = this._logger;

        var required = this._fields.filter(function(field) {
            return field.isRequired();
        });
        var requiredFieldWithNoFormData = _.find(required, function(field) {
            return !field.isPresent(formData);
        });

        if (requiredFieldWithNoFormData) {
            //All the required fields
            //Should have a respective field in formData
            //If any required fields do not match, instantly resolve with "false"

            logger
                .debug()
                .type('validate')
                .write('Required field %s is not present in the form', requiredFieldWithNoFormData.getID());
            return when.promise(function(resolve) {
                resolve(false);
            });
        }

        var that = this;

        return when
            .reduce(
                this._fields
                    //Filter fields that match the input
                    .filter(function(field) {
                        return field.isPresent(formData);
                    })

                    //Validate those fields
                    .map(function(field) {
                        logger
                            .verbose()
                            .type('validate')
                            .write('Validating', field.getID());
                        return when.resolve(field.isEmpty(formData, api)).then(function(isEmpty) {
                            if (isEmpty) {
                                logger
                                    .verbose()
                                    .type('validate')
                                    .write('%s is empty', field.getID());

                                //Empty fields should not be validated
                                return when.resolve(field.onEmpty(formData, api)).then(function() {
                                    var isRequired = field.isRequired();

                                    if (isRequired) {
                                        logger
                                            .debug()
                                            .type('validate')
                                            .write('Required field %s is empty', field.getID());
                                    }
                                    return !isRequired;
                                });
                            }

                            logger
                                .verbose()
                                .type('validate')
                                .write('%s is not empty, validating it.', field.getID());
                            return when.resolve(field.validate(formData, api)).then(function(errors) {
                                var isValid = errors.length === 0;

                                if (!isValid) {
                                    logger
                                        .debug()
                                        .type('validate')
                                        .write('Field %s is invalid:', field.getID(), errors);
                                }

                                var onValidation;

                                if (isValid) {
                                    onValidation = when.resolve(field.onValid(formData, api));
                                } else {
                                    onValidation = when.resolve(field.onInvalid(errors, formData, api));
                                }

                                return onValidation.then(function() {
                                    return isValid;
                                });
                            });
                        });
                    }),
                function(result, oneValidationResult) {
                    return result && oneValidationResult;
                }
            )
            .then(function(isValid) {
                if (isValid) {
                    logger
                        .debug()
                        .type('validate')
                        .write('Field is valid');
                    that._isValid = true;
                } else {
                    logger
                        .debug()
                        .type('validate')
                        .write('Field is invalid');
                }

                return isValid;
            });
    },

    /**
     * Normalize the form by asking each field to normalize itself
     * @param {Object} formData     POSTed form data
     */
    normalize: function(formData) {
        return this._fields
            .map(function(field) {
                return field.normalize(formData);
            })
            .reduce(function(result, normalized) {
                return _.extend(result, normalized);
            }, {});
    },

    /**
     * Set api for the form, to be passed to field validations
     * @param {API} api Passport API to be passed to Field validations
     * @returns {Form}
     */
    setApi: function(api) {
        this.api = api;
        return this;
    }
});

module.exports = Form;
