var _ = require('lodash');
var util = require('util');
var assert = require('assert');

/**
 * Logger configurator
 * @class Configurator
 */
var Configurator = require('inherit')(
    {
        __constructor: function() {
            this._formatter = new (require('./formatter/DevFormatter'))(this);
            this._handler = new (require('./handler/ConsoleHandler'))(this);

            this.minLevel(this.__self.knownLevels[this.__self.knownLevels.length - 1]); //All logs are shown by default
        },

        getFormatter: function() {
            return this._formatter;
        },

        getFormatterByName: function(name) {
            var availableFormatters = _.map(this.__self.formatters, function(value, key) {
                return key;
            }).join(', ');

            assert(
                name in this.__self.formatters,
                util.format('Unknown formatter %s, available formatters: %s', name, availableFormatters)
            );

            return new this.__self.formatters[name](this);
        },

        setFormatter: function(formatter) {
            assert(
                formatter instanceof require('./formatter/AbstractFormatter'),
                'Formatter should be an AbstractFormatter instance'
            );
            this._formatter = formatter;
            return this;
        },

        getHandler: function() {
            return this._handler;
        },

        getHandlerByName: function(name) {
            var availableHandlers = _.map(this.__self.handlers, function(value, key) {
                return key;
            }).join(', ');

            assert(
                name in this.__self.handlers,
                util.format('Unknown handler %s, available handlers: %s', name, availableHandlers)
            );

            return new this.__self.handlers[name](this);
        },

        setHandler: function(handler) {
            assert(
                handler instanceof require('./handler/AbstractHandler'),
                'Handler should be an AbstractHandler instance'
            );
            this._handler = handler;
            return this;
        },

        _checkLevel: function(level) {
            assert(this.__self.knownLevels.indexOf(level) > -1, 'Unknown loglevel ' + level);
        },

        levelImportantEnough: function(level) {
            this._checkLevel(level);
            return this.__self.knownLevels.indexOf(level) <= this.__self.knownLevels.indexOf(this._minLevel);
        },

        logIdRquiredForLevel: function(level) {
            this._checkLevel(level);

            //All the levels above INFO should have logId
            return this.__self.knownLevels.indexOf(level) <= this.__self.knownLevels.indexOf('info');
        },

        minLevel: function(level) {
            this._checkLevel(level);
            this._minLevel = level;
            return this;
        },

        getCurrentMinLevel: function() {
            return this._minLevel;
        },

        getKnownLevels: function() {
            return this.__self.knownLevels;
        }
    },
    {
        //Log levels sorted by severity
        knownLevels: ['critical', 'error', 'warn', 'info', 'debug', 'verbose'],

        instance: (function() {
            /**
             * Configurator instance
             * @type {Configurator}
             */
            var configurator = null;

            /**
             * Configurator singleton getter
             * @returns Configurator
             */
            return function() {
                if (configurator === null) {
                    configurator = new Configurator();
                }

                return configurator;
            };
        })(),

        handlers: {
            abstract: require('./handler/AbstractHandler'),
            buffering: require('./handler/BufferingHandler'),
            console: require('./handler/ConsoleHandler')
        },

        formatters: {
            abstract: require('./formatter/AbstractFormatter'),
            dev: require('./formatter/DevFormatter'),
            production: require('./formatter/ProductionFormatter')
        }
    }
);

module.exports = Configurator;
